/*
 * auth.c - deal with authentication.
 *
 * This file implements the VNC authentication protocol when setting up an RFB
 * connection.
 */

/*
 *  OSXvnc Copyright (C) 2001 Dan McGuirk <mcguirk@incompleteness.net>.
 *  Original Xvnc code Copyright (C) 1999 AT&T Laboratories Cambridge.  
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */


#include <rsa_crypt_lib.h>
#include <rfb/rfb.h>
#include <italc_rfb_ext.h>
#include <paths.h>

extern int no_auth;

/*
 * rfbAuthNewClient is called when we reach the point of authenticating
 * a new client.  If authentication isn't being used then we simply send
 * rfbNoAuth.  Otherwise we send rfbVncAuth plus the challenge.
 */

void
rfbAuthNewClient(cl)
    rfbClientPtr cl;
{
    char buf[4 + CHALLENGESIZE];
    int len;
    uint32_t auth; 
    s_rfb_authentication ra;
    char * uname;
    char * extension = ".public";
    char * file;
    RSA * pubkey;
    int encrypted_size;
    Q_UINT8 * encrypted;

    cl->state = RFB_AUTHENTICATION;
#ifdef RSA_CRYPT_AVAILABLE
    if( no_auth )
    {
#endif
/*        if (cl->screen->authPasswdData && !cl->reverseConnection) {
            *(uint32_t *)buf = Swap32IfLE(rfbVncAuth);
            rfbRandomBytes(cl->authChallenge);
            memcpy(&buf[4], (char *)cl->authChallenge, CHALLENGESIZE);
            len = 4 + CHALLENGESIZE;
        } else {*/
            *(uint32_t *)buf = Swap32IfLE(rfbNoAuth);
            len = 4;
            cl->state = RFB_INITIALISATION;
    /*    }*/

        if (WriteExact(cl, buf, len) < 0) {
            rfbLogPerror("rfbAuthNewClient: write");
            rfbCloseClient(cl);
            return;
        }
	return;
#ifdef RSA_CRYPT_AVAILABLE
    }

    auth = Swap32IfLE( RFB_VNC_CHALL_RESP_AUTHENTICATION );
    if( WriteExact(cl, (char*)&auth, sizeof( auth ) ) < 0 )
    {
        rfbLogPerror("rfbAuthNewClient: write");
        rfbCloseClient(cl);
        return;
    }
    if( ReadExact(cl, (char *)&ra, sizeof( ra ) ) <= 0 )
    {
        rfbLogPerror("rfbAuthNewClient: read");
        rfbCloseClient(cl);
        return;
    }
    ra.user_name_len = Swap16IfLE( ra.user_name_len );
    uname = malloc( ra.user_name_len );
    if( ReadExact(cl, uname, ra.user_name_len ) <= 0 )
    {
        rfbLogPerror("rfbAuthNewClient: read");
        rfbCloseClient(cl);
        return;
    }
    file = malloc( strlen( ITALC_CLIENT_PUBLIC_KEYS ) + strlen( uname ) + strlen( extension ) + 1 );
    strcpy( file, ITALC_CLIENT_PUBLIC_KEYS );
    strcat( file, uname );
    strcat( file, extension );
    pubkey = readPublicKey( file );
    if( pubkey == NULL )
    {
        rfbLogPerror("rfbAuthNewClient: readPublicKey");
        rfbCloseClient(cl);
        return;
    }

    rsaGenerateChallenge( &cl->challenge );

    encrypted_size = RSA_size( pubkey );
    encrypted = malloc( encrypted_size );
    RSA_public_encrypt( DEFAULT_CHALLENGE_SIZE, cl->challenge, encrypted, pubkey, RSA_PKCS1_PADDING );
    if( WriteExact(cl, encrypted, encrypted_size ) < 0 )
    {
        rfbLogPerror("rfbAuthNewClient: write");
        rfbCloseClient(cl);
        return;
    }
    free( encrypted );
    RSA_free( pubkey );
    free( file );
    free( uname );
#endif
}

/*
 * rfbAuthProcessClientMessage is called when the client sends its
 * authentication response.
 */

void
rfbAuthProcessClientMessage(cl)
    rfbClientPtr cl;
{
#ifdef RSA_CRYPT_AVAILABLE
    int n;
    uint32_t authResult;
    char response[DEFAULT_CHALLENGE_SIZE];

    if( ( n = ReadExact( cl, (char *)response, DEFAULT_CHALLENGE_SIZE ) ) <= 0 )
    {
        rfbLogPerror( "rfbAuthProcessClientMessage: read" );
        rfbCloseClient( cl );
        return;
    }
    if( memcmp( response, cl->challenge, DEFAULT_CHALLENGE_SIZE ) != 0 )
    {
        rfbLog( "Authentication failed!\n" );
        authResult = Swap32IfLE( rfbVncAuthFailed );
    }
    else
    {
        rfbLog( "Authentication succeeded.\n" );
        authResult = Swap32IfLE( rfbVncAuthOK );
    }

    if( WriteExact( cl, (char *)&authResult, 4 ) < 0 )
    {
        rfbLogPerror( "rfbAuthProcessClientMessage: write" );
        rfbCloseClient( cl );
        return;
    }
    if( authResult == Swap32IfLE( rfbVncAuthFailed ) )
    {
        rfbCloseClient( cl );
    }

    rsaFreeChallenge( cl->challenge );
#endif
    cl->state = RFB_INITIALISATION;
}
