/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: FirewallDialog.cpp,v 1.27 2004/07/13 22:42:58 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"
#include "platforms.h"

#include "FWBTree.h"
#include "FirewallDialog.h"
#include "ObjectManipulator.h"
#include "DialogFactory.h"
#include "FWWindow.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/Management.h"
#include "fwbuilder/FWException.h"
#include "fwbuilder/Resources.h"

#include <vector>
#include <map>

#include <qlineedit.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qtextedit.h>
#include <qcombobox.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qiconview.h>

#include <iostream>

using namespace std;
using namespace libfwbuilder;

void FirewallDialog::loadFWObject(FWObject *o)
{
    obj=o;
    Firewall *s = dynamic_cast<Firewall*>(obj);
    assert(s!=NULL);

    init=true;

    fillLibraries(libs,obj);

/* fill in platform */
    setPlatform(platform, obj->getStr("platform").c_str() );

    fillVersion();

/* fill in host OS  */
    setHostOS(hostOS, obj->getStr("host_OS").c_str() );

/* ---------------- */

    Management *mgmt=s->getManagementObject();
    assert(mgmt!=NULL);

//    FWOptions  *opt =s->getOptionsObject();

    obj_name->setText( QString::fromUtf8(s->getName().c_str()) );
    snmpCommunity->setText( mgmt->getSNMPManagement()->getReadCommunity().c_str() );

    comment->setTextFormat(QTextEdit::PlainText);

    comment->setText( QString::fromUtf8(s->getComment().c_str()) );

    apply->setEnabled( false );


    obj_name->setEnabled(!o->isReadOnly());
    setDisabledPalette(obj_name);

    libs->setEnabled(!o->isReadOnly());
    setDisabledPalette(libs);

    platform->setEnabled(!o->isReadOnly());
    setDisabledPalette(platform);

    version->setEnabled(!o->isReadOnly());
    setDisabledPalette(version);

    hostOS->setEnabled(!o->isReadOnly());
    setDisabledPalette(hostOS);

    fwAdvanced->setEnabled(!o->isReadOnly());
    setDisabledPalette(fwAdvanced);

    osAdvanced->setEnabled(!o->isReadOnly());
    setDisabledPalette(osAdvanced);

    comment->setReadOnly(o->isReadOnly());
    setDisabledPalette(comment);

    snmpCommunity->setEnabled(!o->isReadOnly());
    setDisabledPalette(snmpCommunity);

    fwTemplatesView->setEnabled(!o->isReadOnly());
    setDisabledPalette(fwTemplatesView);


    init=false;
}

/* fill in version */
void FirewallDialog::fillVersion()
{
    version->clear();

    QMap<QString,QString> vl=getVersionsForPlatform( readPlatform(platform) );
    QString                v=obj->getStr("version").c_str();
    int cp=0;
    for (QMap<QString,QString>::iterator i1=vl.begin(); i1!=vl.end(); i1++,cp++)
    {
        version->insertItem( i1.data() );
        if ( v == i1.key() ) version->setCurrentItem( cp );
    }
}

void FirewallDialog::saveVersion()
{
    QString pl = readPlatform(platform);

    QMap<QString,QString> vl=getVersionsForPlatform( pl.latin1() );
    QString               v = version->currentText();
    int cp=0;
    for (QMap<QString,QString>::iterator i1=vl.begin(); i1!=vl.end(); i1++,cp++)
    {
        if ( v == i1.data() )
        {
            obj->setStr("version", i1.key().latin1() );
            break;
        }
    }
    
}

void FirewallDialog::platformChanged()
{
    fillVersion();
    changed();

    QString so=Resources::platform_res[readPlatform(platform).latin1()]->getResourceStr("/FWBuilderResources/Target/supported_os").c_str();
    if (so.isEmpty()) return;

    cerr << "so=" << so << endl;

    QString ho=so.section(",",0);

    cerr << "ho=" << ho << endl;

    setHostOS( hostOS, ho.latin1() );
}

void FirewallDialog::changed()
{
    apply->setEnabled( true );
}

void FirewallDialog::validate(bool *res)
{
    *res=true;
}

void FirewallDialog::isChanged(bool *res)
{
    *res=(!init && apply->isEnabled());
}

void FirewallDialog::libChanged()
{
    changed();
}

void FirewallDialog::applyChanges()
{
    if (!isTreeReadWrite(this,obj)) return;
    if (!validateName(this,obj,obj_name->text())) return;

    bool res=true;
    validate( &res );
    if (!res) return;

    Firewall *s = dynamic_cast<Firewall*>(obj);
    Management *mgmt=s->getManagementObject();
    assert(mgmt!=NULL);

//    FWOptions  *opt =s->getOptionsObject();

    assert(s!=NULL);

    string oldname=obj->getName();
    string newname=string(obj_name->text().utf8());
    string oldplatform=obj->getStr("platform");

    obj->setName( newname );
    obj->setComment( string(comment->text().utf8()) );
    mgmt->getSNMPManagement()->setReadCommunity( snmpCommunity->text().latin1() );

    om->updateObjName(obj,QString::fromUtf8(oldname.c_str()));

    string pl = readPlatform(platform).latin1();
    obj->setStr("platform", pl );

    obj->setStr("host_OS", readHostOS(hostOS).latin1() );

    saveVersion();

    if (oldplatform!=pl || oldname!=newname)  mw->reopenFirewall();

    init=true;

/* move to another lib if we have to */
    if (! FWBTree::isSystem(obj) && libs->currentText() != QString(obj->getLibrary()->getName().c_str()))
        om->moveObject(libs->currentText(), obj);

    init=false;

    apply->setEnabled( false );
}

void FirewallDialog::discardChanges()
{
    loadFWObject(obj);
}

void FirewallDialog::openFWDialog()
{
    if (apply->isEnabled()) applyChanges();

    if (obj->getStr("version").empty()) saveVersion();

    QWidget *w = DialogFactory::createFWDialog(this,obj);
    if (w==NULL)   return;   // some dialogs may not be implemented yet

    QDialog *d=dynamic_cast<QDialog*>(w);
    assert(d!=NULL);

    d->exec();   // modal dialog, dialog saves data into the object
}


void FirewallDialog::openOSDialog()
{
    if (apply->isEnabled()) applyChanges();

    QWidget *w = DialogFactory::createOSDialog(this,obj);
    if (w==NULL)   return;   // some dialogs may not be implemented yet

    QDialog *d=dynamic_cast<QDialog*>(w);
    assert(d!=NULL);

    d->exec();   // modal dialog, dialog saves data into the object
}

/* ObjectEditor class connects its slot to this signal and does all
 * the verification for us, then accepts (or not) the event. So we do
 * nothing here and defer all the processing to ObjectEditor
 */
void FirewallDialog::closeEvent(QCloseEvent *e)
{
    emit close_sign(e);

}

