#define SCROLL_TIMEOUT 100
#define BLINK_TIMEOUT 500
#define MARGIN 2
#define MARGIN_SPACING 6
#define TIMESTAMP_BUFFER 30
#define WORDWRAP_LIMIT 20

#include "config.h"
#include <string.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkselection.h>

#if defined(IMLIB) || defined(PIXBUF)
#   include <gdk/gdkx.h>
#endif
#ifdef PIXBUF
#   include <gdk-pixbuf/gdk-pixbuf.h>
#endif
#ifdef IMLIB
#   include <gdk_imlib.h>
#endif

#include "gtkgmo.h"

static GtkWidgetClass *parent_class = NULL;
enum {
    RESIZED_IN_CHARS,
    LAST_SIGNAL
};
static guint gmo_signals[LAST_SIGNAL] = { 0 };


static void gtk_gmo_init(GtkGmo *gmo);
static void gtk_gmo_adjustment_changed(GtkAdjustment *adj, GtkGmo *gmo);
static void gtk_gmo_adjustment_value_changed(GtkAdjustment *adj, GtkGmo *gmo);
static int gtk_gmo_button_release(GtkWidget *widget, GdkEventButton *event);
static int gtk_gmo_button_press(GtkWidget *widget, GdkEventButton *event);
static int gtk_gmo_motion_notify(GtkWidget *widget, GdkEventMotion *event);
static int gtk_gmo_leave_notify(GtkWidget *widget, GdkEventCrossing *event);

static void gtk_gmo_destroy(GtkObject *object);
static void gtk_gmo_render(GtkGmo *gmo);
static void gtk_gmo_calc_lines(GtkGmo *gmo);
int         gtk_gmo_line_used_lines(GtkGmo *gmo, GtkGmoLine *line);
GtkGmoLine *gtk_gmo_line_new       (GtkGmo *gmo, const char *line, int len);
void        gtk_gmo_line_free      (GtkGmoLine *l);
GtkGmoLine *gtk_gmo_get_line(GtkGmo *gmo, int x, int y, int in_screen,int *pos);

int  gtk_gmo_selection_clear(GtkGmo *gmo);
void gtk_gmo_line_get_line(GtkGmo *gmo, GtkGmoLine *line,
			   int n, int *begin, int *end);

static int  gtk_gmo_selection_clear_event(GtkWidget *widget, GdkEventSelection *event);
static void gtk_gmo_selection_get(GtkWidget *widget,
				  GtkSelectionData *selection_data_ptr,
				  guint info, guint time);


void gtk_gmo_update_background(GtkGmo *gmo);

#define MAX_ARGS 20
void gtk_gmo_line_set_ansi(GtkGmoLine *line, int pos,
			   int argv[], int argc, char command) {
    int i;

    if(command == 'm') { /* Only know the m command now - ansi text attributes*/
	for(i = 0; i < argc; i++) {
	    if(argv[i] == 0) {/* reset */
		ATTR_CLEAR(line->attr[pos]);
	    } else if(argv[i] == 1) {/* bold */
		ATTR_SET_BOLD(line->attr[pos]);
	    } else if(argv[i] == 4) {/* underline */
		ATTR_SET_UNDERLINE(line->attr[pos]);
	    } else if(argv[i] == 5) {/* blink */
		ATTR_SET_BLINK(line->attr[pos]);
	    } else if(argv[i] >= 30 && argv[i] <= 37) {
		ATTR_SET_FG(line->attr[pos], argv[i] - 30);
	    } else if(argv[i] >= 40 && argv[i] <= 47) {
		ATTR_SET_BG(line->attr[pos], argv[i] - 40);
	    } else if (debug)
		printf("Unknown ansi text attribute: %d\n", argv[i]);
	}
    } else if (debug)
	printf("Unknown escape command: %c (%d numeric args came with it\n",
	       command, argc);
}

#define BEEP     7
#define TAB      9
#define ESCAPE   27
GtkGmoLine *gtk_gmo_line_new(GtkGmo *gmo, const char *str, int len) {
    int i = 0;
    int args[MAX_ARGS];
    int arg;
    GtkGmoLine *line;

    if(debug) printf("GtkGmoLine::new()\n");

    line = g_malloc(sizeof(GtkGmoLine));
    line->line = g_malloc(len + 1);
    line->attr = g_malloc((len + 1) * sizeof(guint16));

    line->attr[i] = gmo->attr;
    while(*str) {
	if(*str == ESCAPE) {
	    str++;
	    if(*str == '[') {
		str++;
		arg = 0;
		args[arg] = 0;
		/* now let's read in the args in the args var. */
		while(*str) {
		    if(*str == ';') {
			arg++;
			args[arg] = 0;
		    } else if(*str >= '0' && *str <= '9') {
			if(arg < MAX_ARGS) {
			    args[arg] *= 10;
			    args[arg] += *str - '0';
			}
		    } else {
			arg = MIN(arg + 1, MAX_ARGS);
			gtk_gmo_line_set_ansi(line, i, args, arg, *str);
			str++;
			break;
		    }
		    str++;
		}
	    }
	} else if (*str == BEEP) {
	    if(gmo->beep) gdk_beep();
	    str++;
	} else if (*str == TAB) {
	    arg = 8 - i % 8;
	    if(!arg) arg = 8;

	    len += arg;
	    line->line = g_realloc(line->line, len + 1);
	    line->attr = g_realloc(line->attr, (len + 1) * sizeof(guint16));
	    for(; arg > 0; arg--) {
		line->attr[i+1] = line->attr[i];
		line->line[i++] = ' ';
	    }
	    str++;
	} else {
	    line->attr[i+1] = line->attr[i];
	    line->line[i++] = *str;
	    str++;
	}
    }
    line->line[i] = '\0';
    gmo->attr = line->attr[i];

    line->length = i;
    if(debug) printf("  %d plain chars\n", i);

    line->used_lines = gtk_gmo_line_used_lines(gmo, line);
    if(debug) printf("  %d lines used with current screensize (%d)\n",
		     line->used_lines, gmo->width);

    line->select_start = line->select_end = -1;
    line->timestamp = time(NULL);

    return line;
}

void gtk_gmo_line_free(GtkGmoLine *line) {
    g_free(line->line);
    g_free(line->attr);
    g_free(line);
}

void gtk_gmo_set_adjustments(GtkGmo *gmo,
			     GtkAdjustment *hadj, GtkAdjustment *vadj) {
    if(!vadj) {
	vadj = (GtkAdjustment *) gtk_adjustment_new(0, 0, 0, 1, 0, 0);
    }

    if(gmo->adj && gmo->adj != vadj) {
	vadj->step_increment = 1;
	vadj->lower = gmo->adj->lower;
	vadj->upper = gmo->adj->upper;
	vadj->value = gmo->adj->value;
	vadj->page_size = gmo->adj->page_size;
	vadj->page_increment = gmo->adj->page_increment;
	gtk_signal_disconnect_by_data(GTK_OBJECT(gmo->adj), gmo);
	gtk_object_unref(GTK_OBJECT(gmo->adj));
    }
    gmo->adj = vadj;
    gtk_object_ref((GtkObject *) gmo->adj);
    gtk_object_sink((GtkObject *) gmo->adj);
    gtk_signal_connect(GTK_OBJECT(gmo->adj), "changed",
		       GTK_SIGNAL_FUNC(gtk_gmo_adjustment_changed), gmo);
    gtk_signal_connect(GTK_OBJECT(gmo->adj), "value_changed",
		       GTK_SIGNAL_FUNC(gtk_gmo_adjustment_value_changed),
		       gmo);
}



static void gtk_gmo_init(GtkGmo *gmo) { 
    if(debug) printf("GtkGmo::init()\n");
    gmo->adj = (GtkAdjustment *) gtk_adjustment_new(0, 0, 0, 1, 0, 0);
    gtk_object_ref((GtkObject *) gmo->adj);
    gtk_object_sink((GtkObject *) gmo->adj);
    gtk_signal_connect(GTK_OBJECT(gmo->adj), "changed",
		       GTK_SIGNAL_FUNC(gtk_gmo_adjustment_changed), gmo);
    gtk_signal_connect(GTK_OBJECT(gmo->adj), "value_changed",
		       GTK_SIGNAL_FUNC(gtk_gmo_adjustment_value_changed),
		       gmo);

    gmo->last_value = -1;

    gmo->buffer = NULL;
    gmo->background = NULL;
    gmo->bgfilename = NULL;
    gmo->transparent = FALSE;
    gmo->tinted = FALSE;
    gmo->tint_r = gmo->tint_g = gmo->tint_b = 0xFF;
    gmo->tiled = TRUE;

    gmo->fgc = gmo->bgc = NULL;

    gmo->first_line = gmo->last_line = NULL;
    gmo->num_lines = 0;
    gmo->max_lines = 0;

    gmo->font = NULL;
    gmo->height = gmo->width = 0;
    gmo->charheight = gmo->charwidth = 0;

    gmo->margin_width = MARGIN;
    gmo->timestamp = NULL;

    gmo->scroll_id = -1;
    gmo->redraw_id = -1;

    gmo->blink_id = -1;
    gmo->blinking = FALSE;
    gmo->blink_on = TRUE;
    gmo->trans_refresh_id = -1;

    ATTR_CLEAR(gmo->attr);

    gmo->selecting = FALSE;
    gmo->beep = TRUE;
    gmo->scroll_on_text = FALSE;

    gtk_selection_add_target(GTK_WIDGET (gmo),
			     GDK_SELECTION_PRIMARY,
			     GDK_SELECTION_TYPE_STRING,
			     1);
}


GtkWidget *gtk_gmo_new() {
    if(debug) printf("GtkGmo::new()\n");
    return GTK_WIDGET(gtk_type_new(gtk_gmo_get_type()));
}

static void gtk_gmo_destroy(GtkObject *object) {
    GtkGmo *gmo = GTK_GMO(object);

    if(debug) printf("GtkGmo::destroy()\n");

    if(gmo->scroll_id != -1) {
	gtk_timeout_remove(gmo->scroll_id);
	gmo->scroll_id = -1;
    }
    if(gmo->redraw_id != -1) {
	gtk_timeout_remove(gmo->redraw_id);
	gmo->redraw_id = -1;
    }
    if(gmo->blink_id != -1) {
	gtk_timeout_remove(gmo->blink_id);
	gmo->blink_id = -1;
    }

    if(gmo->background) {
	if(debug) printf("  Unreferencing background pixmap\n");
	gdk_pixmap_unref(gmo->background);
	gmo->background = NULL;
    }

    if (gmo->font) {
	if(debug) printf("  Unreferencing font\n");
	gdk_font_unref(gmo->font);
	gmo->font = NULL;
    }

    if (gmo->adj) {
	if(debug) printf("  Disconnecting the adjustement\n");
	gtk_signal_disconnect_by_data(GTK_OBJECT(gmo->adj), gmo);
	gtk_object_unref(GTK_OBJECT(gmo->adj));
	gmo->adj = NULL;
    }

    if(debug) printf("  Free()ing output data\n");
    g_list_foreach(gmo->first_line, (GFunc) gtk_gmo_line_free, NULL);
    g_list_free(gmo->first_line);
    gmo->first_line = gmo->last_line = NULL;

    if(debug) printf("  Forwarding signal to parent...\n");
    if(GTK_OBJECT_CLASS(parent_class)->destroy)
	(*GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}

static void gtk_gmo_realize(GtkWidget *widget) {
    GtkGmo *gmo;
    GdkWindowAttr attributes;
    GdkGCValues val;

    if(debug) printf("GtkGmo::realize()\n");

    GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
    gmo = GTK_GMO(widget);

    attributes.x = widget->allocation.x;
    attributes.y = widget->allocation.y;
    attributes.width = widget->allocation.width;
    attributes.height = widget->allocation.height;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.event_mask = gtk_widget_get_events(widget) |
	GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	GDK_POINTER_MOTION_MASK | GDK_LEAVE_NOTIFY_MASK;
    attributes.visual = gtk_widget_get_visual(widget);
    attributes.colormap = gtk_widget_get_colormap(widget);

    widget->window = gdk_window_new(widget->parent->window, &attributes,
				    GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL |
				    GDK_WA_COLORMAP);

    gdk_window_set_user_data(widget->window, widget);

    val.subwindow_mode = GDK_INCLUDE_INFERIORS;
    val.graphics_exposures = 0;

    gmo->bgc = gdk_gc_new_with_values (widget->window, &val,
				       GDK_GC_EXPOSURES | GDK_GC_SUBWINDOW);
    gmo->fgc = gdk_gc_new_with_values (widget->window, &val,
				       GDK_GC_EXPOSURES | GDK_GC_SUBWINDOW);

    if(gmo->font == NULL)
	gtk_gmo_set_font(gmo, NULL, "fixed");
    if(gmo->font)
	gdk_gc_set_font(gmo->fgc, gmo->font);

    gdk_gc_set_foreground(gmo->fgc, &gmo->colors[FG_COLOR]);
    gdk_gc_set_foreground(gmo->bgc, &gmo->colors[BG_COLOR]);
}

static void gtk_gmo_unrealize(GtkWidget *widget) {
    GtkGmo *gmo = GTK_GMO(widget);

    if(debug) printf("GtkGmo::unrealize()\n");

    if(debug) printf("  Unrealising GdkWindow\n");
    gdk_window_destroy(widget->window);
    widget->window = NULL;

    if(debug) printf("  Destroying graphics contents\n");
    gdk_gc_destroy(gmo->bgc);
    gmo->bgc = NULL;
    gdk_gc_destroy (gmo->fgc);
    gmo->fgc = NULL;
}

static void gtk_gmo_size_request(GtkWidget *widget, GtkRequisition *requisition) {
    if(debug) printf("GtkGmo::size_request()\n");
    requisition->width  = GTK_GMO(widget)->margin_width + 300;
    requisition->height = 100;
}

static void gtk_gmo_size_allocate(GtkWidget *widget, GtkAllocation *allocation) {
    GtkGmo *gmo = GTK_GMO(widget);
    int scroll_down = FALSE;
    int width, height, resized = FALSE;

    if(debug) printf("GtkGmo::size_allocate()\n");

    if (allocation->width == widget->allocation.width &&
	allocation->height == widget->allocation.height &&
	allocation->x == widget->allocation.x &&
	allocation->y == widget->allocation.y) {
	if(debug) printf("  Nothing to do.\n");
	return;
    } 
    if(gmo->buffer) { /* Resizeing, clear the buffer first. */
	gdk_pixmap_unref(gmo->buffer);
	gmo->buffer = NULL;
    }

    widget->allocation = *allocation;

    width  = (GTK_WIDGET(gmo)->allocation.width -
	      (gmo->margin_width + 2 * MARGIN)) / gmo->charwidth;
    height = (GTK_WIDGET(gmo)->allocation.height - MARGIN) / gmo->charheight;
    if(gmo->width != width || gmo->height != height)
	resized = TRUE;
    gmo->width = width;
    gmo->height = height;

    if(gmo->adj->value >= gmo->adj->upper - gmo->adj->page_size)
	scroll_down = TRUE;

    gtk_gmo_calc_lines(gmo);

    if(scroll_down || gmo->adj->value >= gmo->adj->upper - gmo->adj->page_size){
	gmo->adj->value = MAX((int) gmo->adj->upper - gmo->height, 0);
    }
    gmo->adj->page_size = MIN(gmo->height, gmo->adj->upper);
    gmo->adj->page_increment = gmo->adj->page_size;

    if(GTK_WIDGET_REALIZED (widget)) {
	gdk_window_move_resize(widget->window,
			       allocation->x, allocation->y, 
			       allocation->width, allocation->height);
	if(gmo->background) {
	    gdk_pixmap_unref(gmo->background);
	    gmo->background = NULL;
	}
    }
    if(resized)
	gtk_signal_emit(GTK_OBJECT(gmo), gmo_signals[RESIZED_IN_CHARS],
			width, height);
}

static int gtk_gmo_expose(GtkWidget *w, GdkEventExpose *event) {
    GdkRectangle r;
    GtkGmo *gmo = GTK_GMO(w);
    if(debug) printf("GtkGmo::expose()");
    if(gmo->buffer) { /* Ha, cached buffer, let's just draw it ok :) */
	r = event->area;
	if(debug) printf(" - cached screen, copying %dx%d+%d+%d\n",
			 r.width, r.height, r.x, r.y);
	gdk_draw_pixmap(w->window, gmo->fgc, gmo->buffer,
			r.x, r.y, r.x, r.y, r.width, r.height);

    } else { /* No cached buffer, probably resizeing the window. */ 
	if(debug) printf(" - nothing cached.\n");
	gtk_gmo_render(gmo);
    }
    return FALSE;
}

static void gtk_gmo_class_init(GtkGmoClass *class) {
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;
    GtkGmoClass *gmo_class;

    if(debug) printf("GtkGmo::class_init()\n");

    object_class = (GtkObjectClass *) class;
    widget_class = (GtkWidgetClass *) class;
    gmo_class = (GtkGmoClass *) class;

    parent_class = gtk_type_class(gtk_widget_get_type());

    gmo_signals[RESIZED_IN_CHARS] =
	gtk_signal_new("resized_in_chars",            /*name*/
		       GTK_RUN_FIRST,                 /*GtkSignalRunType*/
		       object_class->type,            /*GtkType*/
		       GTK_SIGNAL_OFFSET(GtkGmoClass, /*funcoffset*/
					 resized_in_chars),
		       gtk_marshal_NONE__INT_INT,     /*GtkSignalMarshaller*/
		       GTK_TYPE_NONE,                 /*returnval*/
		       2,                             /*num args*/
		       GTK_TYPE_INT,                  /*args*/
		       GTK_TYPE_INT);
    gtk_object_class_add_signals(object_class, gmo_signals, LAST_SIGNAL);

    object_class->destroy = gtk_gmo_destroy;

    widget_class->realize = gtk_gmo_realize;
    widget_class->unrealize = gtk_gmo_unrealize;

    widget_class->size_request = gtk_gmo_size_request;
    widget_class->size_allocate = gtk_gmo_size_allocate;

    widget_class->expose_event = gtk_gmo_expose;

    widget_class->button_press_event = gtk_gmo_button_press;
    widget_class->button_release_event = gtk_gmo_button_release;
    widget_class->motion_notify_event = gtk_gmo_motion_notify;
    widget_class->leave_notify_event = gtk_gmo_leave_notify;

    widget_class->selection_clear_event   = gtk_gmo_selection_clear_event;
    widget_class->selection_get           = gtk_gmo_selection_get;

    widget_class->set_scroll_adjustments_signal =
	gtk_signal_new("set_scroll_adjustments",
		       GTK_RUN_LAST,
		       object_class->type,
		       GTK_SIGNAL_OFFSET(GtkGmoClass, set_scroll_adjustments),
		       gtk_marshal_NONE__POINTER_POINTER,
		       GTK_TYPE_NONE, 2,
		       GTK_TYPE_ADJUSTMENT, GTK_TYPE_ADJUSTMENT);

    gmo_class->resized_in_chars = NULL;
    gmo_class->set_scroll_adjustments = gtk_gmo_set_adjustments;
}

guint gtk_gmo_get_type() {
    static guint gmo_type = 0;

    if(!gmo_type) {
	GtkTypeInfo gmo_info = {
	    "GtkGmo",
	    sizeof (GtkGmo),
	    sizeof (GtkGmoClass),
	    (GtkClassInitFunc) gtk_gmo_class_init,
	    (GtkObjectInitFunc) gtk_gmo_init,
	    (GtkArgSetFunc) NULL,
	    (GtkArgGetFunc) NULL,
	};
	gmo_type = gtk_type_unique(gtk_widget_get_type(), &gmo_info);
    }

    return gmo_type;
}


/**************** Selecting and stuff ****************************************/
static int  gtk_gmo_selection_clear_event(GtkWidget *widget, GdkEventSelection *event) {
    GtkGmo *gmo = GTK_GMO(widget);
    if(debug) printf("\x1b[35mGtkGmo::selection_clear_event()\x1b[0m\n");

    if(!gtk_selection_clear(widget, event)) {
	if(debug) printf("\x1b[35m\tHEJ! gtk_selection_clear() wasn't TRUE!!\n\t[Is this a good thing?]\x1b[0m\n");
	return FALSE;
    }

    if(gtk_gmo_selection_clear(gmo))
	gtk_gmo_render(gmo);
    if(gdk_selection_owner_get(GDK_SELECTION_PRIMARY) == widget->window) {
	if(debug) printf("\x1b[35m\tUnclaiming selection\x1b[0m\n");
	gtk_selection_owner_set(NULL, GDK_SELECTION_PRIMARY, event->time);
    }
    return TRUE;
}

static void gtk_gmo_selection_get(GtkWidget *widget,
				  GtkSelectionData *selection_data,
				  guint info, guint time) {
    char *txt, *pos;
    int len = 1;
    GList *list;
    GtkGmoLine *line;
    int newline;
    GtkGmo *gmo = GTK_GMO(widget);

    if(debug) printf("\x1b[35mGtkGmo::selection_get()\n");

    for(list = gmo->first_line; list; list = g_list_next(list)) {
	line = list->data;
	if(line->select_start > -1) {
	    len += (line->select_end - line->select_start) + 2;
	}
    }
    txt = pos = g_malloc(len);
    newline = FALSE;
    for(list = gmo->first_line; list; list = g_list_next(list)) {
	line = list->data;
	if(line->select_start > -1) {
	    if(newline) {
		*pos = '\n';
		pos++;
	    }
	    memcpy(pos, line->line + line->select_start,
		   line->select_end - line->select_start);
	    pos += line->select_end - line->select_start;
	    newline = TRUE;
	}
    }
    *pos = '\0';

    if(debug) printf("\tsending: \"%s\"\x1b[0m\n", txt);
    gtk_selection_data_set(selection_data, GDK_SELECTION_TYPE_STRING,
			   8, txt, strlen(txt));
    g_free(txt);
}


void gtk_gmo_put_selection_in_lines(GtkGmo *gmo) {
    int ux, uy, lx, ly;
    GtkGmoLine *start, *end;
    int pos;
    GList *list;

    if(gmo->select_start_y < gmo->select_end_y ||
       (gmo->select_start_y == gmo->select_end_y &&
	gmo->select_start_x <= gmo->select_end_x)) {
	ux = gmo->select_start_x;
	uy = gmo->select_start_y;
	lx = gmo->select_end_x;
	ly = gmo->select_end_y;
    } else {
	ux = gmo->select_end_x;
	uy = gmo->select_end_y;
	lx = gmo->select_start_x;
	ly = gmo->select_start_y;
    }

    start = gtk_gmo_get_line(gmo, ux, uy, FALSE, &pos);
    start->select_start = pos;
    end = gtk_gmo_get_line(gmo, lx, ly, FALSE, &pos);
    end->select_end = pos;

    if(start != end) {
	list = g_list_find(gmo->first_line, start);
	while(start != end) {
	    start->select_end = start->length;
	    list = g_list_next(list);
	    start = list->data;
	    start->select_start = 0;
	}
    }
}

void gtk_gmo_selection_update(GtkGmo *gmo, int x, int y, int time, int render) {
    GList *list;
    GtkGmoLine *line;
    GdkWindow *w;
    int ok;

    if(gmo->select_end_x != x || gmo->select_end_y != y) {
	gmo->select_end_x = x;
	gmo->select_end_y = y;
	
	gmo->select_start_x = MAX(gmo->select_start_x, 0);
	gmo->select_end_x   = MAX(gmo->select_end_x,   0);

	if(debug) printf("Selected from: %d,%d to %d,%d - rerendering\n",
			 gmo->select_start_x, gmo->select_start_y,
			 gmo->select_end_x, gmo->select_end_y);
	/* clear the currend pared selection in the GtkGmoLine's */
	for(list = gmo->first_line; list; list = g_list_next(list)) {
	    line = list->data;
	    line->select_start = line->select_end = -1;
	}
	gtk_gmo_put_selection_in_lines(gmo);
	if((w = gdk_selection_owner_get(GDK_SELECTION_PRIMARY)) != GTK_WIDGET(gmo)->window) {
	    if(debug) printf("# widget is: %x  owner is %x -> Setting#\n", (int)gmo, (int)w);
	    ok = gtk_selection_owner_set(GTK_WIDGET(gmo),
					 GDK_SELECTION_PRIMARY, time);
	    if(!ok) {
		if(debug) printf("\x1b[31m**** Error, couldn't claim selection ! :(\x1b[0m\n");
		gtk_gmo_selection_clear(gmo);
	    }
	}
	if(render)
	    gtk_gmo_render(gmo);
    }
}

int gtk_gmo_selection_clear(GtkGmo *gmo) {
    GList *list;
    GtkGmoLine *line;
    int ret = FALSE;

    if(debug) printf("GtkGmo::selection_clear()\n");

    for(list = gmo->first_line; list; list = g_list_next(list)) {
	line = list->data;
	if(line->select_start >= 0 || line->select_end >= 0)
	    ret = TRUE;
	line->select_start = line->select_end = -1;
    }

    gmo->select_start_x = gmo->select_start_y = -1;
    gmo->select_end_x = gmo->select_end_y = -1;

    return ret;
}

GtkGmoLine *gtk_gmo_get_line(GtkGmo *gmo, int x, int y, int in_screen, int *pos) {
    GList *list;
    GtkGmoLine *line = NULL;
    int begin, end;
    int i;

    if(debug) printf("  Getting line for %dx%d\n", x, y);

    for(list = gmo->first_line; list; list = g_list_next(list)) {
	line = list->data;
	if(y < line->used_lines)
	    break;
	y -= line->used_lines;
    }
    if(!list) {
	if(in_screen || !gmo->last_line)
	    return NULL;
	line = gmo->last_line->data;
	y = line->used_lines - 1;
    }

    y = MAX(y, 0);
    *pos = 0;
    for(i = 0; i < y; i++) {
	gtk_gmo_line_get_line(gmo, line, i, &begin, &end);
	*pos += end - begin;
    }
    gtk_gmo_line_get_line(gmo, line, y, &begin, &end);
    if(x <= end - begin) { /* the x fits in the line */
	*pos += x;
    } else if (in_screen) { /* x > line length, and must be in the screen */
	return NULL;
    } else {                /* x > line lengt, return the line length */
	*pos += end - begin;
    }

    return line;
}

int gtk_gmo_line_width(GtkGmo *gmo, int y) {
    GtkGmoLine *line = NULL;
    int begin, end;
    GList *list;

    for(list = gmo->first_line; list; list = g_list_next(list)) {
	line = list->data;
	if(y < line->used_lines)
	    break;
	y -= line->used_lines;
    }

    gtk_gmo_line_get_line(gmo, line, y, &begin, &end);
    return end - begin;
}


void gtk_gmo_select_word(GtkGmo *gmo, int x, int y, GdkEventButton *event) {
    GtkGmoLine *line;
    int pos;
    int offset;
    int len;

    if(debug) printf("GtkGmo::select_word() - double click handler\n");
    gmo->select_start_x = x;
    gmo->select_start_y = y;

    if(y > (int) gmo->adj->upper) {
	if(debug) printf("  Not clicked on visible text (below.)!\n"
			 "    Clicked on line %d of %d\n", 
			 y, (int)gmo->adj->upper);
	return;
    }
    if(x < 0)
	return;

    if((line = gtk_gmo_get_line(gmo, x, y, TRUE, &pos))) {
	if(line->line[pos] == ' ') { /* user managed to d-click on a space ;) */
	    if(debug) printf("  Great job - you hit a space ;)\n");
	    return;
	}
	/* TODO: change the harcoded ' ' in a uses selectable list of chars. */

	/* Set the start position */
	offset = pos - x;
	while(gmo->select_start_x + offset > 0 &&
	      line->line[gmo->select_start_x + offset] != ' ') {
	    gmo->select_start_x--;
	    if(gmo->select_start_x < 0) {
		gmo->select_start_y = MAX(0, gmo->select_start_y - 1);
		len = gtk_gmo_line_width(gmo, gmo->select_start_y);
		gmo->select_start_x += len;
		offset = MAX(0, offset - len);
	    }
	}
	if(line->line[gmo->select_start_x + offset] == ' ')
	    gmo->select_start_x++;

	/* Set the endposition */
	offset = pos - x;
	len = gtk_gmo_line_width(gmo, y);
	while(x + offset < line->length && line->line[x + offset] != ' ') {
	    x++;
	    if(x > len) {
		x = 0;
		offset += len;
		y++;
		len = gtk_gmo_line_width(gmo, y);
	    }
	}

	gtk_gmo_selection_update(gmo, x, y, event->time, TRUE);
    } else if(debug) {
	printf("  Not clicked on visible text !\n");
    }
}


static int gtk_gmo_leave_notify(GtkWidget *widget, GdkEventCrossing *event) {
    if(debug) printf("GtkGmo::leave_notify()\n");
    return FALSE;
}

int gtk_gmo_scroll_up_timeout(GtkGmo *gmo) {
    int x, y;

    if(debug) printf("Scroll up\n");

    gdk_window_get_pointer(GTK_WIDGET(gmo)->window, &x, &y, 0);
    x -= gmo->margin_width;
    x /= gmo->charwidth;
    y = (int) gmo->adj->value - 1;

    if(y < 0) {
	gtk_gmo_selection_update(gmo, x, y, GDK_CURRENT_TIME, TRUE);
	gmo->scroll_id = -1;
	return FALSE; /* Let's not be called again */
    }

    gmo->adj->value = y;
    gtk_gmo_selection_update(gmo, x, y, GDK_CURRENT_TIME, FALSE);
    gtk_adjustment_value_changed(gmo->adj);

    return TRUE;
}

int gtk_gmo_scroll_down_timeout(GtkGmo *gmo) {
    int x, y;

    if(debug) printf("Scroll down\n");

    gdk_window_get_pointer(GTK_WIDGET(gmo)->window, &x, &y, 0);
    x -= gmo->margin_width;
    x /= gmo->charwidth;
    y = (int) gmo->adj->value + 1;

    if(y > (int) gmo->adj->upper - (int) gmo->adj->page_size) {
	gtk_gmo_selection_update(gmo, x, y + (int)gmo->adj->page_size,
				 GDK_CURRENT_TIME, TRUE);
	gmo->scroll_id = -1;
	return FALSE; /* Let's not be called again */
    }

    gmo->adj->value = y;
    gtk_gmo_selection_update(gmo, x, y + (int)gmo->adj->page_size,
			     GDK_CURRENT_TIME, FALSE);
    gtk_adjustment_value_changed(gmo->adj);

    return TRUE;
}

    void gtk_gmo_scroll_up(GtkGmo *gmo) {
	if(gmo->scroll_id != -1)
	    return;
	gmo->scroll_id = gtk_timeout_add(SCROLL_TIMEOUT,
					 (GtkFunction) gtk_gmo_scroll_up_timeout,
					 gmo);
    }

    void gtk_gmo_scroll_down(GtkGmo *gmo) {
	if(gmo->scroll_id != -1)
	    return;
	gmo->scroll_id = 
	    gtk_timeout_add(SCROLL_TIMEOUT,
			    (GtkFunction) gtk_gmo_scroll_down_timeout,
			    gmo);
    }

static int gtk_gmo_motion_notify(GtkWidget *widget, GdkEventMotion *event) {
    GtkGmo *gmo = GTK_GMO(widget);
    int x, y;

    if(gmo->selecting) {
	if(debug) printf("GtkGmo::motion_notify() - and selecting ...\n");

	gtk_grab_add(widget);
	gdk_window_get_pointer(widget->window, &x, &y, 0);
	x -= gmo->margin_width;
	x /= gmo->charwidth;
	y = y / gmo->charheight + (int) gmo->adj->value;
	if(y < (int) gmo->adj->value && gmo->adj->value > gmo->adj->lower) {
	    gtk_gmo_scroll_up(gmo);
	} else if(y > (int) gmo->adj->value + (int) gmo->adj->page_size &&
		  gmo->adj->value < gmo->adj->upper - gmo->adj->page_size) {
	    gtk_gmo_scroll_down(gmo);
	} else {
	    if(gmo->scroll_id != -1) {
		gtk_timeout_remove(gmo->scroll_id);
		gmo->scroll_id = -1;
	    }
	    x = CLAMP(x, 0, gmo->width);
	    gtk_gmo_selection_update(gmo, x, y, event->time, TRUE);
	}
    }
    return FALSE;
}

static int gtk_gmo_button_press(GtkWidget *widget, GdkEventButton *event) {
    GtkGmo *gmo = GTK_GMO(widget);
    int x, y;
    int need_redraw;

    if(debug) printf("GtkGmo::button_press()\n");

    gdk_window_get_pointer(widget->window, &x, &y, 0);
    x -= gmo->margin_width;
    x /= gmo->charwidth;
    y = y / gmo->charheight + (int) gmo->adj->value;

    if(event->button == 3) { /* right click */
	if(debug) printf("  TODO: Right click - popup some menu or so...\n");
    } else if (event->button == 1) {
	need_redraw = gtk_gmo_selection_clear(gmo);
	gmo->selecting = TRUE;
	if(event->type == GDK_2BUTTON_PRESS) { /* Select a word */
	    gmo->word_or_line_select = TRUE;
	    gtk_gmo_select_word(gmo, x, y, event);
	} else if (event->type == GDK_3BUTTON_PRESS) { /* Select a line */
	    if(debug) printf("  Tripple click:\n");
	    gmo->select_start_x = 0;
	    gmo->select_start_y = y;
	    gmo->word_or_line_select = TRUE;
	    gtk_gmo_selection_update(gmo, gtk_gmo_line_width(gmo, y),
				     y, event->time, TRUE);
	} else {
	    gmo->select_start_x = x;
	    gmo->select_start_y = y;
	    if(need_redraw)
		gtk_gmo_render(gmo);
	}
    }
    return FALSE;
}

static int gtk_gmo_button_release(GtkWidget *widget, GdkEventButton *event) {
    GtkGmo *gmo = GTK_GMO(widget);
    int x, y;

    if(debug) printf("GtkGmo::button_release()\n");

    if(gmo->selecting) {
	gmo->selecting = FALSE;
	gtk_grab_remove(widget);

	if(gmo->scroll_id != -1) {
	    gtk_timeout_remove(gmo->scroll_id);
	    gmo->scroll_id = -1;
	}

	if(!gmo->word_or_line_select) {
	    gdk_window_get_pointer(widget->window, &x, &y, 0);
	    x -= gmo->margin_width;
	    x /= gmo->charwidth;
	    y = y / gmo->charheight + (int) gmo->adj->value;
	    gtk_gmo_selection_update(gmo, x, y, event->time, TRUE);
	} else {
	    gmo->word_or_line_select = FALSE;
	}
    }
    return FALSE;
}


static void gtk_gmo_adjustment_changed(GtkAdjustment *adj, GtkGmo *gmo) {
    if(debug) printf("GtkGmo::adjustment_changed()\n  l,v,u,pz -> %d, "
		     "%d, %d, %d\n", (int) adj->lower, (int) adj->value,
		     (int) adj->upper, (int) adj->page_size);
    gtk_gmo_render(gmo);
}

static void gtk_gmo_adjustment_value_changed(GtkAdjustment *adj, GtkGmo *gmo) {
    if(debug) printf("GtkGmo::adjustment_value_changed()\n");
    if((int) adj->value != gmo->last_value) { /* must redraw now ! */
	if(debug) printf("  Value actually changed from %d to %d\n",
			 gmo->last_value, (int) adj->value);
	gtk_gmo_render(gmo);
    } else if (debug) {
	printf("  Nothing to do\n");
    }
}


void gtk_gmo_set_palette(GtkGmo *gmo, GdkColor palette[]) {
    int i;

    if(debug) printf("GtkGmo::set_palette()\n");

    for (i = 0; i < NUM_COLORS; i++) {
	gmo->colors[i].pixel = palette[i].pixel;
	gmo->colors[i].red   = palette[i].red;
	gmo->colors[i].green = palette[i].green;
	gmo->colors[i].blue  = palette[i].blue;
	if(debug) printf("  color %d -> %d %d %d\n", i,
			 palette[i].red, palette[i].green, palette[i].blue);

    }

    if (GTK_WIDGET_REALIZED(gmo)) {
	gdk_gc_set_foreground(gmo->fgc, &gmo->colors[FG_COLOR]);
	gdk_gc_set_background(gmo->fgc, &gmo->colors[BG_COLOR]);
	gdk_gc_set_foreground(gmo->bgc, &gmo->colors[BG_COLOR]);
	gtk_gmo_render(gmo);
    }
}

void gtk_gmo_set_font(GtkGmo *gmo, GdkFont *font, const char *name) {
    int width, height, resized = FALSE;
    char *tmp;
    if(debug) printf("GtkGmo::set_font(%s)\n",
		     font ? "GdkFont" : name ? "fontname" : "no font given");
    if(gmo->font)
	gdk_font_unref (gmo->font);

    if(font) {
	gmo->font = font;
    } else if(name) {
	font = gmo->font = gdk_font_load (name);
	if (!font)
	    font = gmo->font = gdk_font_load ("fixed");
    }

    if(gmo->font) {
	gdk_font_ref(font);

	gmo->charwidth = gdk_char_width(gmo->font, 'm');
	gmo->charheight = gmo->font->ascent + gmo->font->descent;

	/* update the ->margin_width */
	tmp = gmo->timestamp;
	gmo->timestamp = NULL;
	if(GTK_WIDGET_REALIZED(GTK_WIDGET(gmo))) {
	    GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(gmo), GTK_REALIZED);
	    gtk_gmo_set_timestamp(gmo, tmp);
	    GTK_WIDGET_SET_FLAGS(GTK_WIDGET(gmo), GTK_REALIZED);
	} else {
	    gtk_gmo_set_timestamp(gmo, tmp);
	} 
	g_free(tmp);

	width  = (GTK_WIDGET(gmo)->allocation.width - 
		  (gmo->margin_width + 2 * MARGIN)) / gmo->charwidth;
	height =(GTK_WIDGET(gmo)->allocation.height - MARGIN) / gmo->charheight;
	if(gmo->width != width || gmo->height != height)
	    resized = TRUE;
	gmo->width = width;
	gmo->height = height;

	gtk_gmo_calc_lines(gmo);
	if(resized)
	    gtk_signal_emit(GTK_OBJECT(gmo), gmo_signals[RESIZED_IN_CHARS],
			    width, height);
	gmo->adj->page_size = gmo->adj->page_increment = gmo->height;
	gmo->adj->value = gmo->adj->upper - gmo->adj->page_size;
	gmo->last_value = gmo->adj->value;

	gtk_gmo_render(gmo);
    }
}

char *find_eol(char *str, int max_len) {
    char *end = str;
    char *tmp;

    while(1) {
	tmp = strchr(end, ' ');
	if(!tmp) { /* no (more) matches */
	    if(end == str) /* no match at al */
		return str + max_len;
	    if(max_len - (end - str) > WORDWRAP_LIMIT)
		return str + max_len;
	    return end;
	}
	if(tmp - str > max_len) { /* one match to many */
	    if(max_len - (end - str) > WORDWRAP_LIMIT)
		return str + max_len;
	    return end;
	}
	end = tmp + 1;
    }
}

int gtk_gmo_line_used_lines(GtkGmo *gmo, GtkGmoLine *line) {
    int lines = 0;
    int len = line->length;
    char *str = line->line;
    char *end;

    if(gmo->width < 1)
	return 1;

    while(*str) {
	lines++;

	if(len <= gmo->width)
	    break;
	end = find_eol(str, gmo->width);
	len -= (end - str);
	str = end;
	if(*str == ' ') {
	    str++;
	    len--;
	}
    }

    return MAX(1, lines);
}


void gtk_gmo_calc_line(GtkGmoLine *line, GtkGmo *gmo) {
    line->used_lines = gtk_gmo_line_used_lines(gmo, line);
    gmo->adj->upper += line->used_lines;
}

static void gtk_gmo_calc_lines(GtkGmo *gmo) {
    if(debug) printf("GtkGmo::calc_lines()\n");

    gmo->adj->upper = 0;
    g_list_foreach(gmo->first_line, (GFunc) gtk_gmo_calc_line, gmo);
}

/* Check if the relevan attributes are equel, there are: bold, underline */
int attr_eq_4_rss(guint16 a, guint16 b) {
    return ATTR_IS_BOLD(a) == ATTR_IS_BOLD(b) &&
	ATTR_IS_UNDERLINE(a) == ATTR_IS_UNDERLINE(b);
}

void gtk_gmo_render_string_selected(GtkGmo *gmo, GtkGmoLine *line,
				    GdkDrawable *buffer,
				    int begin, int end,
				    int x, int y) {
    int b, e, len;

    b = begin;

    while(b < end) {
	for(e = b; e < end && attr_eq_4_rss(line->attr[b], line->attr[e]); e++);
	len = e - b;

	if(!len) /* Let's not render empty strings ;) */
	    continue;

	gdk_gc_set_foreground(gmo->fgc, &gmo->colors[BG_SELECT]);
	gdk_draw_rectangle(buffer, gmo->fgc, TRUE,
			   x, y - gmo->font->ascent,
			   len * gmo->charwidth, gmo->charheight);

	gdk_gc_set_foreground(gmo->fgc, &gmo->colors[FG_SELECT]);
	gdk_draw_text(buffer, gmo->font, gmo->fgc, x, y, line->line + b, len);
	if(ATTR_IS_BOLD(line->attr[b])) {
	    gdk_draw_text(buffer, gmo->font, gmo->fgc,
			  x + 1, y, line->line + b, len);
	}
	if(ATTR_IS_UNDERLINE(line->attr[b])) {
	    gdk_draw_line(buffer, gmo->fgc,
			  x, y + gmo->font->descent - 1,
			  x + len * gmo->charwidth, y + gmo->font->descent - 1);
	}
	x += len * gmo->charwidth;
	b = e;
    }
}


void gtk_gmo_render_string(GtkGmo *gmo, GtkGmoLine *line, GdkDrawable *buffer,
			   int begin, int end, int x, int y) {
    int b, e, len;
    b = begin;

    while(b < end) {
	for(e = b; e < end && line->attr[b] == line->attr[e]; e++) {
	    if(e < line->select_end && e >= line->select_start) {
		if(b != e)
		    break;
		while(e < end && e < line->select_end)
		    e++;
		if((len = e - b)) {
		    gtk_gmo_render_string_selected(gmo, line, buffer, b, e,x,y);
		    x += len * gmo->charwidth;
		    b = e--;
		}
	    }
	}
	len = e - b;
	if(!len) /* Let's not render empty strings ;) */
	    continue;

	if(ATTR_IS_BLINK(line->attr[b])) {
	    gmo->blinking = TRUE;
	}

	if(ATTR_IS_BG(line->attr[b])) {
	    gdk_gc_set_foreground(gmo->fgc,
				  &gmo->colors[ATTR_GET_BG(line->attr[b])]);
	    gdk_draw_rectangle(buffer, gmo->fgc, TRUE,
			       x, y - gmo->font->ascent,
			       len * gmo->charwidth, gmo->charheight);
	}

	if(!ATTR_IS_BLINK(line->attr[b]) || gmo->blink_on) {
	    if(ATTR_IS_FG(line->attr[b])) {
		if(ATTR_IS_BOLD(line->attr[b])) {
		    gdk_gc_set_foreground(gmo->fgc,
					  &gmo->colors[
					  ATTR_GET_FG(line->attr[b]) +
					  BRIGHT]);
		} else {

		    gdk_gc_set_foreground(gmo->fgc,
					  &gmo->colors[
					  ATTR_GET_FG(line->attr[b])]);
		}
	    } else {
		gdk_gc_set_foreground(gmo->fgc,
				      &gmo->colors[FG_COLOR]);
	    }

	    gdk_draw_text(buffer, gmo->font, gmo->fgc, x, y, line->line+b, len);
	    if(ATTR_IS_BOLD(line->attr[b])) {
		gdk_draw_text(buffer, gmo->font, gmo->fgc,
			      x + 1, y, line->line + b, len);
	    }
	    if(ATTR_IS_UNDERLINE(line->attr[b])) {
		gdk_draw_line(buffer, gmo->fgc,
			      x, y + gmo->font->descent - 1,
			      x + len * gmo->charwidth,
			      y + gmo->font->descent - 1);
	    }
	}
	x += len * gmo->charwidth;
	b = e;
    }
}

void gtk_gmo_line_get_line(GtkGmo *gmo, GtkGmoLine *line,
			   int n, int *begin, int *end) {
    char *tmp;
    int current_line = -1;
    *begin = 0;

    while(current_line < n) {
	current_line++;
	if(line->length - *begin <= gmo->width) {
	    *end = line->length;
	    return; /* the last line */
	}
	tmp = find_eol(line->line + *begin, gmo->width);
	*end = tmp - line->line;
	if(current_line == n)
	    return;
	if(line->line[*end] == ' ')
	    *begin = *end + 1;
	else
	    *begin = *end;
    }
}

void gtk_gmo_render_margin(GtkGmo *gmo, GtkGmoLine *line, GdkDrawable *buffer,
			   int y, int sub_line) {
    struct tm *tm;
    char str[TIMESTAMP_BUFFER + 1];
    int len;

    /* Only print the timestamp on the 1st subline */
    if(sub_line == 0 && gmo->timestamp) {
	tm = localtime(&line->timestamp);
	len = strftime(str, TIMESTAMP_BUFFER, gmo->timestamp, tm);

	gdk_gc_set_foreground(gmo->fgc, &gmo->colors[FG_COLOR]);
	gdk_draw_text(buffer, gmo->font, gmo->fgc, MARGIN, y, str, len);
    }
}

void gtk_gmo_render_line(GtkGmo *gmo, GtkGmoLine *line, GdkDrawable *buffer,
			 int *y, int lines_to_skip, int total_lines_to_print) {
    char *tmp;
    int current_line = 0;
    int begin = 0;
    int end = -1;


    while(end < line->length) {
	if(line->length - begin <= gmo->width) { /* the remainder of the line */
	    end = line->length;
	} else { /* Get the next chunk */
	    tmp = find_eol(line->line + begin, gmo->width);
	    end = tmp - line->line;
	}
	/* render the current chunk */
	if(current_line >= lines_to_skip) {
	    gtk_gmo_render_margin(gmo, line, buffer, *y, current_line);
	    gtk_gmo_render_string(gmo, line, buffer, begin, end,
				  gmo->margin_width, *y);
	    *y += gmo->charheight;
	}

	/* and reset for the next one */
	if(line->line[end] == ' ')
	    begin = end + 1;
	else
	    begin = end;
	current_line++;
	if(current_line >= total_lines_to_print + lines_to_skip)
	    break;
    }
}

int gtk_gmo_blink(GtkGmo *gmo) {
    if(debug) printf("Blinking...\n");
    if(gmo->blinking) {
	gmo->blink_on = !gmo->blink_on;
	gtk_gmo_render(gmo);
	return TRUE;
    }
    gmo->blink_on = 1;
    gmo->blink_id = -1;
    if(debug) printf("  NOT !\n");
    return FALSE;
}

/* Forces reload of the background */
int gtk_gmo_do_update_bg(GtkGmo *gmo) {
    gmo->trans_refresh_id = -1;
    if(gmo->background) {
	gdk_pixmap_unref(gmo->background);
	gmo->background = NULL;
    }
    gtk_gmo_render(gmo);
    return FALSE;
}

void gtk_gmo_refresh_background(GtkGmo *gmo) {
    if(gmo->transparent) {
	if(gmo->trans_refresh_id != -1) {
	    /* An update is scheduled, so we must be dragging the window or so,
	     * let's not do it now, but in a little while.
	     */
	    if(debug) printf("\tUpdate cancelled...");
	    gtk_timeout_remove(gmo->trans_refresh_id);
	    gmo->trans_refresh_id =
		gtk_timeout_add(500, (GtkFunction)gtk_gmo_do_update_bg, gmo);
	    if(debug) printf("\tNew update in 500 ms.\n");
	} else {
	    /* Do the update now, but wait a tincy bit, so see if more updates
	     * follow.
	     */
	    gmo->trans_refresh_id =
		gtk_timeout_add(10, (GtkFunction)gtk_gmo_do_update_bg, gmo);
	}
    }
}

#ifdef IMLIB
void tint_image(GtkGmo *gmo, GdkImlibImage *im) {
    GdkImlibColorModifier mod;
    mod.gamma = 255;
    mod.contrast = 255;

    mod.brightness = gmo->tint_r;
    gdk_imlib_set_image_red_modifier(im, &mod);

    mod.brightness = gmo->tint_g;
    gdk_imlib_set_image_green_modifier(im, &mod);

    mod.brightness = gmo->tint_b;
    gdk_imlib_set_image_blue_modifier(im, &mod);

    gdk_imlib_apply_modifiers_to_rgb(im);
}

void build_background_imlib(GtkGmo *gmo) {
    GtkWidget *w = GTK_WIDGET(gmo);
    GdkImlibImage *im;
    GdkPixmap *root;
    GdkAtom type;
    guchar *data;
    int height, width, depth, tile_width, tile_height, x, y;

    gdk_window_get_geometry(w->window, NULL, NULL,
			    &width, &height, &depth);

    if(gmo->transparent) {
	if(debug) printf("\t[IMLIB] Build background, transparent\n");
	gdk_window_get_deskrelative_origin(w->window, &x, &y);
	if (gdk_property_get(GDK_ROOT_PARENT(),
			     gdk_atom_intern("_XROOTPMAP_ID", TRUE),
			     0, 0, 10, FALSE,
			     &type, NULL, NULL, &data)) {
	    if (type == GDK_TARGET_PIXMAP) {
		root = gdk_pixmap_foreign_new(*((Pixmap *)data));
		im = gdk_imlib_create_image_from_drawable(root, NULL,
							  x, y, width, height);
		if(!im)
		    return;
		if(gmo->tinted)
		    tint_image(gmo, im);
		gdk_imlib_render(im, width, height);
		gmo->background = gdk_imlib_copy_image(im);
		gdk_imlib_destroy_image(im);
	    } else if(debug) {
		g_warning("Root pixmap is not a pixmap ???");
	    }
	    g_free(data);
	} else {
	    g_warning("Cannot open root window :(");
	}
    } else {
	if(debug) printf("\t[IMLIB] Building background from: %s\n",
			 gmo->bgfilename);
	im = gdk_imlib_load_image(gmo->bgfilename);
	if(gmo->tinted) { /* Tint the background */
	    tint_image(gmo, im);
	}
	if(!gmo->tiled) { /* Stretch it */
	    gdk_imlib_render(im, width, height);
	    gmo->background = gdk_imlib_copy_image(im);
	} else { /* Tile it ... */
	    gmo->background = gdk_pixmap_new(w->window, width, height, depth);
	    tile_width  = im->rgb_width;
	    tile_height = im->rgb_height;
	    for(x = 0; x < width; x += tile_width) {
		for(y = 0; y < height; y += tile_height) {
		    gdk_imlib_paste_image(im, gmo->background,
					  x, y, tile_width, tile_height);

		}
	    }
	}
	gdk_imlib_destroy_image(im);
    }
}
#endif

#ifdef PIXBUF
/* Build the background from a file, useing GdkPixbuf */
void build_background_pixbuf(GtkGmo *gmo) {
    GtkWidget *w = GTK_WIDGET(gmo);
    GdkPixbuf *pixbuf, *scaled;
    GdkPixmap *tile, *root;
    GdkAtom type;
    guchar *data;
    int height, width, depth, tile_width, tile_height, x, y;
    
    gdk_window_get_geometry(w->window, NULL, NULL,
			    &width, &height, &depth);

    if(gmo->transparent) {
	if(debug) printf("\t[PIXBUF] Build background, transparent\n");
	gdk_window_get_deskrelative_origin(w->window, &x, &y);
	if (gdk_property_get(GDK_ROOT_PARENT(),
			     gdk_atom_intern("_XROOTPMAP_ID", TRUE),
			     0, 0, 10, FALSE,
			     &type, NULL, NULL, &data)) {
	    if (type == GDK_TARGET_PIXMAP) {
		root = gdk_pixmap_foreign_new(*((Pixmap *)data));
		gmo->background = gdk_pixmap_new(w->window, width, height, depth);
		gdk_draw_pixmap(gmo->background, gmo->bgc, root,
				x,y,0,0,width,height);
	    } else if(debug) {
		g_warning("Root pixmap is not a pixmap ???");
	    }
	    g_free(data);
	} else {
	    g_warning("Cannot open root window :(");
	}
    } else {
	if(debug) printf("\t[PIXBUF] Building background from: %s\n",
			 gmo->bgfilename);
	pixbuf = gdk_pixbuf_new_from_file(gmo->bgfilename);
	gmo->background = gdk_pixmap_new(w->window, width, height, depth);
	if(!gmo->tiled) { /* scale it */
	    scaled = gdk_pixbuf_scale_simple(pixbuf, width, height,
					     GDK_INTERP_NEAREST);
	    gdk_pixbuf_render_to_drawable(scaled,
					  gmo->background, gmo->bgc,
					  0, 0, 0, 0,
					  width, height,
					  GDK_RGB_DITHER_NONE,
					  0, 0);
	    gdk_pixbuf_finalize(scaled);
	} else { /* tile it ! */
	    tile_width = gdk_pixbuf_get_width(pixbuf);
	    tile_height = gdk_pixbuf_get_height(pixbuf);
	    tile = gdk_pixmap_new(w->window, tile_width, tile_height, depth);
	    gdk_pixbuf_render_to_drawable(pixbuf, tile, gmo->bgc, 0,0,0,0,
					  tile_width, tile_height,
					  GDK_RGB_DITHER_NONE,
					  0, 0);
	    /* Let's tile the stuff */
	    for(x = 0; x < width; x += tile_width) {
		for(y = 0; y < height; y += tile_height) {
		    gdk_draw_pixmap(gmo->background, gmo->bgc, tile, 0, 0, x, y,
				    tile_width, tile_height);
		}
	    }
	    gdk_pixmap_unref(tile);
	}

	if(debug) printf("\tUnrefiing GdkPixbufs\n");
	gdk_pixbuf_finalize(pixbuf);
	if(debug) printf("\tAll done.\n");
    }
}
#endif

/* load the background if possible */
void gtk_gmo_update_background(GtkGmo *gmo) {
    GtkWidget *w = GTK_WIDGET(gmo);

    gmo->trans_refresh_id = -1;

    if(gmo->background) /* No need to do this already */
	return;

    if(GTK_WIDGET_REALIZED(w)) {
	if(gmo->transparent || gmo->bgfilename) {
#if defined(IMLIB) && defined(PIXBUF)
	    if(gmo->tinted)  /* we only have tinting with imlib :( */
		build_background_imlib(gmo);
	    else
		build_background_pixbuf(gmo);
#elif defined(PIXBUF)
	    build_background_pixbuf(gmo);
#elif defined(IMLIB)
	    build_background_imlib(gmo);
#endif
	}
    }
}

/* Draws the background in the buffer */
void gtk_gmo_render_background(GtkGmo *gmo, GdkDrawable *buffer) {
    GtkWidget *w = GTK_WIDGET(gmo);
    gtk_gmo_update_background(gmo);
    if(gmo->background) {
	gdk_draw_pixmap(buffer, gmo->bgc, gmo->background,
			0,0,0,0,w->allocation.width,w->allocation.height);
    } else {
	gdk_draw_rectangle(buffer, gmo->bgc, TRUE, 0, 0,
			   w->allocation.width, w->allocation.height);
    }
}

/* Sets a new (type) of background */
void gtk_gmo_set_background(GtkGmo *gmo, const char *bgfilename,
			    int tiled,
			    int transparent,
			    int tinted,
			    guchar tint_r, guchar tint_g, guchar tint_b) {
    if(debug) printf("GtkGmo::set_background(file: %s, transparent: %s, "
		     "tinted: %s (%d %d %d)\n",
		     bgfilename == NULL ? "(null)" : bgfilename,
		     transparent ? "YES" : "NO",
		     tinted ? "YES" : "NO",
		     tint_r, tint_g, tint_b);
    if(gmo->background) { /* free old background, if any */
	gdk_pixmap_unref(gmo->background);
    }
    g_free(gmo->bgfilename);

    gmo->transparent = FALSE;
    gmo->bgfilename = NULL;
    gmo->background = NULL;
    gmo->tiled = tiled;
    gmo->tinted = tinted;
    gmo->tint_r = tint_r;
    gmo->tint_g = tint_g;
    gmo->tint_b = tint_b;

    if(transparent) { /* Load transparency */
	if(debug) printf("\tSet to transparent\n");
	gmo->transparent = TRUE;
    } else if(bgfilename) { /* Load a bg pixmap */
	if(debug) printf("\tSet to bg pixmap\n");
	gmo->bgfilename = g_strdup(bgfilename);
    }
    else if(debug) printf("\tSet to normal bgcolor\n");
    if(gmo->buffer) { /* ok, lets' make sure the next update uses this pix */
	gdk_pixmap_unref(gmo->buffer);
	gmo->buffer = NULL;
    }
}

static void gtk_gmo_render(GtkGmo *gmo) {
    GtkWidget *w = GTK_WIDGET(gmo);
    GtkGmoLine *line = NULL;
    GList *l;
    int lines_to_print = gmo->adj->page_size;
    int lines_to_skip = gmo->adj->value;
    int y = gmo->font->ascent;
    int width, height, depth;

    if(debug) printf("GtkGmo::render()\n");
    if(GTK_WIDGET_REALIZED(w)) {
	gmo->blinking = FALSE;
	gdk_window_get_geometry(w->window, NULL, NULL,
				&width, &height, &depth);

	if(!gmo->buffer) 
	    gmo->buffer = gdk_pixmap_new(w->window, width, height, depth);
	gtk_gmo_render_background(gmo, gmo->buffer);
	if(gmo->num_lines) {
	    for(l = gmo->first_line; l; l = g_list_next(l)) {
		line = l->data;
		lines_to_skip -= line->used_lines;
		if(lines_to_skip < 1)
		    break;
	    }
	    if(lines_to_skip) {
		lines_to_skip = line->used_lines + lines_to_skip;
	    } else {
		l = g_list_next(l);
	    }
	    while(lines_to_print > 0 && l) {
		line = l->data;
		gtk_gmo_render_line(gmo, line, gmo->buffer, &y, lines_to_skip,
				    lines_to_print);
		lines_to_print -= line->used_lines - lines_to_skip;
		lines_to_skip = 0;
		l = g_list_next(l);
	    }
	}
	gdk_draw_pixmap(w->window, gmo->fgc, gmo->buffer,
			0,0,0,0,width,height);

	if(gmo->blinking && gmo->blink_id == -1)
	    gmo->blink_id = gtk_timeout_add(BLINK_TIMEOUT,
					    (GtkFunction) gtk_gmo_blink, gmo);
    } else if (debug) {
	printf("  Widget isn't realised (anymore/yet)\n");
    }
    gmo->last_value = (int) gmo->adj->value;
}

int gtk_gmo_force_adj_changed(GtkGmo *gmo) {
    if(debug) printf("*** Forcing scheduled redraw!\n");

    gtk_adjustment_changed(gmo->adj);
    gmo->redraw_id = -1;

    return FALSE;
}

void gtk_gmo_append_line(GtkGmo *gmo, const char *text, int len) {
    int scroll_down = FALSE;

    if(debug) printf("GtkGmo::append_line(\"%s\", %d)\n", text, len);
    if(!text) return;
    if(len < 0) len = strlen(text);

    if(gmo->scroll_on_text || (int) gmo->adj->value >=
       (int) gmo->adj->upper - (int) gmo->adj->page_size) {
	scroll_down = TRUE;
    }

    gmo->last_line = g_list_append(gmo->last_line,
				   gtk_gmo_line_new(gmo, text, len));
    gmo->last_line = g_list_last(gmo->last_line);
    if(!gmo->first_line) gmo->first_line = gmo->last_line;

    gmo->adj->upper += ((GtkGmoLine *) gmo->last_line->data)->used_lines;

    if(gmo->max_lines > 1 && gmo->num_lines > gmo->max_lines) {
	gmo->adj->upper -= ((GtkGmoLine *) gmo->first_line->data)->used_lines;
	gtk_gmo_line_free(gmo->first_line->data);
	gmo->first_line = g_list_remove_link(gmo->first_line, gmo->first_line);
    } else {
	gmo->num_lines++;
    }

    gmo->adj->page_size = MIN(gmo->height, gmo->adj->upper);
    gmo->adj->page_increment = gmo->adj->page_size;

    if(scroll_down) {
	gmo->adj->value = gmo->adj->upper - gmo->adj->page_size;
    }
    if(gmo->redraw_id == -1)
	gmo->redraw_id = gtk_timeout_add(0,
					 (GtkFunction) gtk_gmo_force_adj_changed,
					 gmo);
    else if(debug)
	printf("  Not redrawing - redraw already scheduled\n");
}

void gtk_gmo_clear(GtkGmo *gmo) {
    if(debug) printf("GtkGmo::clear()\n");

    g_list_foreach(gmo->first_line, (GFunc) gtk_gmo_line_free, NULL);
    g_list_free(gmo->first_line);
    gmo->first_line = gmo->last_line = NULL;
    gmo->num_lines = 0;

    gmo->select_start_x = gmo->select_start_y = -1;
    gmo->select_end_x = gmo->select_end_y = -1;

    gmo->adj->upper = 0;
    gmo->adj->page_size = 0;
    gtk_adjustment_changed(gmo->adj);
}

void gtk_gmo_set_timestamp(GtkGmo *gmo, const char *format) {
    char buffer[TIMESTAMP_BUFFER + 1];
    int len;
    time_t t = time(NULL);
    struct tm *tm = localtime(&t);
    int scroll_down;

    if(debug) printf("GtkGMo::set_timestamp(%s%s%s)\n",
		     format ? "\"" : "",
		     format ? format : "NULL",
		     format ? "\"" : "");

    gmo->margin_width = MARGIN;
    if(format) {
	g_free(gmo->timestamp);
	len = strftime(buffer, TIMESTAMP_BUFFER, format, tm);
	if(len > 0) {
	    gmo->timestamp = g_strdup(format);
	    gmo->margin_width += len * gmo->charwidth;
	    gmo->margin_width += MARGIN_SPACING;
	} else {
	    gmo->timestamp = NULL;
	}
    }
    if(GTK_WIDGET_REALIZED(GTK_WIDGET(gmo))) {
	gmo->width  = (GTK_WIDGET(gmo)->allocation.width -
		       (gmo->margin_width + 2 * MARGIN)) / gmo->charwidth;

	if(gmo->adj->value >= gmo->adj->upper - gmo->adj->page_size)
	    scroll_down = TRUE;
	else
	    scroll_down = FALSE;

	gtk_gmo_calc_lines(gmo);

	if(scroll_down ||
	   gmo->adj->value >= gmo->adj->upper - gmo->adj->page_size) {
	    gmo->adj->value = MAX((int) gmo->adj->upper - gmo->height, 0);
	}
	gtk_adjustment_changed(gmo->adj);
    }
}

void gtk_gmo_set_beep(GtkGmo *gmo, int beep) {
    gmo->beep = beep;
}

void gtk_gmo_set_scroll_on_text(GtkGmo *gmo, int scroll) {
    gmo->scroll_on_text = scroll;
}

void gtk_gmo_set_max_lines(GtkGmo *gmo, int max_lines) {
    int to_remove;
    gmo->max_lines = MAX(0, max_lines);

    if(gmo->max_lines < 2)
	return;

    to_remove = gmo->num_lines - gmo->max_lines;
    while(to_remove > 0) {
	gmo->adj->upper -= ((GtkGmoLine *) gmo->first_line->data)->used_lines;
	gtk_gmo_line_free(gmo->first_line->data);
	gmo->first_line = g_list_remove_link(gmo->first_line, gmo->first_line);
	to_remove--;
	gmo->num_lines--;
    }
    gtk_adjustment_changed(gmo->adj);
}
