/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <mntent.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-connection-funcs.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-main-win-funcs.h"
#include "gnome-cmd-file-selector-funcs.h"
#include "gnome-cmd-data.h"
#include "gnome-cmd-ftp-server-funcs.h"
#include "utils.h"


static gint g_id = 0;
GList *con_list = NULL;
GnomeCmdConnection *con_home = NULL;
GnomeCmdConnection *con_root = NULL;


static gboolean is_mounted (GnomeCmdDevice *device)
{
	gchar *s;
	FILE *fd;
	gchar tmp[256];
	const gchar *dev = gnome_cmd_device_get_device_fn (device);
	const gchar *dir = gnome_cmd_device_get_mountp (device);

	fd = fopen ("/etc/mtab", "r");
	if (!fd) return FALSE;

	while ((s = fgets (tmp, 256, fd)))
	{
		char **v = g_strsplit (s, " ", 3);
		if ((v[0] && strcmp (v[0], dev) == 0)
			&& (v[1] && strcmp (v[1], dir) == 0))
			return TRUE;
	}

	return FALSE;
}


static void
do_mount (GnomeCmdDevice *device)
{
	gint i;
	const gchar *dev = gnome_cmd_device_get_device_fn (device);
	const gchar *dir = gnome_cmd_device_get_mountp (device);
	gchar *cmd = g_strdup_printf ("mount %s", dir);
	DEBUG ('m', _("Mounting %s in %s\n"), dev, dir);
	i = system (cmd);
	DEBUG ('m', "mount returned %d\n", i);
}


static void
do_umount (GnomeCmdDevice *device)
{
	gint i;
	const gchar *dev = gnome_cmd_device_get_device_fn (device);
	const gchar *dir = gnome_cmd_device_get_mountp (device);
	gchar *cmd = g_strdup_printf ("umount %s", dir);
	DEBUG ('m', _("Unmounting %s from %s\n"), dev, dir);
	i = system (cmd);
	DEBUG ('m', "umount returned %d\n", i);
}


static void
update_connections (void)
{
	GnomeCmdFileSelector *fs, *fs2;
	
	fs = gnome_cmd_main_win_get_active_fs (main_win);
	fs2 = gnome_cmd_main_win_get_inactive_fs (main_win);

	gnome_cmd_file_selector_update_connections (fs);
	gnome_cmd_file_selector_update_connections (fs2);
}

static void
add_connection (GnomeCmdConnection *connection)
{	
	g_return_if_fail (connection != NULL);

	connection->id = g_id++;
	con_list = g_list_append (con_list, connection);
}


static void
remove_connection (GnomeCmdConnection *connection)
{
	g_return_if_fail (connection != NULL);
	g_return_if_fail (con_list != NULL);
	
	con_list = g_list_remove (con_list, connection);
}


GList*
gnome_cmd_connection_get_list (void)
{
	return con_list;
}


static GnomeCmdConnection *
gnome_cmd_connection_new ()
{
	GnomeCmdConnection *con = g_new (GnomeCmdConnection, 1);

	con->type = -1;
	con->cwd = NULL;
	con->default_dir = NULL;
	con->base_uri = NULL;
	con->alias = NULL;
	con->host = NULL;
	con->desc = NULL;
	con->pixmap = NULL;
	con->tooltip = NULL;
	con->server = NULL;

	con->con_drop_tooltip = NULL;
	con->con_drop_text = NULL;
	con->device = NULL;
	con->dir_history = NULL;

	return con;
}


static void
do_free_connection (GnomeCmdConnection *con)
{
	g_return_if_fail (con != NULL);
	
	if (con->cwd)
		gnome_cmd_dir_unref (con->cwd);
	if (con->default_dir)
		gnome_cmd_dir_unref (con->default_dir);
	if (con->base_uri)
		gnome_vfs_uri_unref (con->base_uri);
	
	if (con->alias)
		g_free (con->alias);
	if (con->host)
		g_free (con->host);
	if (con->desc)
		g_free (con->desc);
	if (con->tooltip)
		g_free (con->tooltip);

	if (con->con_drop_tooltip)
		g_free (con->con_drop_tooltip);
	if (con->con_drop_text)
		g_free (con->con_drop_text);
//	if (con->dir_history)
//		history_free (con_home->dir_history);
	
	g_free (con);
}


static void
add_device (GnomeCmdDevice *dev)
{
	GnomeCmdConnection *con = gnome_cmd_connection_new ();

	con->type = CON_TYPE_DEV;
	con->alias = g_strdup (gnome_cmd_device_get_alias (dev));
	con->base_uri = gnome_vfs_uri_new ("file://");
	con->cwd = gnome_cmd_dir_new_with_values (
		con->base_uri, gnome_cmd_device_get_mountp (dev), TRUE);
	gnome_cmd_dir_ref (con->cwd);
	con->default_dir = con->cwd;
	gnome_cmd_dir_ref (con->cwd);
	con->desc = g_strdup (gnome_cmd_device_get_alias (dev));
	con->device = dev;
	con->remember_dir = FALSE;
	con->tooltip = g_strdup_printf (
		"%s: [%s] (%s)",
		gnome_cmd_device_get_alias (con->device),
		gnome_cmd_device_get_mountp (con->device),
		gnome_cmd_device_get_device_fn (con->device));
	con->con_drop_tooltip = g_strdup_printf (
		"Unmount %s: [%s] (%s)",
		gnome_cmd_device_get_alias (con->device),
		gnome_cmd_device_get_mountp (con->device),
		gnome_cmd_device_get_device_fn (con->device));
	con->con_drop_text = g_strdup ("Unmount");

	add_connection (con);
}


void
gnome_cmd_connection_create_devices (void)
{
	GList *devices;

	devices = gnome_cmd_data_get_devices ();
	while (devices) {
		GnomeCmdDevice *dev = (GnomeCmdDevice*)devices->data;
		add_device (dev);
		devices = devices->next;
	}
}


void
gnome_cmd_connection_device_added (GnomeCmdDevice *dev)
{
	add_device (dev);
}


void
gnome_cmd_connection_device_removed (GnomeCmdDevice *dev)
{
	GList *tmp = con_list;
	while (tmp) {
		GnomeCmdConnection *con = (GnomeCmdConnection*)tmp->data;
		if (con->type == CON_TYPE_DEV && con->device == dev) {
			remove_connection (con);
			do_free_connection (con);
			return;
		}
		tmp = tmp->next;
	}
}


void
gnome_cmd_connection_create_local (void)
{
	con_home = gnome_cmd_connection_new ();
	
	con_home->type = CON_TYPE_LOCAL;
	con_home->alias = g_strdup (_("home"));
	con_home->base_uri = gnome_vfs_uri_new ("file://");
	con_home->cwd = gnome_cmd_dir_new_with_values (
		con_home->base_uri, g_get_home_dir (), TRUE);
	gnome_cmd_dir_ref (con_home->cwd);
	con_home->default_dir = con_home->cwd;
	gnome_cmd_dir_ref (con_home->cwd);
	con_home->desc = g_strdup (_("local filesystem"));
	con_home->pixmap = IMAGE_get_gnome_cmd_pixmap (PIXMAP_HOME);
	con_home->remember_dir = FALSE;
	con_home->tooltip = g_strdup_printf ("home: [%s]", g_get_home_dir ());
	con_home->dir_history = history_new (DIR_HISTORY_SIZE);

	add_connection (con_home);
}


GnomeCmdConnection*
gnome_cmd_connection_get_root (void)
{
	return con_home;
}


GnomeCmdConnection*
gnome_cmd_connection_get_home (void)
{
	return con_home;
}


GList*
gnome_cmd_connection_get_all_connections (void)
{
	return con_list;
}


GnomeCmdConnection*
gnome_cmd_connection_new_ftp (const gchar *alias,
							  const gchar *host,
							  gshort port,
							  const gchar *user,
							  const gchar *pw_in,
							  const gchar *start_dir)
{
	GnomeCmdConnection *con;
	gchar *pw;
	
	g_return_val_if_fail (host != NULL, NULL);
	g_return_val_if_fail (user != NULL, NULL);

	con = gnome_cmd_connection_new ();
	g_return_val_if_fail (con != NULL, NULL);
	
	con->type = CON_TYPE_FTP;
	con->base_uri = gnome_vfs_uri_new ("ftp://");
	con->alias = g_strdup (alias);
	con->host = g_strdup (host);
	con->desc = g_strdup ("ftp connection");
	con->pixmap = IMAGE_get_gnome_cmd_pixmap (PIXMAP_SERVER_SMALL);
	con->tooltip = g_strdup_printf ("ftp: [%s]", con->host);
	con->con_drop_tooltip = g_strdup_printf (_("Disconnect from %s"), con->alias);
	con->con_drop_text = g_strdup (_("Disconnect"));
	con->dir_history = history_new (DIR_HISTORY_SIZE);

	gnome_vfs_uri_set_host_name (con->base_uri, host);
	gnome_vfs_uri_set_host_port (con->base_uri, port);
	gnome_vfs_uri_set_user_name (con->base_uri, user);

	if (strcmp (user, "anonymous") == 0)	
		pw = gnome_vfs_escape_string (gnome_cmd_data_get_ftp_anonymous_password ());
	else
		pw = gnome_vfs_escape_string (pw_in);

	gnome_vfs_uri_set_password (con->base_uri, pw);
	
	con->cwd = gnome_cmd_dir_new_with_values (
		gnome_cmd_connection_get_baseuri (con),
		start_dir?start_dir:"/", TRUE);
	gnome_cmd_dir_ref (con->cwd);
	con->default_dir = con->cwd;
	gnome_cmd_dir_ref (con->cwd);

	con->remember_dir = TRUE;
	
	g_free (pw);

	return con;
}


GnomeCmdConnection*
gnome_cmd_connection_new_ftp_from_server (GnomeCmdFtpServer *server, const gchar *start_dir)
{
	GnomeCmdConnection *con = gnome_cmd_connection_new_ftp (
		gnome_cmd_ftp_server_get_alias (server),
		gnome_cmd_ftp_server_get_host_name (server),
		gnome_cmd_ftp_server_get_host_port (server),
		gnome_cmd_ftp_server_get_user_name (server),
		gnome_cmd_ftp_server_get_pw (server),
		start_dir);
	
	con->server = server;

	return con;
}


void
gnome_cmd_connection_add (GnomeCmdConnection *con)
{
	g_return_if_fail (con != NULL);
	
	add_connection (con);

	if (con->type == CON_TYPE_FTP)
		update_connections ();
}


void
gnome_cmd_connection_free (GnomeCmdConnection *con)
{	
	remove_connection (con);
	update_connections ();
	do_free_connection (con);
}


void
gnome_cmd_connection_free_all (void)
{
	g_list_foreach (con_list, (GFunc)do_free_connection, NULL);
	g_list_free (con_list);
	con_list = NULL;
}


GnomeCmdDir *
gnome_cmd_connection_get_cwd (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	return con->cwd;
}


GnomeCmdDir*
gnome_cmd_connection_get_default_dir (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);
	
	return con->default_dir;
}


GnomeVFSURI *
gnome_cmd_connection_get_baseuri (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	return con->base_uri;
}


void
gnome_cmd_connection_set_cwd (GnomeCmdConnection *con,
							  GnomeCmdDir *dir)
{
	g_return_if_fail (con != NULL);
	g_return_if_fail (dir != NULL);

	if (con->cwd == dir)
		return;

	if (con->cwd != NULL)
		gnome_cmd_dir_unref (con->cwd);
	
	gnome_cmd_dir_ref (dir);
	con->cwd = dir;
	
	//gnome_cmd_dir_list_files (con->cwd, FALSE); //FIXME: Is this needed?
}


const gchar *
gnome_cmd_connection_get_alias (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	return con->alias;
}


const gchar *
gnome_cmd_connection_get_desc (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	return con->desc;
}


const gchar*
gnome_cmd_connection_get_tooltip (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);
	
	return con->tooltip;
}


GnomeCmdPixmap *
gnome_cmd_connection_get_pixmap (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	if (con->type == CON_TYPE_DEV)
		return gnome_cmd_device_get_pixmap (con->device);
	
	return con->pixmap;
}


gboolean
gnome_cmd_connection_get_remember_dir (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, FALSE);
	
	return con->remember_dir;
}



void
gnome_cmd_connection_mount (GnomeCmdConnection *con)
{
	g_return_if_fail (con != NULL);
	g_return_if_fail (con->type == CON_TYPE_DEV);
	
	if (!is_mounted (con->device))
		do_mount (con->device);
}


gboolean
gnome_cmd_connection_ismounted (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, FALSE);
	g_return_val_if_fail (con->type == CON_TYPE_DEV, FALSE);
	
	return is_mounted (con->device);
}


void
gnome_cmd_connection_umount (GnomeCmdConnection *con)
{
	g_return_if_fail (con != NULL);
	g_return_if_fail (con->type == CON_TYPE_DEV);

	if (is_mounted (con->device)) {
		/* We must destroy all references to the mount directory
		   before we umnount. Otherwise we'll just get the
		   "Device is busy error" */
		gnome_cmd_dir_unref (con->cwd);
		gnome_cmd_dir_unref (con->default_dir);

		do_umount (con->device);

		con->cwd = gnome_cmd_dir_new_with_values (
			con->base_uri, gnome_cmd_device_get_mountp (con->device), TRUE);
		gnome_cmd_dir_ref (con->cwd);
		con->default_dir = con->cwd;
		gnome_cmd_dir_ref (con->cwd);
	}
	else
		DEBUG ('m', "%s was not mounted\n",
			   gnome_cmd_device_get_mountp (con->device));
}




