/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Provide low level serial communication functions
 *
 * Author :         Andreas Schwier (ASC)
 *
 * Last modified:   21.Jan 2000
 *
 *****************************************************************************/


#include <windows.h>


#include "..\sercom.h"


#ifdef DEBUG
static void dump(char *tag, int want, unsigned char *p, int len)

{
  printf("%s(%d) got %d: ", tag, want, len);
  while (len > 0) {
    printf("%02X ", *p);
    len--;
    p++;
  }
  printf("\n");
}
#endif



void usleep(int us)

{
    Sleep(us / 1000);
}



/*
 * Change a communication line parameter
 *
 * If a parameter is set to -1, it remains unchanged
 *
 * Returns      ERR_RS232OPEN           if port is not open
 *              ERR_RS232PARAM          if invalid parameter provided
 */

int rs232Mode(HANDLE fh, long baud, char parity, int bits, int stopbits, long timeout)

{
    DCB dcb;
    COMMTIMEOUTS cto;
    int SetDCB;

    SetDCB = FALSE;
    if (!GetCommState(fh, &dcb))
        return ERR_RS232OPEN;

    if (baud != -1) {
        dcb.BaudRate = baud;
        SetDCB = TRUE;
    }

    if (bits != -1) {
        dcb.ByteSize = bits;
        dcb.fBinary = TRUE;
        dcb.fParity = FALSE;
        dcb.fOutxCtsFlow = FALSE;
        dcb.fOutxDsrFlow = FALSE;
        dcb.fDtrControl = DTR_CONTROL_DISABLE;
        dcb.fDsrSensitivity = FALSE;
        dcb.fTXContinueOnXoff = TRUE;
        dcb.fOutX = FALSE;
        dcb.fInX = FALSE;
        dcb.fErrorChar = FALSE;
        dcb.fNull = FALSE;
        dcb.fRtsControl = RTS_CONTROL_DISABLE;
        dcb.fAbortOnError = FALSE;

        SetDCB = TRUE;
    }
    
    switch (parity) {
        case 'n':
        case 'N':
            dcb.Parity = NOPARITY;
            SetDCB = TRUE;
            break;
        case 'e':
        case 'E':
            dcb.Parity = EVENPARITY;
            SetDCB = TRUE;
            break;
        case 'o':
        case 'O':
            dcb.Parity = ODDPARITY;
            SetDCB = TRUE;
            break;
    }

    switch (stopbits) {
        case 1:
            dcb.StopBits = ONESTOPBIT;
            SetDCB = TRUE;
            break;
        case 2:    
            dcb.StopBits = TWOSTOPBITS;
            SetDCB = TRUE;
            break;
    }

    if (SetDCB) {
        if (!SetCommState(fh, &dcb))
            return ERR_RS232PARAM;
    }

    if (timeout != -1) {
        GetCommTimeouts(fh, &cto);
        cto.ReadIntervalTimeout = timeout;
        cto.ReadTotalTimeoutMultiplier = 0;
        cto.ReadTotalTimeoutConstant = timeout;
        cto.WriteTotalTimeoutMultiplier = 0;
        cto.WriteTotalTimeoutConstant = 0;
        SetCommTimeouts(fh, &cto);
    }

    return 0;
    
#if 0
  int options;
  struct termios tio;

  options = TCSANOW;

  if (tcgetattr(fh, &tio) < 0)          /* Get current configuration         */
    return ERR_RS232OPEN;

  if (baud != -1) {                     /* Change baudrate if specified      */
    options = TCSAFLUSH;
    if (cfsetospeed(&tio, rs232Baud2Speed(baud)) < 0)
      return ERR_RS232PARAM;

    if (cfsetispeed(&tio, 0) < 0)       /* Both speeds are equal             */
      return ERR_RS232PARAM;
  }

  if (parity != -1) {                   /* Change parity if specified        */
    options = TCSAFLUSH;
    parity = toupper(parity);
    tio.c_cflag &= ~(PARENB|PARODD|INPCK);

    if (parity != 'N') {
        tio.c_cflag |= PARENB|INPCK;      /* Enable parity checking            */
      if (parity == 'O')
        tio.c_cflag |= PARODD;          /* Set odd parity                    */
      else if (parity != 'E')
        return ERR_RS232PARAM;
    }
  }

  if (bits != -1) {                     /* Change bits if specified          */
    options = TCSAFLUSH;
    tio.c_cflag &= ~CSIZE;

    switch(bits) {
      case 7:         tio.c_cflag |= CS7; break;
      case 8:         tio.c_cflag |= CS8; break;
      default:
        return ERR_RS232PARAM;
    }
  }

  if (stopbits != -1) {
    options = TCSAFLUSH;
    tio.c_cflag &= ~CSTOPB;
    if (stopbits != 1)
      tio.c_cflag |= CSTOPB;
  }

  if (timeout != -1) {                  /* Timeout is in ms                  */
    memset(tio.c_cc, 0, sizeof(tio.c_cc));
    tio.c_cc[VTIME] = timeout / 100 + (timeout ? 1 : 0);
  }

  tio.c_cflag |= CREAD|HUPCL|CLOCAL;
  tio.c_cflag &= ~CRTSCTS; 

  tio.c_iflag = 0;
  tio.c_oflag = 0;  
  tio.c_lflag = 0; 
        
  tio.c_cc[VMIN]  = 0;

  if (tcsetattr(fh, options, &tio) < 0) {
    return ERR_RS232PARAM;
  }
#endif
}



/*
 * Open a serial port with the parameter specified
 *
 * Returns      ERR_RS232OPEN           if port can not be opened
 *              ERR_RS232PARAM          if invalid parameter provided
 */

int rs232Open(HANDLE *fh, char *port, long baud, char parity, int bits, int stopbits, long timeout)

{
    int rc;
    HANDLE comfd;

    comfd = CreateFile(port, GENERIC_READ|GENERIC_WRITE, 0, NULL,
                       OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

    if (comfd == INVALID_HANDLE_VALUE)
        return ERR_RS232OPEN;

    rc = rs232Mode(comfd, baud, parity, bits, stopbits, timeout);

    if (rc < 0) {
        CloseHandle(comfd);
        return rc;
    }

    rs232LineControl(comfd, 0, 0);           /* Set DTR and RTS off               */
  
    rs232Flush(comfd);

    *fh = comfd;

    return 0;
}



/*
 * Control the status of the DTR and RTS line
 *
 */

int rs232LineControl(HANDLE fh, int dtr, int rts)

{
    if (dtr)
        EscapeCommFunction(fh, SETDTR);
    else
        EscapeCommFunction(fh, CLRDTR);

    if (rts)
        EscapeCommFunction(fh, SETRTS);
    else
        EscapeCommFunction(fh, CLRRTS);

    return 0;
}



/*
 * Test the status of the DSR and CTS signal
 *
 */

int rs232LineTest(HANDLE fh, int *dsr, int *cts)

{
#if 0
  unsigned int mctl;

  if (ioctl(fh, TIOCMGET, &mctl) < 0)
    return ERR_RS232OPEN;

  *dsr = mctl & TIOCM_DSR ? 1 : 0;
  *cts = mctl & TIOCM_CTS ? 1 : 0;
#endif
  return 0;
}



/*
 * Read a block of character from the communication line
 *
 * Return the number of character read
 *
 */

int rs232Read(HANDLE fh, unsigned char *buff, int len)

{
    int rm;
    unsigned char *p;
    DWORD bread;

    rm = len;
    p = buff;

    do {
        if (!ReadFile(fh, (LPVOID)p, (DWORD)rm, (LPDWORD)&bread, NULL))
            return -1;
    
        rm -= bread;
        p += bread;
    } while (bread && rm);

#ifdef DEBUG
    dump("READ", len, buff, len - rm);
#endif

    return len - rm;
}



/* 
 * Write a block of character to the communication line
 *
 * Return the number of character written
 *
 */

int rs232Write(HANDLE fh, unsigned char *buff, int len)

{
    DWORD written;

    if (!WriteFile(fh, (LPCVOID)buff, (DWORD)len, &written, NULL))
        return -1;
        
#ifdef DEBUG
    dump("WRITE", len, buff, written);
#endif

    return written;
}



/*
 * Drain all characters remaining in the outgoing data queue
 *
 */

int rs232Drain(HANDLE fh)

{
    FlushFileBuffers(fh);
    return 0;
}



/*
 * Flush all characters remaining in the incoming data queue
 *
 */

int rs232Flush(HANDLE fh)

{
    PurgeComm(fh, PURGE_TXCLEAR|PURGE_RXCLEAR);
    return 0;
}



/*
 * Close a communication channel
 *
 */

int rs232Close(HANDLE fh)

{
    CloseHandle(fh);
    return 0;
}


