#!/bin/sh
#\
exec wish "$0" "$@"

# Copyright (C) 1998, DGA - part of the Transcriber program
# distributed under the GNU General Public License (see COPYING file)

#######################################################################
# Automatic creation of menus for Tcl/Tk 8.0 (generic functions)
#
# Syntax : create_menu description_list
#
# with description = list of {name type options [arguments]}
# and
#  - name = name of menu or item
#  - type options = command {command content}
#		    radio {variable [value]}
#		    check variable
#		    cascade {sub-items list}
#  - arguments = options for item configuration
#
# At first level, one should use only "cascade"
#

proc add_menu {m liste} {
   global Menu

   if ![winfo exists $m] {
      menu $m -tearoff 0
   }
   if ![info exists Menu(uid)] {
      set Menu(uid) 0
   }
      
   foreach item $liste {
      set name [Local [lindex $item 0]]
      if {[string length $name]==0} {
	 $m add separator
	 continue
      }
      set type ""
      set option ""
      set sequence ""
      foreach {cmd arg} [lrange $item 1 end] {
	 switch -glob -- $cmd {
	    cmd -
	    com* {
	       set type "command"
	       lappend option -command $arg
	    }
	    rad* {
	       set type "radio"
	       if {[llength $arg] > 1} {
		  foreach {var val} $arg break
		  lappend option -variable $var -value $val
	       } else {
		  lappend option -variable $arg
	       }
	    }
	    ch* {
	       set type "check"
	       lappend option -variable $arg
	    }
	    cas* {
	       set type "cascade"
	       if {![string compare $name [Local "Help"]]} {
		  set new_m $m.help
	       } else {
		  set new_m $m.item[incr Menu(uid)]
	       }
	       set Menu(menu,$name) $new_m
	       lappend option -menu [add_menu $new_m $arg]
	    }
	    -bind {
	       lappend option -accelerator $arg
	       if {![regsub -- {-([^-]*)$} $arg {-Key-\1} sequence]} {
		  set sequence "Key-$arg"
	       }
	       foreach {short long} {
		  "C(trl)?-" "Control-"
		  "A-" "Alt-" 
		  "S(hft)?-" "Shift-"
	       } {
		  regsub $short $sequence $long sequence
	       }
	    }
	    -* {
	       lappend option $cmd $arg	       
	    }
	    default {error "$name : Bad menu syntax $cmd $arg"}
	 }
      }
      eval {$m add $type -label $name} $option
      if {$sequence != ""} {
	 bind . <$sequence> [list invoke_from_bind $m [$m index last]]
	 bind . <$sequence> +break
      }
   }
   return $m
}

proc create_menu {liste} {
   global Menu
   catch {unset Menu}
   catch {destroy .menu}
   . configure -menu [add_menu .menu $liste]
}

proc append_menu {menu liste} {
   global Menu
   set menu [Local $menu]
   add_menu $Menu(menu,$menu) $liste
}

proc config_menu {menu args} {
   global Menu
   set menu [Local $menu]
   if [catch {set Menu(menu,$menu)} menu_id] {
      error "Unknown menu $menu"
   }
   eval $menu_id config $args
}


proc eval_menu {menu args} {
   global Menu
   set menu [Local $menu]
   if [catch {set Menu(menu,$menu)} menu_id] {
      error "Unknown menu $menu"
   }
   eval $menu_id $args
}

proc config_entry {menu item args} {
   global Menu
   set menu [Local $menu]
   set item [Local $item]
   if [catch {set Menu(menu,$menu)} menu_id] {
      error "Unknown menu $menu"
   }
   eval [list $menu_id entryconfig $item] $args
}

proc invoke_from_bind {menu_id item_id} {
   # just to be sure menu is in the right state, call post command
   set cmd [$menu_id cget -post]
   if {$cmd != ""} {
      eval $cmd
   }
   eval $menu_id invoke $item_id
}

proc bind_menu {sequence menu item} {
   global Menu
   set menu [Local $menu]
   set item [Local $item]
   if [catch {set Menu(menu,$menu)} menu_id] {
      error "Unknown menu $menu"
   }
   if [catch {$menu_id index $item} item_id] {
      error "Unknown item $item"
   }
   bind . $sequence "invoke_from_bind $menu_id $item_id; break"
   set sequence [string trim $sequence "<>"]
   regsub "Key(Press)?-" $sequence "" sequence
   regsub "Control-" $sequence "Ctrl-" sequence
   $menu_id entryconfigure $item_id -accelerator $sequence
}

#######################################################################

# Menus for Transcriber
proc InitMenus {} {
   global v

   # Global menu
   #{"Close"               -bind "Ctrl-w"  cmd {CloseAndDestroyTrans}}
   create_menu {
      {"File" -underline 0	cascade {
	 {"New trans"		-bind "Ctrl-n"	cmd {NewTrans}}
	 {"Open trans..." 	-bind "Ctrl-o"	cmd {OpenTransFile}}
	 {"Save"		-bind "Ctrl-s"	cmd {SaveTrans}}
	 {"Save as..."				cmd {SaveTrans as}}
	 {"Export"		cascade {
	    {"Export to .typ format..."	cmd {SaveTrans as "typ"}}
	    {"Export to .stm format..."	cmd {SaveTrans as "stm"}}
	 }}
	 {""}
	 {"Revert"				cmd {RevertTrans}}
	 {""}
	 {"Informations"	cmd {CreateInfoFrame}}
	 {"Edit episode attributes..."	cmd {EditEpisode}}	
	 {"Open audio file..." 	-bind "Ctrl-a"	cmd {OpenAudioFile}}
	 {""}
	 {"Quit"		-bind "Ctrl-q"	cmd {Quit}}
      }}
      {"Edit" -underline 0	cascade {
	 {"Undo"	-bind "Ctrl-z"		cmd {Undo} -state disabled}
	 {""}
	 {"Cut"		-acc "Ctrl-x"		cmd { TextCmd Cut }}
	 {"Copy"	-acc "Ctrl-c"		cmd { TextCmd Copy }}
	 {"Paste"	-acc "Ctrl-v"		cmd { TextCmd Paste }}
	 {""}
	 {"Find/Replace"   	cmd {Find} -bind "Ctrl-f"}
	 {"Spell checking"   	cmd {SpellChecking}}
	 {"Glossary"		cmd {EditGlossary} -bind "Ctrl-k"}
	 {"Speakers"		cascade {
	    {"Find speaker"		cmd {::speaker::find}}	
	    {"Import from file..." 	cmd {::speaker::import}}
	    {"Remove unused speakers" 		cmd {::speaker::purge}}
	 }}
	 {"Topics"		cascade {
	    {"Find topic"		cmd {::topic::find}}	
	    {"Import from file..." 	cmd {::topic::import}}
	    {"Remove unused topics" 		cmd {::topic::purge}}
	 }}
	 {""}
	 {"Insert event..."		cascade {
	    {"Isolated noise"	cmd {CreateEvent "b" "noise" "instantaneous" 1}  -bind "Ctrl-d"}
	    {"Overlapping noise"	cmd {CreateEvent "b" "noise" "previous" 1}}
	    {"Pronounce"		cmd {CreateEvent "" "pronounce" "previous" 1} -bind "Alt-equal"}
	    {"Language"			cmd {CreateEvent "en" "language" "previous" 1}}
	    {"Lexical"			cmd {CreateEvent "" "lexical" "previous" 1}}
	    {"Comment"			cmd {CreateEvent "" "comment"}}
	 }}
      }}
      {"Signal" -underline 0	cascade {
	 {"Play/Pause"		cmd {PlayOrPause} -bind "Tab"}
	 {"Play segment"	cmd {PlayCurrentSegmt} -bind "Shift-Tab"}
	 {"Play around cursor"	cmd {PlayAround} -bind "Alt-space"}
	 {"Backward and play"	cmd {PlayBefore} -bind "Ctrl-Tab"}
	 {"Play with pauses"	cmd {PlayWithPauses}}
	 {""}
	 {"Go to..."		cascade {
	    {"Forward"		cmd {PlayForward +1} -bind "Alt-Right"}
	    {"Backward"		cmd {PlayForward -1} -bind "Alt-Left"}
	    {"Previous"		cmd {MoveNextSegmt -1} -bind "Alt-Up"}
	    {"Next"		cmd {MoveNextSegmt +1} -bind "Alt-Down"} 
	    {""}
	    {"Position"		cmd {EditCursor}} 
	 }}
	 {""}
	 {"Resolution"		cascade {
	    {"1 sec"	cmd {Resolution 1} -bind "Alt-1"}
	    {"10 sec"	cmd {Resolution 10} -bind "Alt-2"}
	    {"30 sec"	cmd {Resolution 30} -bind "Alt-3"}
	    {"1 mn"	cmd {Resolution 60} -bind "Alt-4"}
	    {"5 mn"	cmd {Resolution 300} -bind "Alt-5"}
	    {""}
	    {"up"	cmd {ZoomReso -1} -bind "Alt-9"}
	    {"down"	cmd {ZoomReso +1} -bind "Alt-0"}
	    {""}
	    {"View all"		cmd {ViewAll} -bind "Alt-a"}
	 }}
	 {"Zoom selection"	cmd {ZoomSelection} -state disabled -bind "Alt-z"}
	 {"Unzoom selection"	cmd {UnZoom} -state disabled  -bind "Alt-u"}
	 {""}
	 {"Control panel"	cmd {CreateGainFrame}}
      }}
      {"Segmentation" -underline 0	cascade {
	 {"Move to..."	cascade {
	    {"Next synchro"	cmd {TextNextSync +1} -acc "Down"}
	    {"Previous synchro"	cmd {TextNextSync -1} -acc "Up"}
	    {"First segment"	cmd {TextFirstSync} -acc "Ctrl-Home"}
	    {"Last segment"	cmd {TextLastSync} -acc "Ctrl-End"}
	    {""}
	    {"Next turn"	cmd {TextNextTurn +1} -acc "Ctrl-Down"}
	    {"Previous turn"	cmd {TextNextTurn -1} -acc "Ctrl-Up"}
	    {""}
	    {"Next section"	cmd {TextNextSection +1} -acc "Page Down"}
	    {"Previous section"	cmd {TextNextSection -1} -acc "Page Up"}
	 }}
	 {""}
	 {"Insert breakpoint"	cmd {InsertSegment} -bind "Return"}
	 {"Insert background"	cmd {CreateBackground}}
	 {""}
	 {"Create turn..."	cmd {ChangeSegType Turn} -bind "Ctrl-t"}
	 {"Create section..."	cmd {ChangeSegType Section} -bind "Ctrl-e"}
	 {"Edit turn attributes..."	cmd {::turn::edit}}	
	 {"Edit section attributes..."	cmd {::section::edit}}	
	 {""}
	 {"Move breakpoint"   cmd { tk_messageBox -type ok -message "Just click on the segment boundary with central button (or control-click with left button) and drag it to the new position!"}}
	 {"Delete breakpoint"   cmd { DeleteSegment } -bind "Shift-BackSpace"}
      }}
      {"Options" -underline 0	cascade {
	 {"General..."		cmd {ConfigureGeneral}}
	 {"Audio file..."	cmd {ConfigureAudioFile}}
	 {"Events"	cascade {
	    {"Events display..."	cmd {ConfigureEvents}}
	    {"Edit noise list..."	cmd {ConfEventName "noise" "Noise"}}
	    {"Edit pronounce list..."	cmd {ConfEventName "pronounce" "Pronounce"}}
	    {"Edit lexical list..."	cmd {ConfEventName "lexical" "Lexical"}}
	    {"Edit language list..."	cmd {ConfEventName "language" "Language"}}
	 }}
	 {"Display" -underline 0	cascade {
	    {"Command buttons"	check v(view,.cmd) -command {SwitchFrame .cmd  -after .edit}}
	    {"Second signal view"	check v(view,.snd2) -command {SwitchSoundFrame .snd2}}
	 }}
	 {"Fonts"	cascade {
	    {"Text"		cmd {set v(font,text)  [ChooseFont text] }}
	    {"Events"		cmd {set v(font,event) [ChooseFont event] }}
	    {"Segmentation"	cmd {set v(font,trans) [ChooseFont trans]}}
	    {"Information"	cmd {set v(font,info)  [ChooseFont info] }}
	    {"Messages"		cmd {set v(font,mesg)  [ChooseFont mesg] }}
	    {"Lists"		cmd {set v(font,list)  [ChooseFont list] }}
	    {"Axis"		cmd {set v(font,axis)  [ChooseFont axis] }}
	 }}
	 {"Colors..."		cmd {ConfigureColors}}
	 {"Bindings..."		cmd {ConfigureBindings}}
	 {""}
	 {"Save configuration"	cmd {SaveOptions}}
      }}
      {"Help" -underline 0 cascade {
	 {"About..."		cmd {ViewHelp "Index"}}
	 {""}
	 {"Presentation"	cmd {ViewHelp "Presentation"}}
	 {"Main features"	cmd {ViewHelp "Main features"}}
	 {"User guide"		cmd {ViewHelp "User guide"}}
	 {"Reference manual"	cmd {ViewHelp "Reference manual"}}
      }}
   }
   #{"Close"		-bind "Ctrl-w"	cmd {CloseAndDestroyTrans}}

   config_menu "File" -postcommand UpdateFileMenu
   config_menu "Edit" -postcommand UpdateEditMenu
   config_menu "Segmentation" -postcommand UpdateSegmentationMenu

   # Append specific menu for national shortcuts - OBSOLETE
   if {$v(lang) == "fr" && 0} {
      append_menu "Edit" {
	 {""}
	 {"Prononciation"	cascade {
	    {{[pi] inintelligible}	cmd {CreateEvent "pi"}}
	    {{[pif] inintelligible/faible}	cmd {CreateEvent "pif"}}
	 }}
	 {"Bruits de respiration"	cascade {
	    {{[r] respiration}	cmd {CreateEvent "r"} -bind "Alt-r"}
	    {""}
	    {{[i] inspiration}	cmd {CreateEvent "i"} -bind "Alt-i"}
	    {{[e] expiration}	cmd {CreateEvent "e"}}
	    {{[n] reniflement}	cmd {CreateEvent "n"}}
	    {{[pf] soufle}	cmd {CreateEvent "pf"}}
	 }}
	 {"Bruits de bouche et de gorge"	cascade {
	    {{[bb] bruit de bouche}		cmd {CreateEvent "bb"}}
	    {""}
	    {{[bg] bruit de gorge}		cmd {CreateEvent "bg"}}
	    {{[tx] toux, rclement, ternuement} cmd {CreateEvent "tx"}}
	    {{[rire] rires du locuteur}	command {CreateEvent "rire"}}
	    {{[sif] sifflement du locuteur}	cmd {CreateEvent "sif"}}
	    {{[ch] voix chuchote}		cmd {CreateEvent "ch"}}
	 }}
	 {"Bruits divers"	cascade {
	    {{[b] bruit indtermin}		cmd {CreateEvent "b"} 
	       -bind "Alt-b"}
	    {{[conv] conversations de fond}	cmd {CreateEvent "conv"}}
	    {{[pap] froissement de papiers}	cmd {CreateEvent "pap"}}
	    {{[shh] souffle lectrique}		cmd {CreateEvent "shh"}}
	    {{[mic] bruits micro}		cmd {CreateEvent "mic"}}
	 }}
      }
   }
   if {$v(debug)} {
      append_menu "Help" {
	 {""}
	 {"Debug"		cascade {
	    {"Update"		cmd {LoadModules}}
	    {"Refresh"		cmd {Refresh}}
	    {"Expert mode"	cmd {CreateDebug}}
	 }}
      }
   }
}   

#######################################################################

# post command called by menus

proc UpdateFileMenu {} {
   global v

   if [info exists v(tk,edit)] {
      if {[HasModifs]} {
	 set state normal
      } else {
	 set state disabled
      }
      foreach type {"Save" "Revert"} {
	 config_entry "File" $type -state $state
      }
      foreach type {"Save as..."} {
	 config_entry "File" $type -state normal
      }
      
   } else {
      foreach type {"Save" "Save as..." "Revert"} {
	 config_entry "File" $type -state disabled
      }
   }

}

# set state for undo/cut/copy/paste menus
proc UpdateEditMenu {} {
   global v

   if [info exists v(tk,edit)] {
      # Undo has to be the first Edit menu line
      switch [HasUndo] {
	 0 { config_entry "Edit" 0 -label [Local "Undo"] -state disabled }
	 1 { config_entry "Edit" 0 -label [Local "Undo"] -state normal }
	 2 { config_entry "Edit" 0 -label [Local "Redo"] -state normal }
      }
      if [catch {$v(tk,edit) index sel.first}] {
	 set state disabled
      } else {
	 set state normal
      }
      foreach type {"Cut" "Copy"} {
	 config_entry "Edit" $type -state $state
      }
      foreach type {"Paste" } {
	 config_entry "Edit" $type -state normal
      }
   } else {
      foreach type {"Cut" "Copy" "Paste"} {
	 config_entry "Edit" $type -state disabled
      }
      config_entry "Edit" 0 -label [Local "Undo"] -state disabled 
   }
}

proc UpdateSegmentationMenu {} {
   global v

   if {[GetSegmtNb seg0] > 0} {
      set state normal
   } else {
      set state disabled
   }
   for {set i 0} {$i <= [eval_menu "Segmentation" index end]} {incr i} {
      catch {
	 config_entry "Segmentation" $i -state $state
      }
   }
}

