
//* @package glow.gtk

#ifndef GLOW_GTK_H
#define GLOW_GTK_H

#include <list>
#include <gtk/gtk.h>
#include "glow.object.h"

typedef enum {
  GSActivate=0,
  GSPressedAt,
  GSReleasedAt,
  GSChanged,
  GSClicked,
  GSDelete,
  GSDestroy,
  GSEnter,
  GSLeave,
  GSMotion,
  GSPressed,
  GSReleased,
  GSSelectRow,
  GSSwitchPage,
  GSToggled,
  GSUnselectRow

} GlowGtkSignal;

//* Base GTK toolkit class, provides GUI event signaling for descendants
class Gtk : public Object {
 public:
  //* Default constructor.
  Gtk();

  /** Initializes the GTK engine. Equivalent to gtk_init(...).

      @param argc pointer to main's argc
      @param argv pointer to main's argv
  */
  static void Init(int *argc,char ***argv);

  /** Starts GTK main loop. Should be called after the initial widgets
      have been set. Equivalent to gtk_main().
  */
  static void Main();

  /** Runs only one iteration of GTK's main event loop. You must have called
      Main first.

      @see Gtk::Main
  */
  static void MainIteration();

  /** Exits GTK main loop and proceeds with execution after the the
      calling of Main.

      @see Gtk::Main
  */
  static void MainQuit();

  /** Returns non-zero if there are GUI events pending.

      @return non-zero if GUI events pending.
  */
  static gint EventsPending();
  
  /** Connects a given signal of a GTK Object.

      @param signal the signal to connect
   */
  void connect(GlowGtkSignal signal);

  /** Sets object to forward messages to. GUI events are delivered to the
      destination instead of to this.

      @param dest The destination for GUI events.
      
   */
  void forward(Gtk *dest);

  //* Called on Activated events.
  virtual void activated();

  //* Called on Changed events.
  virtual void changed();

  //* Called on Clicked events.
  virtual void clicked();

  //* Called on Destroyed events.
  virtual void destroyed();

  /** Called on Deleted events.

      @return TRUE of FALSE. TRUE negates the event.
   */
  virtual gint deleted();

  //* Called on Enter events.
  virtual void enter();

  //* Called on Leave events.
  virtual void leave();

  //* Called on Pressed events.
  virtual void pressed();

  //* Called on Released events.
  virtual void released();

  //* Called on PressedAt events.
  virtual void pressedAt(GdkEventButton *geb);

  //* Called on ReleasedAt events.
  virtual void releasedAt(GdkEventButton *geb);

  //* Called on motion (mouse motion) events.
  virtual void motion(GdkEventMotion *gem);

  /** Called on Row Selection events.

      @param row    the selected row
      @param column the selected column
      @param geb    event data
   */
  virtual void rowSelected(int row,int column,GdkEventButton *geb);


  /** Called on Row Unselection events.

      @param row    the selected row
      @param column the selected column
      @param geb    event data
   */
  virtual void rowUnselected(int row,int column,GdkEventButton *geb);

  //* Called on Toggled events.
  virtual void toggled();

  /** Called on Page Switched events.
      @param pagenum the new page.
   */
  virtual void pageSwitched(gint pagenum);

  /** Called on forwarded Activated events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void activated(int id);

  /** Called on forwarded Changed events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void changed(int id);

  /** Called on forwarded Clicked events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void clicked(int id);

  /** Called on forwarded Destroyed events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void destroyed(int id);


  /** Called on forwarded Deleted events.
      @param id the Object id this event refers to
      @return TRUE or FALSE. TRUE negates the event.
      @see Object::getId
  */
  virtual gint deleted(int id);


  /** Called on forwarded Enter events.
      @param id the Object id this event refers to
      @see Object::getId
  */  
  virtual void enter(int id);


  /** Called on forwarded Leave events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void leave(int id);


  /** Called on forwarded Pressed events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void pressed(int id);


  /** Called on forwarded Released events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void released(int id);

  /** Called on forwarded PressedAt events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void pressedAt(int id,GdkEventButton *geb);

  /** Called on forwarded Motion (mouse) events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void motion(int id,GdkEventMotion *gem);

  /** Called on forwarded ReleasedAt events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void releasedAt(int id,GdkEventButton *geb);

  /** Called on forwarded Row Selection events.

      @param  id     the Object id this event refers to
      @param row    the selected row
      @param column the selected column
      @param geb    event data
   */
  virtual void rowSelected(int id,int row,int column,GdkEventButton *geb);

  /** Called on forwarded Row Unselection events.

      @param  id     the Object id this event refers to
      @param row    the selected row
      @param column the selected column
      @param geb    event data
   */
  virtual void rowUnselected(int id,int row,int column,GdkEventButton *geb);

  /** Called on forwarded Toggled events.
      @param id the Object id this event refers to
      @see Object::getId
  */
  virtual void toggled(int id);

  /** Called on forwarded page switched events.
      @param id      the Object id this event refers to
      @param pagenum the new page
      @see Object::getId
  */
  virtual void pageSwitched(int id,gint pagenum);

  /** Called on forwarded toolbar clicked events.
      @param idt     the toolbar id
      @param idb     the button sequence id
  */
  virtual void clicked(int idt,int idb);

  /** Returns the GTK+ level object item.
      @return the GtkObject this object represents.
  */
  GtkObject *getMe();

  /** When TRUE the show() method of Widgets is called automatically
      upon creation. Default is TRUE.
  */
  static gboolean AutoShow;

  static GdkWindow *oh_my;
  
 protected:
  GtkObject *me;

 private:
  Gtk *recipient;

  friend void GlowGtk_callback_activate(GtkWidget *w,gpointer data);
  friend gint GlowGtk_callback_delete(GtkWidget *w,GdkEvent *e,gpointer data);
  friend void GlowGtk_callback_destroy(GtkWidget *w,gpointer data);

  friend void GlowGtk_callback_clicked(GtkButton *w,gpointer data);
  friend void GlowGtk_callback_enter(GtkButton *w,gpointer data);
  friend void GlowGtk_callback_leave(GtkButton *w,gpointer data);
  friend void GlowGtk_callback_pressed(GtkButton *w,gpointer data);
  friend void GlowGtk_callback_released(GtkButton *w,gpointer data);

  friend void GlowGtk_callback_toggled(GtkToggleButton *w,gpointer data);

  friend void GlowGtk_callback_changed(GtkEditable *w,gpointer data);

  friend void GlowGtk_callback_select_row(GtkCList *w,gint row,
					gint column,GdkEventButton *geb,
					gpointer data);
  friend void GlowGtk_callback_unselect_row(GtkCList *w,gint row,
					gint column,GdkEventButton *geb,
					gpointer data);

  friend void GlowGtk_callback_switch_page(GtkNotebook *w,
					   GtkNotebookPage *np,
					   gint pagenum,
					   gpointer data);

  friend gboolean GlowGtk_callback_pressed_at(GtkWidget *w,
					      GdkEventButton *geb,
					      gpointer data);
  friend gboolean GlowGtk_callback_released_at(GtkWidget *w,
					       GdkEventButton *geb,
					       gpointer data);
  friend gboolean GlowGtk_callback_motion(GtkWidget *w,
					  GdkEventMotion *gem,
					  gpointer data);
};

//* Abstract Widget class.
class Widget : public Gtk {
public:

  //* Default constructor.
  Widget();

  ~Widget();
  
  //* Destroys this widget.
  void destroy();
  
  /** Realizes the widget. Realizing means creating all associated
      structures (X windows, etc.) but not showing any elements,
      leaving everything prepared for the show method.
      @see Gtk::show
  */
  void realize();

  /** Enables or disables the widget for input.
      @param s TRUE means enables, FALSE disabled.
  */
  void setSensitive(gboolean s);

  //* Shows the widget.
  void show();

  //* Hides the widget.
  void hide();

  //* Brings input focus to the widget.
  void grabFocus();

  //* Add widget to the top of the input grab stack. (makes it modal)
  void grabAdd();

  //* Removes widget from the input grab stack.
  void grabRemove();

  //* Called upon widget destruction.
  virtual void destroyed();

 protected:
  void setMe(GtkObject *obj);

 private:
  gboolean been_destroyed;
};

//* Abstract Container class
class Container : public Widget {
 public:
  /** adds a widget to this container.
      @param w the widget to add
   */
  void add(Widget *w);

  /** sets the container's border width.
      @param p border width in pixels
  */
  void setBorderWidth(int p);

  virtual void be_abstract()=0;
};

//* Scrolled Window
class ScrolledWindow : public Container {
 public:
  //* Default constructor
  ScrolledWindow();

  /** sets scrollbar policy. GtkPolicyType can be one of
      GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER.
      @param h Horizontal scrollbar policy
      @param v Vertical scrollbar policy
 */
  void setPolicy(GtkPolicyType h,GtkPolicyType v);

  /** use this to add a widget that does not support adjustments 
      @param w the widget to add
  */
  void addWithViewport(Widget *w);

  void be_abstract();
};

//* Window
class Window : public Container {
 public:

  //* constructs a toplevel window with no title.
  Window();

  Window(int a,int b,int c,int d,int e) { };

  /** constructs a window with no title, kind is one of
      GTK_WINDOW_TOPLEVEL, GTK_WINDOW_DIALOG, GTK_WINDOW_POPUP

      @param kind the kind of window to create
 */
  Window(GtkWindowType kind);

  /** constructs a window, kind is one of
      GTK_WINDOW_TOPLEVEL, GTK_WINDOW_DIALOG, GTK_WINDOW_POPUP

      @param kind  the kind of window to create
      @param title the window's title
  */
  Window(GtkWindowType kind,char *title);

  /** constructs a toplevel window.

      @param title the window's title
  */
  Window(char *title);

  /** defines window default size
      @param x width in pixels
      @param y height in pixels
  */
  void setDefaultSize(int x,int y);

  /** sets the window title
      @param s new window title
  */
  void setTitle(char *s);

  /** sets window positioning. pos is one of
      GTK_WIN_POS_CENTER, GTK_WIN_POS_NONE, GTK_WIN_POS_MOUSE
      @param pos window positioning
  */
  void setPosition(GtkWindowPosition pos);

  /** sets window icon from a XPM include
      @param xpm  XPM data string
  */
  void setIcon(char **xpm);

  void be_abstract();

 private:
  void create(GtkWindowType kind);

  GtkWindow *I;
};

//* Abstract Box Container
class Box : public Container {
 public:
  //* default constructor
  Box();

  /** packs a widget in the start of the box
      @param w the widget to add
 */
  void packStart(Widget *w);

  /** packs a widget in the end of the box
      @param w the widget to add
  */
  void packEnd(Widget *w);

  /** packs a widget in the start of the box
      @param w      the widget to add
      @param expand should the child expand
      @param fill   should the child grow to fill the space
      @param gap    separation gap in pixels
  */
  void packStart(Widget *w,gboolean expand,gboolean fill,gint gap);

  /** packs a widget in the end of the box
      @param w      the widget to add
      @param expand should the child expand
      @param fill   should the child grow to fill the space
      @param gap    separation gap in pixels
  */
  void packEnd(Widget *w,gboolean expand,gboolean fill,gint gap);

  //* expand value for packStart and packEnd
  gboolean Expand;

  //* fill value for packStart and packEnd
  gboolean Fill;

  //* gap value for packStart and packEnd
  gint     Gap;
};

//* Vertical box
class VBox : public Box {
 public:
  /** constructs a new VBox
      @param homogeneous should all children be equal-sized ?
      @param spacing spacing between elements
   */
  VBox(gboolean homogeneous,int spacing);

  void be_abstract();
};

//* Horizontal box
class HBox : public Box {
 public:
  /** constructs a new HBox
      @param homogeneous should all children be equal-sized ?
      @param spacing spacing between elements
   */
  HBox(gboolean homogeneous,int spacing);
  void be_abstract();
};

/** The notebook widget allows multiple pages to be switched.
*/
class Notebook : public Container {
 public:
  //* creates a notebook with no pages.
  Notebook();

  /** appends a new page
      @param page page body
      @param tab  widget placed inside tab
 */
  void append(Widget *page,Widget *tab);

  /** prepends a new page
      @param page page body
      @param tab  widget placed inside tab
  */
  void prepend(Widget *page,Widget *tab);

  /** inserts a new page
      @param page page body
      @param tab  widget placed inside tab
      @param pos  0-based index to insert
  */
  void insert(Widget *page,Widget *tab,int pos);

  /** remove a page
      @param pos page to remove
  */
  void remove(gint pos);

  /** sets tab positioning. pos is one of
      GTK_POS_TOP, GTK_POS_LEFT, GTK_POS_BOTTOM, GTK_POS_RIGHT
      @param pos tab position
  */
  void setTabPos(GtkPositionType pos);

  /** set tab visibility
      @param v tab visibility
  */
  void setShowTabs(gboolean v);

  /** set border visibility
      @param v border visibility
  */
  void setShowBorder(gboolean v);

  /** set scrollable
      @param v scrollability
  */
  void setScrollable(gboolean v);

  /** sets equal-sized tabs
      @param v tab size homogeinity
  */
  void setHomogeneousTabs(gboolean v);

  /** sets tab vertical border
      @param b vertical border in pixels
  */
  void setTabVBorder(int b);

  /** sets tab horizontal border
      @param b horizontal border in pixels
  */
  void setTabHBorder(int b);

  /** retrieves current notebook page
      @return zero-based page number
  */
  gint getPage();

  /** switches to a given page
      @param page page to switch to
  */
  void setPage(gint page);

  void be_abstract();
};

//* XPM-based Pixmap
class Pixmap : public Widget {
 public:
  /** creates a new pixmap, requires that your program to have
      realized or shown a toplevel window.
      @param xpmd XPM data string
      @see Window
      @see Widget::show
      @see Widget::realize
 */
  Pixmap(char **xpmd);

};

class ToolPair : public Object {
 public:
  ToolPair() { };
  ToolPair(class Toolbar *t,int i,GtkWidget *it,int actv);
  int operator==(ToolPair &t);
  int operator==(const ToolPair &t);

  class Toolbar *toolbar;
  int index;
  GtkWidget *thewidget;
  int toggly;
};

/** The toolbar widget provides an easy way to pack buttons together.
    The toolbar may be horizontal or vertical. Button clicks are
    received in the <tt>clicked(int)</tt> method of the toolbar or
    in the <tt>clicked(int,int)</tt> method of the forwarded receiver.
    <p>
    Tools are assigned id numbers based on the order they are added
    to the toolbar, starting at zero.
 */
class Toolbar : public Container {
 public:
  /** constructs a new toolbar. orientation is one of
      GTK_ORIENTATION_HORIZONTAL or GTK_ORIENTATION_VERTICAL,
      style is one of GTK_TOOLBAR_ICONS, GTK_TOOLBAR_TEXT,
      GTK_TOOLBAR_BOTH. Default toolbar type is Toolbar::Push
  */
  Toolbar(GtkOrientation orientation,GtkToolbarStyle style);
  
  ~Toolbar();

  /** sets object to forward clicked messages to
      @param dest the destination object
  */
  void forward(Gtk *dest);

  /** appends a button to the toolbar.
      @param icon    the picture on the button
      @param text    the text to show
      @param tooltip tooltip text
      @return button sequence id
  */
  int append(Pixmap *icon,char *text,char *tooltip);

  /** prepends a button to the toolbar.
      @param icon    the picture on the button
      @param text    the text to show
      @param tooltip tooltip text
      @return button sequence id
  */
  int prepend(Pixmap *icon,char *text,char *tooltip);

  /** inserts a button in the toolbar.
      @param icon    the picture on the button
      @param text    the text to show
      @param tooltip tooltip text
      @param pos     position to insert button
      @return button sequence id
  */
  int insert(Pixmap *icon,char *text,char *tooltip,int pos);

  //* appends a space (gap) to the toolbar
  void appendSpace();

  //* prepends a space (gap) to the toolbar
  void prependSpace();

  /** inserts a space (gap) to the toolbar
      @param pos position to insert gap
   */
  void insertSpace(int pos);

  /** called when the user clicks a button.
      @param button the insertion sequence id of the clicked button.
  */
  virtual void clicked(int button);

  void be_abstract();

  /** sets the kind of buttons that will be added from now on
      @param type one of Toolbar::Push, Toolbar::Toggle, Toolbar::Radio
  */
  void setToolbarType(int type);

  /** gets activation state on Toggle and Radio Buttons
      @param id the button id
  */
  gint getActive(int id);

  /** sets activation state on Toggle and Radio Buttons
      @param id button id
      @param state on/off
  */
  void setActive(int id,int state);
  
  /** toggles sensitiveness of a tool button
      @param id button id
      @param state enabled/disabled
  */
  void setItemSensitive(int id,int state);

  //* added buttons are push buttons 
  static const int Push;

  //* added buttons are toggle buttons
  static const int Toggle;

  //* added buttons are radio buttons
  static const int Radio;

 private:
  Gtk *recipient;
  int nbuttons;
  friend void toolbar_callback(GtkWidget *w,gpointer data);
  int method;

  ToolPair * getById(int id);

  static list<ToolPair> binding;

};

/** Text label. Text can be changed dynamically, but not edited
    by the user.

    @see Entry
*/
class Label : public Widget {
 public:
  /** constructs a new label with text s
      @param s label text
  */
  Label(char *s);

  /** changes label text
      @param s the new text
  */
  void setText(char *s);

  /** changes label text justification. just is one of
      GTK_JUSTIFY_LEFT, GTK_JUSTIFY_RIGHT, GTK_JUSTIFY_CENTER.
      Default is to centralize text lines.
      @param just new justification
  */
  void setJustify(GtkJustification just);
};

/** Horizontal separator line. Emits no events.
    @see VSeparator
*/
class HSeparator : public Widget {
 public:
  //* constructs a new separator
  HSeparator();
};

/** Vertical separator line. Emits no events.
    @see HSeparator
*/
class VSeparator : public Widget {
 public:
  //* constructs a new separator
  VSeparator();
};

/** Push button. Emits <b>clicked</b>,<b>pressed</b> and
    <b>released</b> signals.
    @see Gtk
    @see Gtk::clicked
    @see Gtk::pressed
    @see Gtk::released
   */
class Button : public Container {
 public:
  /** constructs a button that is an empty container, use
      Container::add to add a widget to it.
      @see Container::add
  */
  Button();

  /** constructs a button with a label inside.
      @param s text in the label
  */
  Button(char *s);

  Button(int nda,int ndb);

  void be_abstract();

 protected:
  void attachSignals();
};

/** A toggle button is a push button with two states. It emits
    all events a Button emits, plus the <b>toggled</b> event.
    @see Button
    @see Gtk
    @see Gtk::toggled
*/
class ToggleButton : public Button {
 public:

  /** constructs a button that is an empty container, use
      Container::add to add a widget to it.
      @see Container::add
  */
  ToggleButton();

  /** constructs a button with a label inside.
      @param s text in the label
  */
  ToggleButton(char *s);

  ToggleButton(int nda,int ndb);

  /** toggles button state
      @param v non-zero means on
  */
  void setActive(gboolean v);

  /** retrieves button state
      @return button state
  */
  gboolean getActive();

 protected:
  void attachSignals();
};

/** two-state check button.
    From the programmer's point of view operation is the same as
    ToggleButton, just the widget appearance changes.
    @see ToggleButton
*/
class CheckButton : public ToggleButton {
 public:
  
  /** constructs a button that is an empty container, use
      Container::add to add a widget to it.
      @see Container::add
  */
  CheckButton();

  /** constructs a button with a label.
      @param s text in the label
  */
  CheckButton(char *s);

  CheckButton(int nda,int ndb);
};

/** Radio button widget. It is a <i>group checkbutton</i>, where only
    one button may be selected at a time.
    You must create a RadioButtonGroup to group Radio Buttons, or
    they won't work correctly.
    Most methods regarding Radio Buttons are in the ToggleButton class.

    @see ToggleButton
    @see RadioButtonGroup
*/
class RadioButton : public CheckButton {
 public:

  /** constructs a button that is an empty container, use
      Container::add to add a widget to it.
      @see Container::add
  */
  RadioButton();

  /** constructs a button with a label.
      @param s text in the label
  */
  RadioButton(char *s);
};

/** Represents a group of radio buttons where only one can be selected
    at a time. You should simply create one RadioButtonGroup for each
    group and <b>append</b> the RadioButtons to it after they are
    created.

    @see RadioButton
*/
class RadioButtonGroup {
 public:
  //* constructs a new group
  RadioButtonGroup();

  /** destroys a group. It is important to delete RadioButtonGroup
      objects when you are done with them to free memory.
   */
  ~RadioButtonGroup();

  /** appends a RadioButton to the group
      @param radio the RadioButton
  */
  void append(RadioButton *radio);

 private:
  GSList *group;
};

/** Abstract editable widget.

    @see Entry
    @see Text
 */
class Editable : public Widget {
 public:

  /** sets the widget enabled or disabled for editing
      @param v editable ?
  */
  void setEditable(gboolean v);

  /** sets the text insertio mark position.
      This is <b>NOT</b> the cursor position.
      @param pos zero-based character position
  */
  void setPosition(gint pos);

  /** retrieves current position
      @return current position
      @see setPosition
  */
  gint getPosition();

  //* removes currently selected text
  void deleteSelection();

  //* cuts selection to clipboard
  void cutClipboard();

  //* copies selection to clipboard
  void copyClipboard();

  //* pastes clipboard into widget
  void pasteClipboard();

  /** retrieves text from the widget.
      @param start starting position, zero-based
      @param end   ending position, -1 means EOF.
   */
  gchar *getChars(gint start,gint end);

  /** deletes a given range of text.
      @param start starting position, zero-based
      @param end   ending position, -1 means EOF.
  */
  void deleteText(gint start,gint end);

  /** selects a given range of text.
      @param start starting position, zero-based
      @param end   ending position, -1 means EOF.
  */
  void selectRegion(gint start,gint end);

  virtual void be_abstract()=0;
};

//* one line text entry widget
class Entry : public Editable {
 public:
  //* constructs an empty entry
  Entry();

  /** constructs an empty entry with a maximum
      number of characters limit.
      @param max maximum data size
   */
  Entry(int max);

  /** retrieves all text from the entry.
      @return current text
  */
  gchar *getText();

  /** sets entry text
      @param text the new text
  */
  void setText(char *text);

  /** sets <i>password</i> mode on or off. When visibility if FALSE
      characters show as asterisks, useful for password fields.
      @param v visiblity
  */
  void setVisibility(gboolean v);

  /** sets maximum data length
      @param max maximum length
  */
  void setMaxLength(int max);

  void be_abstract();
};


//* multi-line text widget.
class Text : public Editable {
 public:
  //* constructs a new empty Text widget
  Text();
  
  /** sets word wrapping on or off
      @param v wrap words ?
   */
  void setWordWrap(gboolean v);

  /** sets line wrapping on of off
      @param v wrap lines ?
  */
  void setLineWrap(gboolean v);

  /** freezes widget refresh until thaw is called, speeds up
      things when you're changing the contents and don't want
      the widget to be redrawn before you are finished.
      @see thaw
  */
  void freeze();

  /** unfreezes a frozen widget.
      @see freeze
  */
  void thaw();

  /** inserts text at insertion mark
      @param s text to be inserted
  */
  void insert(char *s);
  
  void be_abstract();
};

//* multi-column list
class CList : public Container {
public:

  /** constructs a new list
      @param columns number of columns
 */
  CList(int columns);
 
  /** sets the border shadow type. stis one of
      GTK_SHADOW_NONE, GTK_SHADOW_IN, GTK_SHADOW_OUT,
      GTK_SHADOW_ETCHED_IN, GTK_SHADOW_ETCHED_OUT.
      @param st shadow type
  */
  void setShadowType(GtkShadowType st);

  /** sets selection mode, which is one of
      GTK_SELECTION_SINGLE, GTK_SELECTION_BROWSE,
      GTK_SELECTION_MULTIPLE, GTK_SELECTION_EXTENDED
      
      @param sm selection mode
  */
  void setSelectionMode(GtkSelectionMode sm);

  /** sets list user-reorderable or not.
      @param reorderable is it ?
  */
  void setReorderable(gboolean reorderable);

  //* makes titles visible
  void columnTitlesShow();

  //* makes titles not visible
  void columnTitlesHide();

  //* allow title buttons to generate clicked events
  void columnTitlesActive();

  //* do not allow title buttons to generate clicked events
  void columnTitlesPassive();

  /** sets a column title active
      @param col columns
      @see columnTitlesActive
   */
  void columnTitleActive(int col);

  /** sets a column title passive
      @param col columns
      @see columnTitlesPassive
   */
  void columnTitlePassive(int col);

  /** sets a column's title
      @param col   the column
      @param title the title
  */
  void setColumnTitle(int col,char *title);

  /** sets all column titles from an STL list of (char *).
      The list must have enough items to set all column titles.
      @param titles list of titles
  */
  void setColumnTitles(list<char *> & titles);

  /** retrieves the title of a column
      @param col the column
      @return the column's title
  */
  char *getColumnTitle(int col);

  /** sets justification on a column.
      j is one of GTK_JUSTIFY_LEFT, GTK_JUSTIFY_RIGHT, GTK_JUSTIFY_CENTER.
      @param col the column
      @param j   justification
  */
  void setColumnJustification(int col,GtkJustification j);

  /** shows or hides a column
      @param col     the column
      @param visible is it ?
  */
  void setColumnVisibility(int col,gboolean visible);

  /** sets a column user-resizeable or not
      @param col        the column
      @param resizeable is it ?
  */
  void setColumnResizeable(int col,gboolean resizeable);

  /** sets auto resize (size to fit) on a column or not
      @param col        the column
      @param autoresize should it ?
  */
  void setColumnAutoResize(int col,gboolean autoresize);

  //* not documented
  int  columnsAutosize();

  /** sets the column width
      @param col   column
      @param width width in pixels
   */
  void setColumnWidth(int col,int width);

  /** sets the column maximum width
      @param col   column
      @param width maximum width in pixels
   */
  void setColumnMaxWidth(int col,int width);

  /** sets the column minimum width
      @param col   column
      @param width minimum width in pixels
   */
  void setColumnMinWidth(int col,int width);

  /** sets global row height
      @param height row height in pixels
  */
  void setRowHeight(int height);

  /** scrolls list to make a given cell visible. The align fields
      vary from 0.0 to 1.0. 0.0 is closer to the top/left, 1.0
      closer to right/bottom.
      @param row       the row
      @param column    the column
      @param row_align row alignment [0.0, 1.0]
      @param col_align column alignment [0.0, 1.0]
   */
  void moveTo(int row,int column,float row_align,float col_align);

  /** sets cell text
      @param row    row
      @param column column
      @param text   text
  */
  void setText(int row,int column,char *text);

  /** sets a full row of text from an STL list of (char *).
      the list must have at least as many items as the widget
      has columns.
      @param row  the row
      @param data row text
  */
  void setRowText(int row,list<char *> & data);

  /** move row
      @param src  source row index
      @param dest destination row index
  */
  void rowMove(int src,int dest);

  /** retrieves pointer to cell text
      @param row    row index
      @param column column index
      @param text   reference to pointer which will point to text on return
      @return not documented
  */
  int  getText(int row,int column,char **text);

  /** appends a row of text based on a STL list, which must have at least
      as many items as this widget has columns.
      @param data list of (char *) strings.
  */
  void append(list<char *> & data);

  /** prepends a row of text based on a STL list, which must have at least
      as many items as this widget has columns.
      @param data list of (char *) strings.
  */
  void prepend(list<char *> & data);

  /** inserts a row of text based on a STL list, which must have at least
      as many items as this widget has columns.
      @param row  row to insert
      @param data list of (char *) strings.
  */
  void insert(int row,list<char *> & data);

  /** removes a row
      @param row index of row to remove
  */
  void remove(int row);

  /** selects a row
      @param row row
      @param column column
  */
  void selectRow(int row,int column);

  /** unselects a row
      @param row row
      @param column column
  */
  void unselectRow(int row,int column);

  /** select all (when applicable)
      @see setSelectionMode
  */
  void selectAll();

  //* unselect all rows
  void unselectAll();

  /** swaps list rows
      @param row1 one row
      @param row2 another row
   */
  void swapRows(int row1,int row2);

  /** freezes widget refresh until thaw is called, speeds up
      things when you're changing the contents and don't want
      the widget to be redrawn before you are finished.
      @see thaw
  */  
  void freeze();

  /** unfreezes a frozen widget.
      @see freeze
  */
  void thaw();

  //* removes all rows from this list
  void clear();

  void be_abstract();

private:
  int ncolumns;
};

/** Handle Box
 */
class HandleBox : public Container {
 public:
  //* constructs a new HandleBox
  HandleBox();

  /** sets the shadow type. st is one o
      GTK_SHADOW_NONE, GTK_SHADOW_IN, GTK_SHADOW_OUT,
      GTK_SHADOW_ETCHED_IN, GTK_SHADOW_ETCHED_OUT.
      @param st shadow type
   */
  void setShadowType(GtkShadowType st);

  /** sets the handle position. pos is one of
      GTK_POS_TOP, GTK_POS_LEFT, GTK_POS_BOTTOM, GTK_POS_RIGHT
      @param pos handle position
  */
  void setHandlePosition(GtkPositionType pos);

  /** sets the edge to snap to. pos is one of
      GTK_POS_TOP, GTK_POS_LEFT, GTK_POS_BOTTOM, GTK_POS_RIGHT
      @param pos edge to snap to
  */
  void setSnapEdge(GtkPositionType pos);

  void be_abstract();
};

/** X11 Font
    @see Canvas
 */
class Font {
 public:
  /** creates non-bold, non-italic font.
   */
  Font(char *face,int size);

  /** creates font with given face, size, boldness and italicness */
  Font(char *face,int size,gboolean bold,gboolean italic);

  /** creates font from full X11 spec
   */
  Font(char *fullspec);

  /** deallocates font structures */
  ~Font();

  GdkFont *handle;
};

/** A Drawing canvas widget. GTK's DrawingArea on steroids.
 */
class Canvas : public Widget {
 public:
  //* creates anew 64x64 canvas
  Canvas();

  /** creates a canvas with the desired size
      @param w width
      @param h height
   */
  Canvas(int w,int h);

  //* deallocs the double buffering buffer
  ~Canvas();

  //* sets widget size (incomplete)
  void setSize(int w,int h);

  //* updates the widget content
  void update();

  /** sets current drawing color.
      @see glow.image.Triplet
  */
  void setColor(int triplet);

  /** draws a straight line from (x1,y1) to (x2,y2) with
      the current color.
      @see setColor
  */
  void drawLine(int x1,int y1,int x2,int y2);

  /** draws a rectangular border with top-left corner at (x1,y1)
   */
  void drawRect(int x1,int y1,int w,int h);

  /** fills a rectangle with top-left corner at (x1,y1)
   */
  void fillRect(int x1,int y1,int w,int h);

  /** fills a 3D-izes rectangle with top-left corner at (x1,y1)
      and 3d-ish border of width border.
  */
  void fillRect3D(int x1,int y1,int w,int h,int border,gboolean raised);

  //* inhibits widget refreshing
  void freeze();

  //* allows widget refreshing
  void thaw();

  /** sets current font.
      @see Font
   */
  void setFont(Font *font);

  /** draws string with current font
      @param str the string
      @param x   start of text coordinate
      @param y   baseline of text coordinate
      @see Font
      @see setFont
  */
  void drawString(char *str,int x,int y);

  /** sets a lonely pixel with the current color */
  void drawPoint(int x,int y);

  /** draws an elliptical arc with the current color
      @param x left edge of ellipsis hull
      @param y top edge of ellipsis hull
      @param w width of ellipsis hull
      @param h height of ellipsis hull
      @angstart starting angle, in degrees
      @angend   ending angle, in degrees
  */
  void drawArc(int x,int y,int w,int h,int angstart,int angend);

  /** fills an elliptical arc with the current color
      @param x left edge of ellipsis hull
      @param y top edge of ellipsis hull
      @param w width of ellipsis hull
      @param h height of ellipsis hull
      @angstart starting angle, in degrees
      @angend   ending angle, in degrees
  */
  void fillArc(int x,int y,int w,int h,int angstart,int angend);

  //* draws a polygon
  void drawPolygon(list<int> &x,list<int> &y);

  //* fills a polygon
  void fillPolygon(list<int> &x,list<int> &y);

  /** set to non-zero to draw ASAP on a frozen canvas. As soon as it
      unfreezes it will be restores to the data before fast drawings */
  int FastDraw;

 private:
  GdkGC *gc;
  GdkPixmap *buffer;
  Font *curfont;
  int frost;
  int curcolor;
  GdkWindow *window;

  void init(int w,int h);

  friend gboolean canvas_expose(GtkWidget *widget,
				GdkEventExpose *gee,
				gpointer data);
};

/** button values for Message Boxes */
typedef enum {
  MsgNone=   0,
  MsgOk=     1<<0,
  MsgYes=    1<<1,
  MsgNo=     1<<2,
  MsgCancel= 1<<3
} MessageButton;

/** A MessageBox window, it is poped on creation, and is application
    modal. Suggested icons can be found in the MessageIcon class.
    Result comes in the <b>Result</b> attribute.
    @see MessageIcon
    
*/
class MessageBox : public Window {
  public:
  /** creates ans pops a new MessageBox.
      @param text text to show
      @param title window title
      @param icon icon to use, can be NULL.
      @mb    buttons to show, ORed from MessageButton.
      @see   MessageButton
  */
    MessageBox(char *text,char *title,char **icon,int mb);
  
  /** contains the result the user clicked */
    MessageButton Result;  

    virtual void clicked(int id);

  private:
    int ids[60];
};

/** File Selection Dialog
    @see Window
 */
class FileSelection : public Window {
 public:
  /** constructs a new File Selection with given title.
      you must call show() to execute it

      @see show
  */
  FileSelection(char *title);

  /** sets the filename */
  void setFilename(char *filename);

  /** returns currently selected file name */
  char *getFilename();

  /** shows file operation buttons */
  void showFileOp();

  /** hides file operation buttons */
  void hideFileOp();

  /** executes dialog, returns zero if cancel clicked */
  int show();

  virtual gint deleted();

 private:
  int result;

  friend void fs_ok_callback(GtkButton *b,gpointer data);
  friend void fs_cancel_callback(GtkButton *b,gpointer data);
};

#endif


