(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Franois Pessaux, projet Cristal, INRIA Rocquencourt     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*            Jun Furuse, projet Cristal, INRIA Rocquencourt           *)
(*                                                                     *)
(*  Copyright 1999,2000,2001,2002,2001,2002                            *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)
(* RGBA 24 bit depth image format *)

type elt = Color.rgba

type t = { 
    width: int; 
    height: int; 
    mutable infos: Info.info list; 
    data: Bitmap.t
}

(* creation functions *)

val create_with : int -> int -> string -> t
    (* [create_with width height infos data] creates a rgb24 image of 
       a size [width] x [height] and a string representation of 
       the bitmap [data] *)
val create : int -> int -> t
    (* creates a non-initialized image without informations *)
val make : int -> int -> elt -> t
    (* creates an image without informations filled with elt *)

(* Methods *)

val get : t -> int -> int -> elt
val set : t -> int -> int -> elt -> unit
    (* [get image x y] and [set image x y v] reads/writes the pixel 
       information at (x,y) of [image]. If (x,y) is out of the image,
       they raise Out_of_image exception. *)

val unsafe_get : t -> int -> int -> elt
val unsafe_set : t -> int -> int -> elt -> unit
    (* [unsafe_get] and [unsafe_set] are the same functions as [get] 
       and [set], but they lack the image region checks. So it is fast.
       But you have to use them with being sure that the specified point 
       is in the image. Otherwise the result is unknown, and sometimes 
       a runtime error occurs. *)

val destroy : t -> unit
    (* You need to call [destroy t] to free the swap files maybe created
       for [t] *)

val sub : t -> int -> int -> int -> int -> t
    (* [sub dst x y width height] returns sub-bitmap of [dst],
       at (x,y)-(x+width-1,y+height-1). *)

val blit : t -> int -> int -> t -> int -> int -> int -> int -> unit
    (* [blit src sx sy dst dx dy width height] copies the rectangle
       region of [src] at (sx,sy)-(sx+width-1,sy+height-1) to [dst], at
       (dx,dy)-(dx+width-1,dy+height-1) *)

(* Tool functions *)

val resize : (float -> unit) option -> t -> int -> int -> t
    (* [resize prog src new_width new_height] resizes an original image [src]
       and creates a new image with the size [new_width]x[new_height]. 
       [prog] specifies a progressive monitor function. *)
