#!/usr/bin/env python
# Copyright 2018 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
"""code generator for webgpu command buffers."""

import os
import os.path
import sys
from optparse import OptionParser

import build_cmd_buffer_lib

# Named type info object represents a named type that is used in API call
# arguments. The named types are used in 'webgpu_cmd_buffer_functions.txt'.
#
# Options are documented in build_gles2_cmd_buffer.py/build_raster_cmd_buffer.py
_NAMED_TYPE_INFO = {
}

# A function info object specifies the type and other special data for the
# command that will be generated. A base function info object is generated by
# parsing the "webgpu_cmd_buffer_functions.txt", one for each function in the
# file. These function info objects can be augmented and their values can be
# overridden by adding an object to the table below.
#
# Must match function names specified in "webgpu_cmd_buffer_functions.txt".
#
# Options are documented in build_gles2_cmd_buffer.py/build_raster_cmd_buffer.py
# (Note: some options (like decoder_func and unit_test) currently have no
# effect, because WriteServiceImplementation and WriteServiceUnitTests are not
# used below.)
_FUNCTION_INFO = {
  'Dummy': {
    'impl_func': False,
  },
}


def main(argv):
  """This is the main function."""
  parser = OptionParser()
  parser.add_option(
      "--output-dir",
      help="base directory for resulting files, under chrome/src. default is "
      "empty. Use this if you want the result stored under gen.")
  parser.add_option(
      "-v", "--verbose", action="store_true",
      help="prints more output.")

  (options, _) = parser.parse_args(args=argv)

  # This script lives under gpu/command_buffer, cd to base directory.
  os.chdir(os.path.dirname(__file__) + "/../..")
  base_dir = os.getcwd()
  build_cmd_buffer_lib.InitializePrefix("WebGPU")
  gen = build_cmd_buffer_lib.GLGenerator(options.verbose, "2018",
                                         _FUNCTION_INFO, _NAMED_TYPE_INFO)
  gen.ParseGLH("gpu/command_buffer/webgpu_cmd_buffer_functions.txt")

  # Support generating files under gen/
  if options.output_dir != None:
    os.chdir(options.output_dir)

  os.chdir(base_dir)

  gen.WriteCommandIds("gpu/command_buffer/common/webgpu_cmd_ids_autogen.h")
  gen.WriteFormat("gpu/command_buffer/common/webgpu_cmd_format_autogen.h")
  gen.WriteFormatTest(
    "gpu/command_buffer/common/webgpu_cmd_format_test_autogen.h")
  gen.WriteGLES2InterfaceHeader(
    "gpu/command_buffer/client/webgpu_interface_autogen.h")
  gen.WriteGLES2ImplementationHeader(
    "gpu/command_buffer/client/webgpu_implementation_autogen.h")
  gen.WriteGLES2Implementation(
    "gpu/command_buffer/client/webgpu_implementation_impl_autogen.h")
  gen.WriteGLES2ImplementationUnitTests(
    "gpu/command_buffer/client/webgpu_implementation_unittest_autogen.h")
  gen.WriteCmdHelperHeader(
     "gpu/command_buffer/client/webgpu_cmd_helper_autogen.h")
  # Note: No gen.WriteServiceImplementation
  # Note: No gen.WriteServiceUnitTests
  gen.WriteServiceUtilsHeader(
    "gpu/command_buffer/service/webgpu_cmd_validation_autogen.h")
  gen.WriteServiceUtilsImplementation(
    "gpu/command_buffer/service/"
    "webgpu_cmd_validation_implementation_autogen.h")

  build_cmd_buffer_lib.Format(gen.generated_cpp_filenames)

  if gen.errors > 0:
    print "%d errors" % gen.errors
    return 1
  return 0


if __name__ == '__main__':
  sys.exit(main(sys.argv[1:]))
