/*
 * Copyright (c) 2004 Edd Dumbill
 *
 * This file is based on the Mono.Posix.Catalog distributed with the Mono
 * project.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
*/

using System;
using System.IO;
using System.Runtime.InteropServices;

namespace Cowbell.Base
{
	public class Catalog
	{
		/* public methods */
		public static void Init ()
		{
			IntPtr package = Marshal.StringToHGlobalAuto ("cowbell");
			IntPtr localedir = Marshal.StringToHGlobalAuto (Globals.LocaleDir);
			IntPtr utf8 = Marshal.StringToHGlobalAuto ("UTF-8");
			BindTextDomain (package, localedir);
			BindTextDomainCodeset (package, utf8);
			TextDomain (package);
			Marshal.FreeHGlobal (package);
			Marshal.FreeHGlobal (localedir);
			Marshal.FreeHGlobal (utf8);
		}

		public static string GetString (string str)
		{
			IntPtr singular = Marshal.StringToHGlobalAuto (str);
			string ret = Marshal.PtrToStringAuto (Gettext (singular));
			Marshal.FreeHGlobal (singular);
			return ret;
		}
		
		public static string GetPluralString (string singular, string plural, int n)
		{
			IntPtr s = Marshal.StringToHGlobalAuto (singular);
			IntPtr p = Marshal.StringToHGlobalAuto (plural);
			string ret = Marshal.PtrToStringAnsi (NGettext (s, p, n));
			Marshal.FreeHGlobal (s);
			Marshal.FreeHGlobal (p);
			return ret;
		}

		/* private methods */
		[DllImport("intl", EntryPoint="bindtextdomain")]
		private static extern IntPtr intl_bindtextdomain (IntPtr domainname, IntPtr dirname);

		[DllImport("libintl", EntryPoint="bindtextdomain")]
		private static extern IntPtr libintl_bindtextdomain (IntPtr domainname, IntPtr dirname);

		private static void BindTextDomain (IntPtr domainname, IntPtr dirname)
		{
			// XXX: Hack around the libintl -> intl change for mono 1.1.13.

			try {
				intl_bindtextdomain (domainname, dirname);
			} catch (DllNotFoundException e) {
				libintl_bindtextdomain (domainname, dirname);
			}
		}

		[DllImport("intl", EntryPoint="bind_textdomain_codeset")]
		private static extern IntPtr intl_bind_textdomain_codeset (IntPtr domainname, IntPtr codeset);

		[DllImport("libintl", EntryPoint="bind_textdomain_codeset")]
		private static extern IntPtr libintl_bind_textdomain_codeset (IntPtr domainname, IntPtr codeset);

		private static void BindTextDomainCodeset (IntPtr domainname, IntPtr codeset)
		{
			try {
				intl_bind_textdomain_codeset (domainname, codeset);
			} catch (DllNotFoundException e) {
				libintl_bind_textdomain_codeset (domainname, codeset);
			}
		}

		[DllImport("intl", EntryPoint="textdomain")]
		private static extern IntPtr intl_textdomain (IntPtr domainname);

		[DllImport("libintl", EntryPoint="textdomain")]
		private static extern IntPtr libintl_textdomain (IntPtr domainname);

		private static void TextDomain (IntPtr domainname)
		{
			try {
				intl_textdomain (domainname);
			} catch (DllNotFoundException e) {
				libintl_textdomain (domainname);
			}
		}

		[DllImport("intl", EntryPoint="gettext")]
		private static extern IntPtr intl_gettext (IntPtr instring);

		[DllImport("libintl", EntryPoint="gettext")]
		private static extern IntPtr libintl_gettext (IntPtr instring);

		private static IntPtr Gettext (IntPtr instring)
		{
			IntPtr ptr = IntPtr.Zero;

			try {
				ptr = intl_gettext (instring);
			} catch (DllNotFoundException e) {
				ptr = libintl_gettext (instring);
			}

			return ptr;
		}

		[DllImport("intl", EntryPoint="ngettext")]
		private static extern IntPtr intl_ngettext (IntPtr singular, IntPtr plural, Int32 n);

		[DllImport("libintl", EntryPoint="ngettext")]
		private static extern IntPtr libintl_ngettext (IntPtr singular, IntPtr plural, Int32 n);

		private static IntPtr NGettext (IntPtr s, IntPtr p, Int32 n)
		{
			IntPtr ptr = IntPtr.Zero;

			try {
				ptr = intl_ngettext (s, p, n);
			} catch (DllNotFoundException e) {
				ptr = libintl_ngettext (s, p, n);
			}
			
			return ptr;
		}
	}
}
