/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-global-edit.c,v 1.12 2005/02/01 02:31:58 hoa Exp $
 */

#include "etpan-cfg-global-edit.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include "etpan-cfg-storage.h"
#include "etpan-cfg-common.h"
#include <stdlib.h>
#include "etpan-thread-manager.h"
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "etpan-cfg-edit-common.h"
#include "etpan-help-viewer.h"
#include "etpan-search-input.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static int display_init(struct etpan_subapp * app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);

static struct etpan_subapp_driver etpan_cfg_global_edit_app_driver = {
  .name = "cfg-global-edit",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .get_idle_udelay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

#define INFO_TYPE_MAX 20

enum {
  INFO_TYPE_DISPLAY_CHARSET,
  INFO_TYPE_EDITOR_CHARSET,
  INFO_TYPE_MESSAGE_CHARSET,
  INFO_TYPE_EDITOR,
  INFO_TYPE_REPLY_QUOTE_LIMIT,
  INFO_TYPE_NETWORK_TIMEOUT,
  INFO_TYPE_POLL_DELAY,
  INFO_TYPE_SENDMAIL_PATH,
  INFO_TYPE_THREAD_TYPE,
};

#define TABLE_SIZE(a) (sizeof(a) / sizeof(a[0]))

struct app_state {
  char display_charset[256];
  char editor_charset[256];
  char message_charset[256];
  char editor[PATH_MAX];
  int reply_quote_limit;
  int network_timeout;
  int poll_delay;
  char sendmail_path[PATH_MAX];
  int thread_type;

  unsigned int index;
  unsigned int first;
  unsigned int count;
  
  int main_attr;
  int selection_attr;
  int status_attr;
  
  /* upcall */
  void (* upcall)(struct etpan_subapp *, int, void *);
  void * upcall_data;
};

static int info_type_tab[] = {
  INFO_TYPE_DISPLAY_CHARSET,
  INFO_TYPE_EDITOR_CHARSET,
  INFO_TYPE_MESSAGE_CHARSET,
  INFO_TYPE_EDITOR,
  INFO_TYPE_SENDMAIL_PATH,
  INFO_TYPE_REPLY_QUOTE_LIMIT,
  INFO_TYPE_NETWORK_TIMEOUT,
  INFO_TYPE_POLL_DELAY,
  INFO_TYPE_THREAD_TYPE,
};

static int show_help(struct etpan_subapp * app);

static void ask_quit(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  int current_type;
  unsigned int list_lines;

  state = app->data;
  
  list_lines = app->display_height - 1;
  
  current_type = info_type_tab[state->index];
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case KEY_UP:
    if (state->index > 0)
      state->index --;
    break;
    
  case KEY_DOWN:
    if (state->index < state->count - 1)
      state->index ++;
    break;
    
  case KEY_NPAGE:
    if (state->index + list_lines - 1 < state->count)
      state->index += list_lines - 1;
    else
      state->index = state->count - 1;
    break;
    
  case KEY_PPAGE:
    if (state->index >= list_lines - 1)
      state->index -= list_lines - 1;
    else
      state->index = 0;
    break;
    
  case KEY_HOME:
    state->index = 0;
    break;
    
  case KEY_END:
    if (state->count > 0)
      state->index = state->count - 1;
    break;
    
  case KEY_CTRL('G'):
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_CFG_GLOBAL_EDIT_CANCEL, state->upcall_data);
    break;
    
  case 'y':
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_CFG_GLOBAL_EDIT_VALID, state->upcall_data);
    break;

  case 'q':
    ask_quit(app);
    break;
    
  case 'd':
    switch (current_type) {
    case INFO_TYPE_REPLY_QUOTE_LIMIT:
      state->reply_quote_limit = -1;
      break;

    case INFO_TYPE_POLL_DELAY:
      state->poll_delay = 0;
      break;
    }
    break;
    
  case KEY_LEFT:
    switch (current_type) {
    case INFO_TYPE_THREAD_TYPE:
      switch (state->thread_type) {
      case ETPAN_THREAD_TYPE_NONE:
        state->thread_type = ETPAN_THREAD_TYPE_BY_REFERENCES_NO_SUBJECT;
        break;
      case ETPAN_THREAD_TYPE_BY_REFERENCES:
        state->thread_type = ETPAN_THREAD_TYPE_NONE;
        break;
      case ETPAN_THREAD_TYPE_BY_REFERENCES_NO_SUBJECT:
        state->thread_type = ETPAN_THREAD_TYPE_BY_REFERENCES;
        break;
      }
      break;
    }
    break;

  case KEY_RIGHT:
    switch (current_type) {
    case INFO_TYPE_THREAD_TYPE:
      switch (state->thread_type) {
      case ETPAN_THREAD_TYPE_NONE:
        state->thread_type = ETPAN_THREAD_TYPE_BY_REFERENCES;
        break;
      case ETPAN_THREAD_TYPE_BY_REFERENCES:
        state->thread_type = ETPAN_THREAD_TYPE_BY_REFERENCES_NO_SUBJECT;
        break;
      case ETPAN_THREAD_TYPE_BY_REFERENCES_NO_SUBJECT:
        state->thread_type = ETPAN_THREAD_TYPE_NONE;
        break;
      }
      break;
    }
    break;

  case '\n':
    switch (current_type) {
    case INFO_TYPE_DISPLAY_CHARSET:
      etpan_cfg_edit_string(app, "Enter display charset: ",
          state->display_charset, sizeof(state->display_charset), 0);
      break;

    case INFO_TYPE_EDITOR_CHARSET:
      etpan_cfg_edit_string(app, "Enter editor charset: ",
          state->editor_charset, sizeof(state->editor_charset), 0);
      break;

    case INFO_TYPE_MESSAGE_CHARSET:
      etpan_cfg_edit_string(app, "Enter message charset: ",
          state->message_charset, sizeof(state->message_charset), 0);
      break;

    case INFO_TYPE_EDITOR:
      etpan_cfg_edit_filename(app, "Enter editor command: ",
          state->editor, sizeof(state->editor));
      break;

    case INFO_TYPE_REPLY_QUOTE_LIMIT:
      etpan_cfg_edit_int(app, "Enter reply quote limit: ",
          &state->reply_quote_limit);
      break;

    case INFO_TYPE_NETWORK_TIMEOUT:
      etpan_cfg_edit_int(app,
          "Enter network timeout (in sec): ",
          &state->network_timeout);
      break;

    case INFO_TYPE_POLL_DELAY:
      etpan_cfg_edit_int(app, "Enter poll delay (in sec, 0 to disable): ",
          &state->poll_delay);
      break;
      
    case INFO_TYPE_SENDMAIL_PATH:
      etpan_cfg_edit_filename(app, "Enter sendmail command: ",
          state->sendmail_path, sizeof(state->sendmail_path));
      break;
    }
    break;
  }
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  unsigned int count;
  unsigned int i;
  unsigned int y;
  unsigned int list_lines;
  char * buffer;
  char * output;
  char * fill;
  unsigned int percent;

  buffer = app->app->buffer;
  output = app->app->output;
  fill = app->app->fill;
  
  list_lines = app->display_height - 1;
  
  state = app->data;
  
  /* update view */
  
  count = TABLE_SIZE(info_type_tab);
  state->count = count;

  if (state->index > state->count - 1)
    state->index = state->count - 1;
  
  if (state->index < state->first)
    state->first = state->index;
  if (state->index - state->first + 1 > (unsigned int) list_lines)
    state->first = state->index - list_lines + 1;
  
  /* display */
  
  wattron(w, state->main_attr);
  y = 0;
  i = state->first;
  while (y < list_lines) {
    
    if (i > count - 1)
      break;
    
    switch (info_type_tab[i]) {
    case INFO_TYPE_DISPLAY_CHARSET:
      snprintf(buffer, app->display_width, "display charset: %s",
          state->display_charset);
      break;
      
    case INFO_TYPE_EDITOR_CHARSET:
      snprintf(buffer, app->display_width, "editor charset: %s",
          state->editor_charset);
      break;

    case INFO_TYPE_MESSAGE_CHARSET:
      snprintf(buffer, app->display_width, "message charset: %s",
          state->message_charset);
      break;

    case INFO_TYPE_EDITOR:
      snprintf(buffer, app->display_width, "editor: %s",
          state->editor);
      break;

    case INFO_TYPE_REPLY_QUOTE_LIMIT:
      if (state->reply_quote_limit == -1)
        snprintf(buffer, app->display_width, "reply quote limit: (no limit)");
      else
        snprintf(buffer, app->display_width, "reply quote limit: %i",
            state->reply_quote_limit);
      break;

    case INFO_TYPE_POLL_DELAY:
      if (state->poll_delay == 0)
        snprintf(buffer, app->display_width, "poll delay: (no poll)");
      else
        snprintf(buffer, app->display_width, "poll delay: %i sec",
            state->poll_delay);
      break;

    case INFO_TYPE_NETWORK_TIMEOUT:
      snprintf(buffer, app->display_width, "network timeout: %i sec",
          state->network_timeout);
      break;

    case INFO_TYPE_SENDMAIL_PATH:
      snprintf(buffer, app->display_width, "sendmail: %s",
          state->sendmail_path);
      break;
      
    case INFO_TYPE_THREAD_TYPE:
      {
        char * str;
        
        switch (state->thread_type) {
        case ETPAN_THREAD_TYPE_BY_REFERENCES:
          str = "references";
          break;
        case ETPAN_THREAD_TYPE_BY_REFERENCES_NO_SUBJECT:
          str = "references without subject";
          break;
        case ETPAN_THREAD_TYPE_NONE:
        default:
          str = "none";
          break;
        }
        snprintf(buffer, app->display_width, "thread type: %s", str);
      }
      break;
    }
    if (i == state->index) {
      wattroff(w, state->main_attr);
      wattron(w, state->selection_attr);
    }
    snprintf(output, app->display_width, "%s%s", buffer, fill);
    mvwprintw(w, y, 0, "%s", output);
    if (i == state->index) {
      wattroff(w, state->selection_attr);
      wattron(w, state->main_attr);
    }
    
    i ++;
    y ++;
  }

  while (y < list_lines) {
    mvwprintw(w, y, 0, "%s", fill);
    y ++;
  }

  wattroff(w, state->main_attr);
  
  /* status bar */

  wattron(w, state->status_attr);
  
  if (state->count == 0)
    percent = 0;
  else if (state->count == 1)
	percent = 100;
  else
    percent = state->index * 100 / (state->count-1);
  
  snprintf(output, app->display_width + 1,
      " %3i %% | y: ok  ^G: cancel%s", percent, fill);
  
  mvwprintw(w, app->display_height - 1, 0, "%s", output);
  
  wattroff(w, state->status_attr);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

static int init(struct etpan_subapp * app)
{
  struct app_state * state;

  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->display_charset[0] = '\0';
  state->editor_charset[0] = '\0';
  state->message_charset[0] = '\0';
  state->editor[0] = '\0';
  state->reply_quote_limit = -1;
  state->network_timeout = 0;
  state->poll_delay = 0;
  state->sendmail_path[0] = '\0';
  state->thread_type = ETPAN_THREAD_TYPE_NONE;

  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;

  state->index = 0;
  state->first = 0;
  state->count = 0;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
  
  app->data = state;
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;
  
  free(state);
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - edit storage");
  return etpan_app_subapp_display_init(app);
}

struct etpan_subapp * etpan_cfg_global_edit_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_cfg_global_edit_app_driver);
}

void etpan_cfg_global_edit_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  state->display_charset[0] = '\0';
  state->editor_charset[0] = '\0';
  state->message_charset[0] = '\0';
  state->editor[0] = '\0';
  state->reply_quote_limit = -1;
  state->network_timeout = 0;
  state->poll_delay = 0;
  state->sendmail_path[0] = '\0';
  state->thread_type = ETPAN_THREAD_TYPE_NONE;

  state->index = 0;
  state->first = 0;
  state->count = 0;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
}


static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_cfg_global_edit_app_flush(app);
}



void etpan_cfg_global_edit_app_set(struct etpan_subapp * app,
    struct etpan_global_config * global_config,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * upcall_data)
{
  struct app_state * state;
  
  state = app->data;
  
  state->upcall = upcall;
  state->upcall_data = upcall_data;
  
  if (global_config->display_charset != NULL)
    strncpy(state->display_charset, global_config->display_charset,
        sizeof(state->display_charset));
  
  if (global_config->editor_charset != NULL)
    strncpy(state->editor_charset, global_config->editor_charset,
        sizeof(state->editor_charset));

  if (global_config->message_charset != NULL)
    strncpy(state->message_charset, global_config->message_charset,
        sizeof(state->message_charset));

  if (global_config->editor != NULL)
    strncpy(state->editor, global_config->editor,
        sizeof(state->editor));
  
  state->reply_quote_limit = global_config->reply_quote_limit;
  state->network_timeout = global_config->network_timeout;
  state->poll_delay = global_config->poll_delay;

  if (global_config->sendmail_path != NULL)
    strncpy(state->sendmail_path, global_config->sendmail_path,
        sizeof(state->sendmail_path));

  state->thread_type = global_config->thread_type;
}

int etpan_cfg_global_edit_app_get(struct etpan_subapp * app,
    struct etpan_global_config * global_config)
{
  struct app_state * state;
  int r;
  char * display_charset;
  char * editor_charset;
  char * message_charset;
  char * editor;
  char * sendmail_path;
  int res;

  state = app->data;
  
  r = etpan_cfg_strdup(&display_charset, state->display_charset);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  r = etpan_cfg_strdup(&editor_charset, state->editor_charset);
  if (r != NO_ERROR) {
    res = r;
    goto free_display_charset;
  }

  r = etpan_cfg_strdup(&message_charset, state->message_charset);
  if (r != NO_ERROR) {
    res = r;
    goto free_editor_charset;
  }

  r = etpan_cfg_strdup(&editor, state->editor);
  if (r != NO_ERROR) {
    res = r;
    goto free_message_charset;
  }

  r = etpan_cfg_strdup(&sendmail_path, state->sendmail_path);
  if (r != NO_ERROR) {
    res = r;
    goto free_editor;
  }

  if (global_config->display_charset != NULL)
    free(global_config->display_charset);
  global_config->display_charset = display_charset;
  
  if (global_config->editor_charset != NULL)
    free(global_config->editor_charset);
  global_config->editor_charset = editor_charset;
  
  if (global_config->message_charset != NULL)
    free(global_config->message_charset);
  global_config->message_charset = message_charset;
  
  if (global_config->editor != NULL)
    free(global_config->editor);
  global_config->editor = editor;

  if (global_config->sendmail_path != NULL)
    free(global_config->sendmail_path);
  global_config->sendmail_path = sendmail_path;
  
  global_config->reply_quote_limit = state->reply_quote_limit;
  global_config->poll_delay = state->poll_delay;
  global_config->network_timeout = state->network_timeout;
  global_config->thread_type = state->thread_type;
  
  return NO_ERROR;
  
 free_editor:
  if (editor != NULL)
    free(editor);
 free_message_charset:
  if (message_charset != NULL)
    free(message_charset);
 free_editor_charset:
  if (editor_charset != NULL)
    free(editor_charset);
 free_display_charset:
  if (display_charset != NULL)
    free(display_charset);
 err:
  return res;
}


static void ask_quit_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct app_state * state;
  char * name;
  char * dup_name;
  struct etpan_subapp * app;
  int r;
  char * result;
  int do_quit;

  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  result = etpan_search_input_get_value(input_app);
  if ((* result != 'y') && (* result != 'n')) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  do_quit = (* result == 'y');
  etpan_app_quit_subapp(input_app);
  
  if (do_quit) {
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_CFG_GLOBAL_EDIT_VALID, state->upcall_data);
  }
  else {
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_CFG_GLOBAL_EDIT_CANCEL, state->upcall_data);
  }
}


static void ask_quit(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * input;
  int r;
  
  input = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (input == NULL) {
    input = etpan_search_input_new(app->app);
    if (input == NULL)
      goto err;
  }
  
  r = etpan_search_input_set(input,
      "save the configuration (y/n) ? ", 1,
      NULL, 0,
      ask_quit_upcall, app);
  if (input == NULL)
    goto err;
  
  etpan_subapp_set_parent(input, app);
  etpan_app_switch_subapp(input, 0);
  
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "configuration - not enough memory"));
}


#define HELP_TEXT \
"\
Help for global configuration editor\n\
------------------------------------\n\
\n\
This application will let you edit the global configuration.\n\
\n\
- up, down\n\
  arrow keys : move cursor\n\
\n\
- left, right\n\
  arrow keys : change value\n\
- d          : clear selected value \n\
- [Enter]    : edit value\n\
\n\
- y          : finished edition of global configuration\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
