/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-header-common.c,v 1.6 2004/11/12 12:13:27 nyoxi Exp $
 */

#include "etpan-header-common.h"

#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include "etpan-imf-helper.h"
#include "etpan-address-input.h"
#include "etpan-search-input.h"
#include "etpan-header-input.h"
#include "etpan-app.h"
#include "etpan-msg-new.h"
#include "etpan-subapp-thread.h"
#include "etpan-mime-tools.h"

void etpan_header_common_set_color(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

static int precalc(struct etpan_header_common_state * state);
static void precalc_free(struct etpan_header_common_state * state);


void etpan_header_common_fields_adjust(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  struct mailimf_single_fields single_fields;
  
  state = app->data;
  
  mailimf_single_fields_init(&single_fields, state->fields);
  
  precalc_free(state);
  precalc(state);
}


static int replace_addr_list(struct mailimf_address_list * addr_list,
    struct mailimf_address_list ** paddr_list)
{
  if (* paddr_list != NULL)
    mailimf_address_list_free(* paddr_list);
  * paddr_list = addr_list;
  
  return NO_ERROR;
}


static int replace_mb(struct mailimf_address_list * addr_list,
    struct mailimf_mailbox ** pmb)
{
  struct mailimf_mailbox_list * new_mb_list;
  clistiter * cur;
  struct mailimf_mailbox * mb;
  
  new_mb_list = etpan_address_to_mailbox_list(addr_list);
  mailimf_address_list_free(addr_list);
  if (new_mb_list == NULL)
    return ERROR_MEMORY;
  
  cur = clist_begin(new_mb_list->mb_list);
  if (cur == NULL)
    return ERROR_INVAL;
  
  mb = cur->data;
  clist_delete(new_mb_list->mb_list, cur);
  mailimf_mailbox_list_free(new_mb_list);
  
  mailimf_mailbox_free(* pmb);
  * pmb = mb;
  
  return NO_ERROR;
}


/*
  replace_mb_list
  free addr_list on return
*/

static int replace_mb_list(struct mailimf_address_list * addr_list,
    struct mailimf_mailbox_list ** pmb_list)
{
  struct mailimf_mailbox_list * new_mb_list;
  
  new_mb_list = etpan_address_to_mailbox_list(addr_list);
  mailimf_address_list_free(addr_list);
  if (new_mb_list == NULL)
    return ERROR_MEMORY;
  
  mailimf_mailbox_list_free(* pmb_list);
  * pmb_list = new_mb_list;

  return NO_ERROR;
}


static void addr_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct etpan_app * app;
  struct mailimf_address_list * new_addr_list;
  char * str;
  size_t cur_token;
  struct mailimf_field * field;
  int r;

  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    goto err;
  }
  
  app = input_app->app;
  
  str = etpan_address_input_get_value(input_app);
  if (str == NULL) {
    ETPAN_APP_DEBUG((app, "header editor - BUG detected, NULL string"));
    goto err;
  }
  
  cur_token = 0;
  r = mailimf_address_list_parse(str, strlen(str), &cur_token,
      &new_addr_list);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_APP_LOG((app, "header editor - error parsing address list"));
    goto err_no_quit;
  }
  
  etpan_addr_list_encode(app, new_addr_list);
  
  /* free old address list and replace with new address list */
  field = data;
  switch (field->fld_type) {
  case MAILIMF_FIELD_RESENT_FROM:
    r = replace_mb_list(new_addr_list,
        &field->fld_data.fld_resent_from->frm_mb_list);
    break;
    
  case MAILIMF_FIELD_RESENT_SENDER:
    r = replace_mb(new_addr_list,
        &field->fld_data.fld_resent_sender->snd_mb);
    break;

  case MAILIMF_FIELD_RESENT_TO:
    r = replace_addr_list(new_addr_list,
        &field->fld_data.fld_resent_to->to_addr_list);
    break;

  case MAILIMF_FIELD_RESENT_CC:
    r = replace_addr_list(new_addr_list,
        &field->fld_data.fld_resent_cc->cc_addr_list);
    break;

  case MAILIMF_FIELD_RESENT_BCC:
    r = replace_addr_list(new_addr_list,
        &field->fld_data.fld_resent_bcc->bcc_addr_list);
    break;

  case MAILIMF_FIELD_FROM:
    r = replace_mb_list(new_addr_list,
        &field->fld_data.fld_from->frm_mb_list);
    break;

  case MAILIMF_FIELD_SENDER:
    r = replace_mb(new_addr_list,
        &field->fld_data.fld_sender->snd_mb);
    break;

  case MAILIMF_FIELD_REPLY_TO:
    r = replace_addr_list(new_addr_list,
        &field->fld_data.fld_reply_to->rt_addr_list);
    break;

  case MAILIMF_FIELD_TO:
    r = replace_addr_list(new_addr_list,
        &field->fld_data.fld_to->to_addr_list);
    break;

  case MAILIMF_FIELD_CC:
    r = replace_addr_list(new_addr_list,
        &field->fld_data.fld_cc->cc_addr_list);
    break;

  case MAILIMF_FIELD_BCC:
    r = replace_addr_list(new_addr_list,
        &field->fld_data.fld_bcc->bcc_addr_list);
    break;
  }
  
  switch (r) {
  case ERROR_MEMORY:
    ETPAN_APP_LOG((app, "header editor - not enought memory"));
    break;
  case ERROR_PARSE:
    ETPAN_APP_LOG((app, "header editor - invalid address"));
    break;
  }
  
  etpan_app_quit_subapp(input_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(input_app);
 err_no_quit:
  return;
}



static int edit_addr_list(struct etpan_subapp * app,
    char * prompt,
    struct mailimf_address_list * addr_list,
    struct mailimf_field * field)
{
  char * str;
  struct etpan_subapp * input_app;
  int r;
  struct mailimf_address_list * dup_addr_list;
  
  dup_addr_list = etpan_dup_address_list(addr_list);
  if (dup_addr_list == NULL)
    goto err;
  etpan_addr_list_decode(app->app, dup_addr_list);
  str = mailimf_address_list_to_string(dup_addr_list);
  mailimf_address_list_free(dup_addr_list);
  if (str == NULL) {
    goto err;
  }
  
  input_app = etpan_app_find_subapp(app->app, "address-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_address_input_new(app->app);
    if (input_app == NULL) {
      free(str);
      goto err;
    }
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  r = etpan_address_input_set(input_app, prompt, 1024, str,
      addr_upcall, field);
  free(str);
  if (r != NO_ERROR) {
    goto err;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static int edit_mb_list(struct etpan_subapp * app,
    char * prompt,
    struct mailimf_mailbox_list * mb_list,
    struct mailimf_field * field)
{
  struct mailimf_address_list * addr_list;
  int r;

  addr_list = etpan_mailbox_to_address_list(mb_list);
  if (addr_list == NULL)
    return ERROR_MEMORY;
  
  r = edit_addr_list(app, prompt, addr_list, field);
  
  mailimf_address_list_free(addr_list);
  
  return r;
}

static int edit_mb(struct etpan_subapp * app,
    char * prompt,
    struct mailimf_mailbox * mb,
    struct mailimf_field * field)
{
  struct mailimf_address_list * addr_list;
  struct mailimf_mailbox_list * mb_list;
  int r;
  
  mb_list = mailimf_mailbox_list_new_empty();
  if (mb_list == NULL)
    return ERROR_MEMORY;
  
  r = mailimf_mailbox_list_add(mb_list, mb);
  if (r != MAILIMF_NO_ERROR) {
    mailimf_mailbox_list_free(mb_list);
    return ERROR_MEMORY;
  }
  
  addr_list = etpan_mailbox_to_address_list(mb_list);
  mailimf_mailbox_list_free(mb_list);
  if (addr_list == NULL)
    return ERROR_MEMORY;
  
  r = edit_addr_list(app, prompt, addr_list, field);
  
  mailimf_address_list_free(addr_list);
  
  return r;
}

static void subject_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct etpan_app * app;
  char * str;
  char * subject_value;
  struct mailimf_field * field;

  if (valid == ETPAN_INPUT_COMMON_CANCEL)
    goto err;
  
  field = data;
  app = input_app->app;
  
  str = etpan_search_input_get_value(input_app);
  if (str == NULL) {
    ETPAN_APP_DEBUG((app, "header editor - BUG detected, NULL string"));
    goto err;
  }

  subject_value = etpan_encode_mime_header(app, str);
  if (subject_value == NULL) {
    ETPAN_APP_LOG((app, "header editor - not enough memory"));
    goto err;
  }
  
  free(field->fld_data.fld_subject->sbj_value);
  field->fld_data.fld_subject->sbj_value = subject_value;
  
  etpan_app_quit_subapp(input_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(input_app);
}

static void delete_field(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  struct mailimf_field * field;
  clistiter * cur;
  
  if (state->selected >= carray_count(state->fields_tab))
    return;
  
  field = carray_get(state->fields_tab, state->selected);
  if (field->fld_type == MAILIMF_FIELD_SUBJECT) {
    ETPAN_APP_LOG((app->app, "header editor - can't delete subject"));
    return;
  }
  if (field->fld_type == MAILIMF_FIELD_ORIG_DATE) {
    ETPAN_APP_LOG((app->app, "header editor - can't delete date"));
    return;
  }
  if (field->fld_type == MAILIMF_FIELD_RESENT_DATE) {
    ETPAN_APP_LOG((app->app, "header editor - can't delete resent date"));
    return;
  }
  
  for(cur = clist_begin(state->fields->fld_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    if (clist_content(cur) == field) {
      clist_delete(state->fields->fld_list, cur);
      mailimf_field_free(field);
      break;
    }
  }
  
  precalc_free(state);
  precalc(state);
}

static int edit_subject(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  char * str;
  struct etpan_subapp * input_app;
  int r;
  struct mailimf_field * field;
  char * decoded_subject;
  char * subject;
  
  input_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_search_input_new(app->app);
    if (input_app == NULL)
      goto err;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  field = carray_get(state->fields_tab, state->selected);
  str = field->fld_data.fld_subject->sbj_value;
  
  decoded_subject = etpan_decode_mime_header(app->app, str);
  if (decoded_subject != NULL)
    subject = decoded_subject;
  else
    subject = str;
  
  r = etpan_search_input_set(input_app, "Subject: ", 1024, subject,
      0, subject_upcall, field);
  if (decoded_subject != NULL)
    free(decoded_subject);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "header input - not enough memory"));
    goto err;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

struct optional_header_edit_info {
  struct mailimf_field * field;
  struct etpan_subapp * app;
  struct etpan_header_common_state * state;
};

static void header_value_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  char * str;
  struct mailimf_field * field;
  struct optional_header_edit_info * optional_info;
  struct etpan_subapp * app;
  char * header_value;
  
  optional_info = data;
  field = optional_info->field;
  app = optional_info->app;
  free(optional_info);
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    goto err;
  }
  
  /* set header name */
  
  str = etpan_search_input_get_value(input_app);
  if (str == NULL) {
    ETPAN_APP_DEBUG((app->app, "header editor - BUG detected, NULL string"));
    goto err;
  }
  
  header_value = etpan_encode_mime_header(app->app, str);
  if (header_value == NULL) {
    ETPAN_APP_LOG((app->app, "header editor - not enough memory"));
    goto err;
  }
  
  free(field->fld_data.fld_optional_field->fld_value);
  field->fld_data.fld_optional_field->fld_value = header_value;

  etpan_app_quit_subapp(input_app);
  
  return;

 err:
  etpan_app_quit_subapp(input_app);
  return;
}


static void header_name_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  char * str;
  int r;
  struct mailimf_field * field;
  size_t cur_token;
  struct optional_header_edit_info * optional_info;
  struct etpan_subapp * app;
  struct etpan_subapp * value_app;
  char * header_name;
  char * decoded_value;
  char * value;
  struct etpan_header_common_state * state;

  optional_info = data;
  field = optional_info->field;
  app = optional_info->app;
  state = optional_info->state;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    goto err;
  }
  
  /* set header name */
  
  str = etpan_search_input_get_value(input_app);
  if (str == NULL) {
    ETPAN_APP_DEBUG((app->app, "header editor - BUG detected, NULL string"));
    goto err;
  }
  
  header_name = etpan_encode_mime_header(app->app, str);
  if (header_name == NULL) {
    ETPAN_APP_LOG((app->app, "header editor - not enough memory"));
    goto err;
  }
  
  free(field->fld_data.fld_optional_field->fld_name);
  field->fld_data.fld_optional_field->fld_name = header_name;
  
  /* set input for value */
  
  str = field->fld_data.fld_optional_field->fld_value;

  decoded_value = NULL;
  cur_token = 0;
  decoded_value = etpan_decode_mime_header(app->app, str);
  if (decoded_value != NULL)
    value = decoded_value;
  else
    value = str;
  
  value_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (value_app == NULL) {
    value_app = etpan_search_input_new(app->app);
    if (input_app == NULL)
      goto err;
  }
  
  etpan_subapp_set_parent(value_app, app);
  
  r = etpan_search_input_set(value_app, "Header value: ", 1024, value,
      0, header_value_upcall, optional_info);
  if (decoded_value != NULL)
    free(decoded_value);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "header editor - not enough memory"));
    goto err;
  }
  
  etpan_app_quit_subapp(input_app);
  
  etpan_app_switch_subapp(value_app, 0);
  
  return;
  
 err:
  etpan_app_quit_subapp(input_app);
  free(optional_info);
  return;
}


static int edit_header_name(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  char * str;
  struct etpan_subapp * input_app;
  int r;
  struct mailimf_field * field;
  struct optional_header_edit_info * optional_info;
  
  input_app = etpan_app_find_subapp(app->app, "header-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_header_input_new(app->app);
    if (input_app == NULL)
      goto err;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  field = carray_get(state->fields_tab, state->selected);
  str = field->fld_data.fld_optional_field->fld_name;
  
  optional_info = malloc(sizeof(* optional_info));
  if (optional_info == NULL)
    goto err;
  
  optional_info->app = app;
  optional_info->field = field;
  optional_info->state = state;
  
  r = etpan_header_input_set(input_app, "Header name: ", 1024, str,
      header_name_upcall, optional_info);
  if (r != NO_ERROR) {
    free(optional_info);
    goto err;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static void edit_field(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  struct mailimf_field * field;
  int r;
  
  if (state->selected >= carray_count(state->fields_tab)) {
    ETPAN_APP_DEBUG((app->app,
                        "header editor - BUG detected in header editor"));
    return;
  }
  
  r = ERROR_INVAL;
  field = carray_get(state->fields_tab, state->selected);
  switch (field->fld_type) {
  case MAILIMF_FIELD_RESENT_FROM:
    r = edit_mb_list(app, "Resent-From: ",
        field->fld_data.fld_resent_from->frm_mb_list, field);
    break;
    
  case MAILIMF_FIELD_RESENT_SENDER:
    r = edit_mb(app, "Resent-Sender: ",
        field->fld_data.fld_resent_sender->snd_mb, field);
    break;

  case MAILIMF_FIELD_RESENT_TO:
    r = edit_addr_list(app, "Resent-To: ",
        field->fld_data.fld_resent_to->to_addr_list, field);
    break;

  case MAILIMF_FIELD_RESENT_CC:
    r = edit_addr_list(app, "Resent-Cc: ",
        field->fld_data.fld_resent_cc->cc_addr_list, field);
    break;

  case MAILIMF_FIELD_RESENT_BCC:
    r = edit_addr_list(app, "Resent-Bcc: ",
        field->fld_data.fld_resent_bcc->bcc_addr_list, field);
    break;

  case MAILIMF_FIELD_FROM:
    r = edit_mb_list(app, "From: ",
        field->fld_data.fld_from->frm_mb_list, field);
    break;

  case MAILIMF_FIELD_SENDER:
    r = edit_mb(app, "Sender: ",
        field->fld_data.fld_sender->snd_mb, field);
    break;

  case MAILIMF_FIELD_REPLY_TO:
    r = edit_addr_list(app, "Reply-To: ",
        field->fld_data.fld_reply_to->rt_addr_list, field);
    break;

  case MAILIMF_FIELD_TO:
    r = edit_addr_list(app, "To: ",
        field->fld_data.fld_to->to_addr_list, field);
    break;

  case MAILIMF_FIELD_CC:
    r = edit_addr_list(app, "Cc: ",
        field->fld_data.fld_cc->cc_addr_list, field);
    break;

  case MAILIMF_FIELD_BCC:
    r = edit_addr_list(app, "Bcc: ",
        field->fld_data.fld_bcc->bcc_addr_list, field);
    break;
    
  case MAILIMF_FIELD_RESENT_DATE:
  case MAILIMF_FIELD_ORIG_DATE:
    ETPAN_APP_LOG((app->app, "header editor - can't edit this field"));
    r = NO_ERROR;
    break;
    
  case MAILIMF_FIELD_SUBJECT:
    r = edit_subject(app, state);
    break;
    
  case MAILIMF_FIELD_OPTIONAL_FIELD:
    r = edit_header_name(app, state);
    break;
  }
  
  if (r != NO_ERROR)
    ETPAN_APP_LOG((app->app, "header editor - not enough memory"));
}

static int add_field(struct etpan_subapp * app,
    struct etpan_header_common_state * state,
    int type)
{
  struct mailimf_address_list * addr;
  struct mailimf_mailbox_list * mb;
  struct mailimf_field * field;
  struct mailimf_from * from;
  struct mailimf_reply_to * reply_to;
  struct mailimf_to * to;
  struct mailimf_cc * cc;
  struct mailimf_bcc * bcc;
  struct mailimf_from * resent_from;
  struct mailimf_to * resent_to;
  struct mailimf_cc * resent_cc;
  struct mailimf_bcc * resent_bcc;
  struct mailimf_subject * subject;
  struct mailimf_optional_field * optional_field;
  unsigned int i;
  int r;
  char * header_name;
  char * header_value;
  char * subject_value;
  
  resent_from = NULL;
  resent_to = NULL;
  resent_cc = NULL;
  resent_bcc = NULL;
  from = NULL;
  reply_to = NULL;
  to = NULL;
  cc = NULL;
  bcc = NULL;
  optional_field = NULL;
  subject = NULL;
  
  switch (type) {
  case MAILIMF_FIELD_RESENT_FROM:
    mb = mailimf_mailbox_list_new_empty();
    if (mb == NULL)
      goto err;
    
    resent_from = mailimf_from_new(mb);
    if (resent_from == NULL) {
      mailimf_mailbox_list_free(mb);
      goto err;
    }
    
    break;

  case MAILIMF_FIELD_RESENT_TO:
    addr = mailimf_address_list_new_empty();
    if (addr == NULL)
      goto err;
    
    resent_to = mailimf_to_new(addr);
    if (resent_to == NULL)
      goto free_addr;
    
    break;

  case MAILIMF_FIELD_RESENT_CC:
    addr = mailimf_address_list_new_empty();
    if (addr == NULL)
      goto err;
    
    resent_cc = mailimf_cc_new(addr);
    if (resent_cc == NULL)
      goto free_addr;
    
    break;

  case MAILIMF_FIELD_RESENT_BCC:
    addr = mailimf_address_list_new_empty();
    if (addr == NULL)
      goto err;
    
    resent_bcc = mailimf_bcc_new(addr);
    if (resent_bcc == NULL)
      goto free_addr;
    
    break;

  case MAILIMF_FIELD_FROM:
    mb = mailimf_mailbox_list_new_empty();
    if (mb == NULL)
      goto err;
    
    from = mailimf_from_new(mb);
    if (from == NULL) {
      mailimf_mailbox_list_free(mb);
      goto err;
    }
    
    break;

  case MAILIMF_FIELD_REPLY_TO:
    addr = mailimf_address_list_new_empty();
    if (addr == NULL)
      goto err;
    
    reply_to = mailimf_reply_to_new(addr);
    if (reply_to == NULL)
      goto free_addr;
    
    break;
    
  case MAILIMF_FIELD_TO:
    addr = mailimf_address_list_new_empty();
    if (addr == NULL)
      goto err;
    
    to = mailimf_to_new(addr);
    if (to == NULL)
      goto free_addr;
    
    break;

  case MAILIMF_FIELD_CC:
    addr = mailimf_address_list_new_empty();
    if (addr == NULL)
      goto err;
    
    cc = mailimf_cc_new(addr);
    if (cc == NULL)
      goto free_addr;
    
    break;

  case MAILIMF_FIELD_BCC:
    addr = mailimf_address_list_new_empty();
    if (addr == NULL)
      goto err;
    
    bcc = mailimf_bcc_new(addr);
    if (bcc == NULL)
      goto free_addr;
    
    break;

  case MAILIMF_FIELD_SUBJECT:
    subject_value = strdup("");
    if (subject_value == NULL)
      goto err;
    
    subject = mailimf_subject_new(subject_value);
    if (subject == NULL) {
      free(subject_value);
      goto err;
    }
    break;

  case MAILIMF_FIELD_OPTIONAL_FIELD:
    header_name = strdup("X-");
    if (header_name == NULL)
      goto err;
    
    header_value = strdup("");
    if (header_value == NULL) {
      free(header_name);
      goto err;
    }
    
    optional_field = mailimf_optional_field_new(header_name, header_value);
    if (optional_field == NULL) {
      free(header_value);
      free(header_name);
      goto err;
    }
    
    break;
  }
  
  field = mailimf_field_new(type,
      NULL /* return-path */,
      NULL /* resent date */,
      resent_from /* resent from */,
      NULL /* resent sender */,
      resent_to /* resent to */,
      resent_cc /* resent cc */,
      resent_bcc /* resent bcc */,
      NULL /* resent msg id */,
      NULL /* date */,
      from /* from */,
      NULL /* sender */,
      reply_to /* reply-to */,
      to /* to */,
      cc /* cc */,
      bcc /* bcc */,
      NULL /* message id */,
      NULL /* in reply to */,
      NULL /* references */,
      subject /* subject */,
      NULL /* comments */,
      NULL /* keywords */,
      optional_field /* optional field */);
  if (field == NULL) {
    if (from != NULL)
      mailimf_from_free(from);
    if (reply_to != NULL)
      mailimf_reply_to_free(reply_to);
    if (to != NULL)
      mailimf_to_free(to);
    if (cc != NULL)
      mailimf_cc_free(cc);
    if (bcc != NULL)
      mailimf_bcc_free(bcc);
    if (optional_field != NULL)
      mailimf_optional_field_free(optional_field);
    if (subject != NULL)
      mailimf_subject_free(subject);
    goto err;
  }
  
  r = mailimf_fields_add(state->fields, field);
  if (r != MAILIMF_NO_ERROR) {
    mailimf_field_free(field);
    goto err;
  }
  
  precalc_free(state);
  precalc(state);

  for(i = 0 ; i < carray_count(state->fields_tab) ; i ++) {
    struct mailimf_field * cur_field;
    
    cur_field = carray_get(state->fields_tab, i);
    if (cur_field == field) {
      state->selected = i;
      edit_field(app, state);
    }
  }
  
  return NO_ERROR;
  
 free_addr:
  mailimf_address_list_free(addr);
 err:
  return ERROR_MEMORY;
}

void etpan_header_common_handle_key(struct etpan_subapp * app,
    struct etpan_header_common_state * state,
    int key)
{
  unsigned int list_lines;
  int do_edit_message;
  
  list_lines = app->display_height - 1;
  
  do_edit_message = 0;
  
  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
  case 'G':
    break;
  default:
    state->chosen = 0;
    break;
  }

  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
    state->chosen *= 10;
    state->chosen += key - '0';
    break;
  case 'G':
    if (state->chosen == 0) {
      state->selected = state->fields_tab->len - 1;
    }
    else {
      state->selected = state->chosen;
    }
    break;
    
  case 'j':
  case KEY_DOWN:
    state->selected ++;
    break;
    
  case 'k':
  case KEY_UP:
    if (state->selected > 0)
      state->selected --;
    break;
    
  case KEY_NPAGE:
    state->selected += list_lines - 1;
    break;
    
  case KEY_PPAGE:
    if (state->selected >= list_lines - 1)
      state->selected -= list_lines - 1;
    else
      state->selected = 0;
    break;
    
  case KEY_HOME:
    state->selected = 0;
    break;
  case KEY_END:
    state->selected = state->fields_tab->len - 1;
    break;

  case 'f':
    add_field(app, state, MAILIMF_FIELD_FROM);
    break;
    
  case 't':
    add_field(app, state, MAILIMF_FIELD_TO);
    break;

  case 'c':
    add_field(app, state, MAILIMF_FIELD_CC);
    break;

  case 'b':
    add_field(app, state, MAILIMF_FIELD_BCC);
    break;

  case 'r':
    add_field(app, state, MAILIMF_FIELD_REPLY_TO);
    break;

  case 'F':
    add_field(app, state, MAILIMF_FIELD_RESENT_FROM);
    break;
    
  case 'T':
    add_field(app, state, MAILIMF_FIELD_RESENT_TO);
    break;

  case 'C':
    add_field(app, state, MAILIMF_FIELD_RESENT_CC);
    break;

  case 'B':
    add_field(app, state, MAILIMF_FIELD_RESENT_BCC);
    break;

  case 'o':
    add_field(app, state, MAILIMF_FIELD_OPTIONAL_FIELD);
    break;

  case 's':
    add_field(app, state, MAILIMF_FIELD_SUBJECT);
    break;
    
  case '\n':
    if (state->selected < carray_count(state->fields_tab))
      edit_field(app, state);
    break;

  case 'd':
    delete_field(app, state);
    break;
  }
}


int etpan_common_header_init(struct etpan_subapp * subapp,
    struct etpan_header_common_state * state)
{
  /* fields */
  state->fields = NULL;
  state->fields_tab = carray_new(128);
  if (state->fields_tab == NULL)
    goto err;
  
  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;
  
  /* cursor & selection */
  state->chosen = 0;
  state->first = 0;
  state->selected = 0;

  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

void etpan_header_common_done(struct etpan_subapp * subapp,
    struct etpan_header_common_state * state)
{
  carray_free(state->fields_tab);
}

void etpan_header_common_flush(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  state->fields = NULL;
  precalc_free(state);
}

void etpan_header_common_leave(struct etpan_subapp * app,
    struct etpan_header_common_state * state,
    struct etpan_subapp * new_app)
{
  etpan_header_common_flush(app, state);
}

void etpan_header_common_set_fields(struct etpan_subapp * app,
    struct etpan_header_common_state * state,
    struct mailimf_fields * fields)
{
  etpan_header_common_flush(app, state);
  
  state->fields = fields;
  state->first = 0;
  state->selected = 0;
  
  precalc(state);
}

struct mailimf_fields *
etpan_header_common_get_fields(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  state = app->data;
  
  return state->fields;
}


/* ******************************************** */
/* implementation */

static int show_field(struct etpan_subapp * app,
    char * output, char * fill,
    struct mailimf_field * field)
{
  char * str;
  size_t cur_token;
  char * decoded_subject;
  char * decoded_value;
  struct mailimf_mailbox_list * dup_mb_list;
  struct mailimf_mailbox * dup_mb;
  struct mailimf_address_list * dup_addr_list;
  
  snprintf(output, app->display_width + 1, "%s", fill);
  switch (field->fld_type) {
  case MAILIMF_FIELD_RESENT_DATE:
    str = mailimf_date_time_to_string(field->fld_data.fld_resent_date->dt_date_time);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Resent-Date: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_RESENT_FROM:
    dup_mb_list = etpan_dup_mailbox_list(field->fld_data.fld_resent_from->frm_mb_list);
    if (dup_mb_list == NULL)
      return ERROR_MEMORY;
    
    etpan_mailbox_list_decode(app->app, dup_mb_list);
    str = mailimf_mailbox_list_to_string(dup_mb_list);
    mailimf_mailbox_list_free(dup_mb_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Resent-From: %s%s",
          str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_RESENT_SENDER:
    dup_mb = etpan_dup_mailbox(field->fld_data.fld_resent_sender->snd_mb);
    if (dup_mb == NULL)
      return ERROR_MEMORY;
    
    etpan_mailbox_decode(app->app, dup_mb);
    str = mailimf_mailbox_to_string(dup_mb);
    mailimf_mailbox_free(dup_mb);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Resent-Sender: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_RESENT_TO:
    dup_addr_list = etpan_dup_address_list(field->fld_data.fld_resent_to->to_addr_list);
    if (dup_addr_list == NULL)
      return ERROR_MEMORY;
    
    etpan_addr_list_decode(app->app, dup_addr_list);
    str = mailimf_address_list_to_string(dup_addr_list);
    mailimf_address_list_free(dup_addr_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Resent-To: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_RESENT_CC:
    dup_addr_list = etpan_dup_address_list(field->fld_data.fld_resent_cc->cc_addr_list);
    if (dup_addr_list == NULL)
      return ERROR_MEMORY;
    
    etpan_addr_list_decode(app->app, dup_addr_list);
    str = mailimf_address_list_to_string(dup_addr_list);
    mailimf_address_list_free(dup_addr_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Resent-Cc: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_RESENT_BCC:
    dup_addr_list = etpan_dup_address_list(field->fld_data.fld_resent_bcc->bcc_addr_list);
    if (dup_addr_list == NULL)
      return ERROR_MEMORY;
    
    etpan_addr_list_decode(app->app, dup_addr_list);
    str = mailimf_address_list_to_string(dup_addr_list);
    mailimf_address_list_free(dup_addr_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Resent-Bcc: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_ORIG_DATE:
    str = mailimf_date_time_to_string(field->fld_data.fld_orig_date->dt_date_time);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Date: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_FROM:
    dup_mb_list = etpan_dup_mailbox_list(field->fld_data.fld_from->frm_mb_list);
    if (dup_mb_list == NULL)
      return ERROR_MEMORY;
    
    etpan_mailbox_list_decode(app->app, dup_mb_list);
    str = mailimf_mailbox_list_to_string(dup_mb_list);
    mailimf_mailbox_list_free(dup_mb_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "From: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_SENDER:
    dup_mb = etpan_dup_mailbox(field->fld_data.fld_sender->snd_mb);
    if (dup_mb == NULL)
      return ERROR_MEMORY;
    
    etpan_mailbox_decode(app->app, dup_mb);
    str = mailimf_mailbox_to_string(dup_mb);
    mailimf_mailbox_free(dup_mb);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Sender: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_REPLY_TO:
    dup_addr_list = etpan_dup_address_list(field->fld_data.fld_reply_to->rt_addr_list);
    if (dup_addr_list == NULL)
      return ERROR_MEMORY;
    
    etpan_addr_list_decode(app->app, dup_addr_list);
    str = mailimf_address_list_to_string(dup_addr_list);
    mailimf_address_list_free(dup_addr_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Reply-To: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_TO:
    dup_addr_list = etpan_dup_address_list(field->fld_data.fld_to->to_addr_list);
    if (dup_addr_list == NULL)
      return ERROR_MEMORY;
    
    etpan_addr_list_decode(app->app, dup_addr_list);
    str = mailimf_address_list_to_string(dup_addr_list);
    mailimf_address_list_free(dup_addr_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "To: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_CC:
    dup_addr_list = etpan_dup_address_list(field->fld_data.fld_cc->cc_addr_list);
    if (dup_addr_list == NULL)
      return ERROR_MEMORY;
    
    etpan_addr_list_decode(app->app, dup_addr_list);
    str = mailimf_address_list_to_string(dup_addr_list);
    mailimf_address_list_free(dup_addr_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Cc: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_BCC:
    dup_addr_list = etpan_dup_address_list(field->fld_data.fld_bcc->bcc_addr_list);
    if (dup_addr_list == NULL)
      return ERROR_MEMORY;
    
    etpan_addr_list_decode(app->app, dup_addr_list);
    str = mailimf_address_list_to_string(dup_addr_list);
    mailimf_address_list_free(dup_addr_list);
    if (str == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Bcc: %s%s",
        str, fill);
    free(str);
    break;
  case MAILIMF_FIELD_SUBJECT:
    decoded_subject = etpan_decode_mime_header(app->app,
        field->fld_data.fld_subject->sbj_value);
    if (decoded_subject == NULL)
      decoded_subject = strdup(field->fld_data.fld_subject->sbj_value);
    
    if (decoded_subject == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "Subject: %s%s",
        decoded_subject, fill);
    
    free(decoded_subject);
    break;
  case MAILIMF_FIELD_OPTIONAL_FIELD:
    cur_token = 0;
    decoded_value = etpan_decode_mime_header(app->app,
        field->fld_data.fld_optional_field->fld_value);
    if (decoded_value == NULL)
      decoded_value = strdup(field->fld_data.fld_optional_field->fld_value);
    
    if (decoded_value == NULL)
      return ERROR_MEMORY;
    
    snprintf(output, app->display_width + 1, "%s: %s%s",
        field->fld_data.fld_optional_field->fld_name, decoded_value, fill);
    
    free(decoded_value);
    break;
  }
  
  return NO_ERROR;
}

static int update_view(struct etpan_subapp * app,
    struct etpan_header_common_state * state)
{
  int list_lines;
  
  list_lines = app->display_height - 1;
  
  if (state->selected > state->fields_tab->len - 1)
    state->selected = state->fields_tab->len - 1;
  
  if (state->selected > state->first + list_lines - 1)
    state->first = state->selected - list_lines + 1;
  if (state->selected < state->first)
    state->first = state->selected;
  
  return NO_ERROR;
}


static int add_fields(struct etpan_header_common_state * state,
    int type)
{
  clistiter * cur;
  int r;
  
  for(cur = clist_begin(state->fields->fld_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_field * field;
    
    field = clist_content(cur);
    
    if (field->fld_type == type) {
      r = carray_add(state->fields_tab, field, NULL);
      if (r < 0) {
        return ERROR_MEMORY;
      }
    }
  }
  
  return NO_ERROR;
}

static int precalc(struct etpan_header_common_state * state)
{
  int res;
  int r;
  
  if (state->fields == NULL)
    return NO_ERROR;
  
  r = add_fields(state, MAILIMF_FIELD_RESENT_DATE);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  add_fields(state, MAILIMF_FIELD_RESENT_SENDER);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  add_fields(state, MAILIMF_FIELD_RESENT_FROM);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_RESENT_TO);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_RESENT_CC);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_RESENT_BCC);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  add_fields(state, MAILIMF_FIELD_ORIG_DATE);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_SENDER);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_FROM);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_REPLY_TO);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_TO);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_CC);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_BCC);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_SUBJECT);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  add_fields(state, MAILIMF_FIELD_OPTIONAL_FIELD);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  return NO_ERROR;
  
 err:
  return res;
}

static void
precalc_free(struct etpan_header_common_state * state)
{
  carray_set_size(state->fields_tab, 0);
}

int etpan_header_common_display(struct etpan_subapp * app,
    struct etpan_header_common_state * state,
    WINDOW * w, char * help_str)
{
  unsigned int y;
  char * output;
  char * fill;
  unsigned int i;
  int r;
  unsigned int list_lines;
  unsigned int percent;
  
  list_lines = app->display_height - 1;

  output = app->app->output;
  fill = app->app->fill;
  
  update_view(app, state);
  
  /* list */
  
  wattron(w, state->main_attr);
  
  y = 0;
  for(i = state->first ; i < carray_count(state->fields_tab) ; i ++) {
    struct mailimf_field * field;
    char * p;
    
    if (y > list_lines)
      break;
    
    field = carray_get(state->fields_tab, i);
    
    r = show_field(app, output, fill, field);
    if (r != NO_ERROR)
      break;
    
    for(p = output ; * p != '\0' ; p ++)
      if ((* p == '\n') || (* p == '\r'))
        * p = ' ';
    
    if (i == state->selected) {
      wattroff(w, state->main_attr);
      wattron(w, state->selection_attr);
    }
    mvwaddstr(w, y, 0, output);
    if (i == state->selected) {
      wattroff(w, state->selection_attr);
      wattron(w, state->main_attr);
    }
    
    y ++;
  }
  
  while (y < list_lines) {
    mvwprintw(w, y, 0, "%s", fill);
    y ++;
  }
  
  wattroff(w, state->main_attr);

  /* status bar */

  wattron(w, state->status_attr);
  
  if (state->fields_tab->len == 0)
    percent = 0;
  else if (state->fields_tab->len == 1)
	percent = 100;
  else
    percent = state->selected * 100 / (state->fields_tab->len-1);
  
  if (help_str != NULL)
    snprintf(output, app->display_width + 1,
        " %3i %% | %s%s", percent, help_str, fill);
  else
    snprintf(output, app->display_width + 1,
        " %3i %%%s", percent, fill);
  
  mvwprintw(w, app->display_height - 1, 0, "%s", output);
  
  wattroff(w, state->status_attr);
  
  return NO_ERROR;
}
