/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: glist.h,v 1.1.26.1 2004/07/09 01:54:36 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#ifndef _GLIST_H
#define _GLIST_H

#include "hxtypes.h"

class GListIterator;
class GList;

class GListNode
{
    friend class GListIterator;
    friend class GList;
private:
    void      *m_pElement;
    GListNode *m_pNext;
    GListNode *m_pPrev;

    GListNode(void *pElement = 0L, GListNode *pNext = 0L, GListNode *pPrev = 0L)
    {
        m_pElement = pElement;
        m_pNext    = pNext;
        m_pPrev    = pPrev;
    };

    ~GListNode() {};

    void      *GetElement() { return m_pElement; }
    GListNode *GetNext()    { return m_pNext;    }
    GListNode *GetPrev()    { return m_pPrev;    }

    void       SetElement(void *pElement) { m_pElement = pElement; }
    void       SetNext(GListNode *pNext)  { m_pNext    = pNext;    }
    void       SetPrev(GListNode *pPrev)  { m_pPrev    = pPrev;    }
};

class GListIterator
{
    friend class GList;
protected:
    GListNode *m_pNode;
    GListIterator(GListNode * const pNode) : m_pNode(pNode) {};
public:
    GListIterator() {};
    GListIterator(const GListIterator & rIterator) : m_pNode(rIterator.m_pNode) {};
    ~GListIterator() {};

    GListIterator & operator = (const GListIterator & rIterator )
    {
        m_pNode = rIterator.m_pNode;
        return *this;
    };

    BOOL operator == (const GListIterator & rIterator) const
    {
        return (m_pNode == rIterator.m_pNode ? TRUE : FALSE);
    };

    BOOL operator != (const GListIterator & rIterator) const
    {
        return (m_pNode != rIterator.m_pNode ? TRUE : FALSE);
    };

    GListIterator & operator ++ ()
    {
        m_pNode = m_pNode->m_pNext;
        return *this;
    }

    GListIterator operator ++ (int)
    {
        GListIterator cIterator = *this;
        ++*this;
        return cIterator;
    };

    GListIterator & operator -- ()
    {
        m_pNode = m_pNode->m_pPrev;
        return *this;
    };

    GListIterator operator -- (int)
    {
        GListIterator cIterator = *this;
        --*this;
        return cIterator;
    };

    GListIterator operator + (int n) const
    {
        GListIterator cIterator = *this;
        if (n > 0)
        {
            while (n--)
            {
                ++cIterator;
            }
        }
        else
        {
            while (n++)
            {
                --cIterator;
            }
        }

        return cIterator;
    };

    GListIterator operator += (int n)
    {
        return *this = *this + n;
    };

    GListIterator operator - (int n) const
    {
        GListIterator cIterator = *this;
        if (n > 0)
        {
            while (n--)
            {
                --cIterator;
            }
        }
        else
        {
            while (n++)
            {
                ++cIterator;
            }
        }

        return cIterator;
    };

    GListIterator operator -= (int n)
    {
        return *this = *this - n;
    };

    void * operator * () const           { return m_pNode->m_pElement; };
    void * operator [] (int index) const { return *(*this + index); };
};

class GList
{
protected:
    GListNode m_cHead;
    ULONG32   m_ulSize;
public:
    GList() : m_cHead(), m_ulSize(0)
    {
        m_cHead.SetNext(&m_cHead);
        m_cHead.SetPrev(&m_cHead);
    };

   ~GList()
   {
       Erase(Begin(), End());
   }

    BOOL    Empty()   { return (m_ulSize ? FALSE : TRUE); }
    ULONG32 Size()    { return m_ulSize; }
	ULONG32 GetCount(){ return m_ulSize; }

    GListIterator Begin() { return GListIterator(m_cHead.m_pNext); }
    GListIterator End()   { return GListIterator(&m_cHead); }

    void * First()                 { return *Begin(); }
    void * Last()                  { return *(--End()); }
    void * operator [] (int index) { return *(Begin() + index); }

    BOOL operator == (GList & rList)
    {
        if (m_ulSize != rList.m_ulSize)
        {
            return FALSE;
        }
        GListIterator start1 = Begin();
        GListIterator end1   = End();
        GListIterator start2 = rList.Begin();
        while (start1 != end1)
        {
            if (*start1++ != *start2++)
            {
                return FALSE;
            }
        }

        return TRUE;
    }

    BOOL operator != (GList & rList)
    {
        return (*this == rList ? FALSE : TRUE);
    }

    GListIterator Insert(GListIterator cPos, void *pElement)
    {
        GListNode *pNode               = new GListNode(pElement);
        pNode->m_pNext                 = cPos.m_pNode;
        pNode->m_pPrev                 = cPos.m_pNode->m_pPrev;
        cPos.m_pNode->m_pPrev->m_pNext = pNode;
        cPos.m_pNode->m_pPrev          = pNode;
        m_ulSize++;

        return GListIterator(pNode);
    }

    GListIterator PushFront(void *pElement)
    {
        return Insert(Begin(), pElement);
    }

    GListIterator PushBack(void *pElement)
    {
        return Insert(End(), pElement);
    }

    void Erase(GListIterator cPos)
    {
        if (cPos == End())
        {
            return;
        }

        cPos.m_pNode->m_pPrev->m_pNext = cPos.m_pNode->m_pNext;
        cPos.m_pNode->m_pNext->m_pPrev = cPos.m_pNode->m_pPrev;

        delete cPos.m_pNode;

        --m_ulSize;

        return;
    }

    void Erase(GListIterator cStart, GListIterator cEnd)
    {
        while (cStart != cEnd && cStart != End())
        {
            Erase(cStart++);
        }

        return;
    }

    void PopFront()
    {
        Erase(Begin());
    }

    void PopBack()
    {
        Erase(--End());
    }

    void EraseAll()
    {
        Erase(Begin(), End());
    }

    void ResetAll()
    {
        GListNode *pCurrent = m_cHead.m_pNext;
        GListNode *pNext    = pCurrent->m_pNext;

        while (pCurrent != pNext)
        {
            pCurrent->m_pPrev->m_pNext = pCurrent->m_pNext;
            pCurrent->m_pNext->m_pPrev = pCurrent->m_pPrev;
            delete pCurrent;
            --m_ulSize;
        }

        m_cHead.m_pNext = m_cHead.m_pPrev = &m_cHead;
    }

    void Remove(void *pElement)
    {
        GListIterator cStart = Begin();
        GListIterator cEnd   = End();

        while (cStart != cEnd)
        {
            GListIterator cNext = cStart;
            ++cNext;
            if (*cStart == pElement)
            {
                Erase(cStart);
            }
            cStart = cNext;
        }
#if 0
        GListIterator itr;
        while (Size() > 0)
        {
            itr = Begin();
            if (*itr == pElement)
            {
                Erase(itr);
            }
        }
#endif
    }
};

#endif


