/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: rpfile.cpp,v 1.1.26.1 2004/07/09 01:54:51 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

// include
#include "hxtypes.h"
#include "hxwintyp.h"
#include "hxcom.h"
#include "hxresult.h"
#include "ihxpckts.h"

// hxmisc
#include "baseobj.h"
#include "unkimp.h"

// hxcont
#include "hxbuffer.h"
#include "hxslist.h"
#include "hxmap.h"

// pxcomlib
#include "pxrect.h"
#include "pxcolor.h"
#include "pxeffect.h"
#include "rpfile.h"

// hxdebug
#include "hxheap.h"
#ifdef _DEBUG
#undef HX_THIS_FILE		
static char HX_THIS_FILE[] = __FILE__;
#endif

BEGIN_INTERFACE_LIST(PXRealPixFile)
END_INTERFACE_LIST


PXRealPixFile::PXRealPixFile()
{
    m_pTitleStr           = NULL;
    m_pAuthorStr          = NULL;
    m_pCopyrightStr       = NULL;
    m_ulStart             = 0;
    m_ulDuration          = 0;
    m_ulPreroll           = 0;
    m_ulBitrate           = 0;
    m_ulWidth             = 0;
    m_ulHeight            = 0;
    m_ulTimeFormat        = kTimeFormatMilliseconds;
    m_ulBackgroundColor   = 0; // in 0x00RRGGBB format
    m_ulBackgroundOpacity = 255;
    m_ulDefaultMaxFps     = 0;
    m_bDefaultAspectFlag  = TRUE;
    m_bDefaultCenterFlag  = FALSE;
    m_pDefaultURLStr      = NULL;
    m_ulContentVersion    = HX_ENCODE_PROD_VERSION(0, 0, 0, 0);
    m_pImageMap           = NULL;
    m_pEffectsList        = NULL;
    m_pFileNameStr        = NULL;
}

PXRealPixFile::~PXRealPixFile()
{
    HX_RELEASE(m_pTitleStr);
    HX_RELEASE(m_pAuthorStr);
    HX_RELEASE(m_pCopyrightStr);
    HX_RELEASE(m_pDefaultURLStr);
    ClearImageMap();
    ClearEffectsList();
    HX_DELETE(m_pImageMap);
    HX_DELETE(m_pEffectsList);
    HX_RELEASE(m_pFileNameStr);
}

void PXRealPixFile::ClearImageMap()
{
    if (m_pImageMap)
    {
        // Run through the map, deleting all image objects
        POSITION pos = m_pImageMap->GetStartPosition();
        while (pos)
        {
            // Get the object pointer at the current position
            // and advance to the next position
            LONG32 lKey     = 0;
            void*  pElement = NULL;
            m_pImageMap->GetNextAssoc(pos, lKey, pElement);

            // Release the object
            PXImageInfo* pInfo = (PXImageInfo*) pElement;
            HX_DELETE(pInfo);
        }
        // Clear the map
        m_pImageMap->RemoveAll();
    }
}

void PXRealPixFile::ClearEffectsList()
{
    if (m_pEffectsList)
    {
        LISTPOSITION pos = m_pEffectsList->GetHeadPosition();
        while (pos)
        {
            PXEffect* pEffect = (PXEffect*) m_pEffectsList->GetNext(pos);
            HX_RELEASE(pEffect);
        }
        m_pEffectsList->RemoveAll();
    }
}

HX_RESULT PXRealPixFile::AddImage(UINT32 ulHandle, const char* pszName)
{
    HX_RESULT retVal = HXR_OK;

    IHXBuffer* pTmp = NULL;
    retVal           = SetString(pszName, &pTmp);
    if (SUCCEEDED(retVal))
    {
        retVal = AddImage(ulHandle, pTmp);
    }
    HX_RELEASE(pTmp);

    return retVal;
}

HX_RESULT PXRealPixFile::AddImage(UINT32 ulHandle, IHXBuffer* pName)
{
    HX_RESULT retVal = HXR_OK;

    if (ulHandle && pName)
    {
        if (!m_pImageMap)
        {
            m_pImageMap = new CHXMapLongToObj();
            if (!m_pImageMap)
            {
                retVal = HXR_OUTOFMEMORY;
            }
        }

        if (SUCCEEDED(retVal))
        {
            // Create a PXImageInfo object
            PXImageInfo* pInfo = new PXImageInfo;
            if (pInfo)
            {
                // Set the handle
                pInfo->m_ulHandle = ulHandle;
                // Set the string
                pInfo->m_pNameStr = pName;
                pInfo->m_pNameStr->AddRef();

                // Check if there is already an entry in the map
                void* pVoid    = NULL;
                BOOL  bPresent = m_pImageMap->Lookup((LONG32) ulHandle, pVoid);
                if (bPresent)
                {
                    retVal = HXR_FAIL;
                }
                else
                {
                    // Put the entry into the map
                    m_pImageMap->SetAt((LONG32) ulHandle, (void*) pInfo);
                }
            }
            else
            {
                retVal = HXR_OUTOFMEMORY;
            }

            if (FAILED(retVal))
            {
                HX_DELETE(pInfo);
            }
        }
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

UINT32 PXRealPixFile::GetNumImages() const
{
    UINT32 ulRet = 0;

    if (m_pImageMap)
    {
        ulRet = m_pImageMap->GetCount();
    }

    return ulRet;
}

UINT32 PXRealPixFile::GetNumImagesWithNoSize() const
{
    UINT32 ulRet = 0;

    if (m_pImageMap)
    {
        POSITION pos = m_pImageMap->GetStartPosition();
        while (pos)
        {
            // Get the object pointer at the current position
            // and advance to the next position
            LONG32 lKey     = 0;
            void*  pElement = NULL;
            m_pImageMap->GetNextAssoc(pos, lKey, pElement);

            // Check the PXImageInfo object
            if (pElement)
            {
                PXImageInfo* pInfo = (PXImageInfo*) pElement;
                if (pInfo->m_lErrorStatus != HXR_OK)
                {
                    ulRet++;
                }
            }
        }
    }

    return ulRet;
}

HX_RESULT PXRealPixFile::GetImageIterator(REF(void*) rpIterator)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pImageMap)
    {
        POSITION pos = m_pImageMap->GetStartPosition();
        if (pos)
        {
            rpIterator = pos;
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_NOT_INITIALIZED;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetNextImageHandle(REF(void*) rpIterator, REF(UINT32) rulHandle)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pImageMap)
    {
        if (rpIterator)
        {
            // Get the object pointer at the current position
            // and advance to the next position
            LONG32 lKey     = 0;
            void*  pElement = NULL;
            m_pImageMap->GetNextAssoc(rpIterator, lKey, pElement);

            // The handle *is* the key
            rulHandle = (UINT32) lKey;
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_NOT_INITIALIZED;
    }

    return retVal;
}

BOOL PXRealPixFile::IsImagePresent(UINT32 ulHandle)
{
    BOOL bPresent = FALSE;

    if (m_pImageMap)
    {
        void* pVoid = NULL;
        bPresent    = m_pImageMap->Lookup((LONG32) ulHandle, pVoid);
    }

    return bPresent;
}

HX_RESULT PXRealPixFile::GetImageName(UINT32 ulHandle, REF(IHXBuffer*) rpNameStr)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        if (pInfo->m_pNameStr)
        {
            HX_RELEASE(rpNameStr);
            rpNameStr = pInfo->m_pNameStr;
            rpNameStr->AddRef();
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetImageName(UINT32 ulHandle, REF(const char*) rpszName)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        rpszName = (const char*) (pInfo->m_pNameStr ? pInfo->m_pNameStr->GetBuffer() : NULL);
    }
    return retVal;
}

const char* PXRealPixFile::GetImageName(UINT32 ulHandle)
{
    const char* pszName = NULL;
    GetImageName(ulHandle, pszName);
    return pszName;
}

HX_RESULT PXRealPixFile::SetImageName(UINT32 ulHandle, IHXBuffer* pNameStr)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        HX_RELEASE(pInfo->m_pNameStr);
        retVal = SetString(pNameStr, &pInfo->m_pNameStr);
    }
    return retVal;
}

HX_RESULT PXRealPixFile::SetImageName(UINT32 ulHandle, const char* pszName)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        HX_RELEASE(pInfo->m_pNameStr);
        retVal = SetString(pszName, &pInfo->m_pNameStr);
    }
    return retVal;
}

HX_RESULT PXRealPixFile::GetImageErrorStatus(UINT32 ulHandle, REF(HX_RESULT) rlStatus)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        rlStatus = pInfo->m_lErrorStatus;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::SetImageErrorStatus(UINT32 ulHandle, HX_RESULT lStatus)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        pInfo->m_lErrorStatus = lStatus;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetImageSize(UINT32 ulHandle, REF(UINT32) rulSize)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        rulSize = pInfo->m_ulSize;
    }
    return retVal;
}

UINT32 PXRealPixFile::GetImageSize(UINT32 ulHandle)
{
    UINT32 ulSize = 0;
    GetImageSize(ulHandle, ulSize);
    return ulSize;
}

HX_RESULT PXRealPixFile::SetImageSize(UINT32 ulHandle, UINT32 ulSize)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        if (ulSize)
        {
            pInfo->m_ulSize       = ulSize;
            pInfo->m_lErrorStatus = HXR_OK;
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    return retVal;
}

BOOL PXRealPixFile::IsImageSizeInitialized(UINT32 ulHandle)
{
    BOOL         bRet  = FALSE;
    PXImageInfo* pInfo = NULL;
    GetImageInfo(ulHandle, pInfo);
    if (pInfo && pInfo->m_lErrorStatus == HXR_OK)
    {
        bRet = TRUE;
    }
    return bRet;
}

HX_RESULT PXRealPixFile::GetImageFileMimeType(UINT32 ulHandle, REF(IHXBuffer*) rpMimeStr)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        if (pInfo->m_pFileMimeStr)
        {
            HX_RELEASE(rpMimeStr);
            rpMimeStr = pInfo->m_pFileMimeStr;
            rpMimeStr->AddRef();
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    return retVal;
}

HX_RESULT PXRealPixFile::GetImageFileMimeType(UINT32 ulHandle, REF(const char*) rpszMime)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        rpszMime = (const char*) (pInfo->m_pFileMimeStr ? pInfo->m_pFileMimeStr->GetBuffer() : NULL);
    }
    return retVal;
}

const char* PXRealPixFile::GetImageFileMimeType(UINT32 ulHandle)
{
    const char* pszMime = NULL;
    GetImageFileMimeType(ulHandle, pszMime);
    return pszMime;
}

HX_RESULT PXRealPixFile::SetImageFileMimeType(UINT32 ulHandle, IHXBuffer* pMimeStr)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        HX_RELEASE(pInfo->m_pFileMimeStr);
        retVal = SetString(pMimeStr, &pInfo->m_pFileMimeStr);
    }
    return retVal;
}

HX_RESULT PXRealPixFile::SetImageFileMimeType(UINT32 ulHandle, const char* pszMime)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        HX_RELEASE(pInfo->m_pFileMimeStr);
        retVal = SetString(pszMime, &pInfo->m_pFileMimeStr);
    }
    return retVal;
}

HX_RESULT PXRealPixFile::GetImageStreamMimeType(UINT32 ulHandle, REF(IHXBuffer*) rpMimeStr)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        if (pInfo->m_pStreamMimeStr)
        {
            HX_RELEASE(rpMimeStr);
            rpMimeStr = pInfo->m_pStreamMimeStr;
            rpMimeStr->AddRef();
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    return retVal;
}

HX_RESULT PXRealPixFile::GetImageStreamMimeType(UINT32 ulHandle, REF(const char*) rpszMime)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        rpszMime = (const char*) (pInfo->m_pStreamMimeStr ? pInfo->m_pStreamMimeStr->GetBuffer() : NULL);
    }
    return retVal;
}

const char* PXRealPixFile::GetImageStreamMimeType(UINT32 ulHandle)
{
    const char* pszMime = NULL;
    GetImageStreamMimeType(ulHandle, pszMime);
    return pszMime;
}

HX_RESULT PXRealPixFile::SetImageStreamMimeType(UINT32 ulHandle, IHXBuffer* pMimeStr)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        HX_RELEASE(pInfo->m_pStreamMimeStr);
        retVal = SetString(pMimeStr, &pInfo->m_pStreamMimeStr);
    }
    return retVal;
}

HX_RESULT PXRealPixFile::SetImageStreamMimeType(UINT32 ulHandle, const char* pszMime)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        HX_RELEASE(pInfo->m_pStreamMimeStr);
        retVal = SetString(pszMime, &pInfo->m_pStreamMimeStr);
    }
    return retVal;
}

HX_RESULT PXRealPixFile::GetAllImageInfo(UINT32 ulHandle, REF(UINT32) rulSize, REF(IHXBuffer*) rpNameStr,
                                         REF(IHXBuffer*) rpFileMimeStr, REF(IHXBuffer*) rpStreamMimeStr)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        rulSize        = pInfo->m_ulSize;
        if (pInfo->m_pNameStr)
        {
            HX_RELEASE(rpNameStr);
            rpNameStr = pInfo->m_pNameStr;
            rpNameStr->AddRef();
        }
        if (pInfo->m_pFileMimeStr)
        {
            HX_RELEASE(rpFileMimeStr);
            rpFileMimeStr = pInfo->m_pFileMimeStr;
            rpFileMimeStr->AddRef();
        }
        if (pInfo->m_pStreamMimeStr)
        {
            HX_RELEASE(rpStreamMimeStr);
            rpStreamMimeStr = pInfo->m_pStreamMimeStr;
            rpStreamMimeStr->AddRef();
        }
    }
    return retVal;
}

HX_RESULT PXRealPixFile::GetAllImageInfo(UINT32 ulHandle, REF(UINT32) rulSize, REF(const char*) rpszName,
                                         REF(const char*) rpszFileMime, REF(const char*) rpszStreamMime)
{
    PXImageInfo* pInfo  = NULL;
    HX_RESULT    retVal = GetImageInfo(ulHandle, pInfo);
    if (SUCCEEDED(retVal))
    {
        rulSize        = pInfo->m_ulSize;
        rpszName       = (const char*) (pInfo->m_pNameStr ? pInfo->m_pNameStr->GetBuffer() : NULL);
        rpszFileMime   = (const char*) (pInfo->m_pFileMimeStr ? pInfo->m_pFileMimeStr->GetBuffer() : NULL);
        rpszStreamMime = (const char*) (pInfo->m_pStreamMimeStr ? pInfo->m_pStreamMimeStr->GetBuffer() : NULL);
    }
    return retVal;
}

BOOL PXRealPixFile::AllImageSizesInitialized()
{
    BOOL bRet = TRUE;

    if (m_pImageMap)
    {
        // Run through the map, deleting all image objects
        POSITION pos = m_pImageMap->GetStartPosition();
        while (pos)
        {
            // Get the object pointer at the current position
            // and advance to the next position
            LONG32 lKey     = 0;
            void*  pElement = NULL;
            m_pImageMap->GetNextAssoc(pos, lKey, pElement);

            // Check the PXImageInfo object
            if (pElement)
            {
                PXImageInfo* pInfo = (PXImageInfo*) pElement;
                if (pInfo->m_lErrorStatus == HXR_NOT_INITIALIZED)
                {
                    bRet = FALSE;
                    break;
                }
            }
        }
    }

    return bRet;
}

BOOL PXRealPixFile::AllImageSizesOK()
{
    BOOL bRet = TRUE;

    if (m_pImageMap)
    {
        // Run through the map, deleting all image objects
        POSITION pos = m_pImageMap->GetStartPosition();
        while (pos)
        {
            // Get the object pointer at the current position
            // and advance to the next position
            LONG32 lKey     = 0;
            void*  pElement = NULL;
            m_pImageMap->GetNextAssoc(pos, lKey, pElement);

            // Check the PXImageInfo object
            if (pElement)
            {
                PXImageInfo* pInfo = (PXImageInfo*) pElement;
                if (pInfo->m_lErrorStatus != HXR_OK)
                {
                    bRet = FALSE;
                    break;
                }
            }
        }
    }

    return bRet;
}

HX_RESULT PXRealPixFile::GetFailedImageName(REF(IHXBuffer*) rpFailedNameStr)
{
    HX_RESULT retVal = HXR_FAIL;

    if (m_pImageMap)
    {
        // Run through the map, deleting all image objects
        POSITION pos = m_pImageMap->GetStartPosition();
        while (pos)
        {
            // Get the object pointer at the current position
            // and advance to the next position
            LONG32 lKey     = 0;
            void*  pElement = NULL;
            m_pImageMap->GetNextAssoc(pos, lKey, pElement);

            // Check the PXImageInfo object
            if (pElement)
            {
                PXImageInfo* pInfo = (PXImageInfo*) pElement;
                if (pInfo->m_lErrorStatus != HXR_OK)
                {
                    if (pInfo->m_pNameStr)
                    {
                        HX_RELEASE(rpFailedNameStr);
                        rpFailedNameStr = pInfo->m_pNameStr;
                        rpFailedNameStr->AddRef();
                        retVal = HXR_OK;
                        break;
                    }
                }
            }
        }
    }

    return retVal;
}

HX_RESULT PXRealPixFile::AddEffect(PXEffect* pEffect)
{
    HX_RESULT retVal = HXR_OK;

    if (pEffect)
    {
        if (!m_pEffectsList)
        {
            m_pEffectsList = new CHXSimpleList();
            if (!m_pEffectsList)
            {
                retVal = HXR_OUTOFMEMORY;
            }
        }

        if (SUCCEEDED(retVal))
        {
            // Check for a few shortcuts before searching the whole list
            // If there aren't any elements in the queue, we know just to
            // add ourselves to the end of the list.
            if (m_pEffectsList->GetCount() > 0)
            {
                // Now we check the last element in the queue. If our start time is
                // greater than or equal to its start time, then we know just to
                // add ourselves at the tail of the queue.
                PXEffect *pTailEffect = (PXEffect *) m_pEffectsList->GetTail();
                if (pTailEffect)
                {
                    if (pEffect->GetStart() < pTailEffect->GetStart())
                    {
                        // Now we have to search the list.
                        // Insert effect into list which is sorted by increasing start time
                        // We look for the first effect which has a start time greater
                        // than the one we're inserting
                        LISTPOSITION pos = m_pEffectsList->GetHeadPosition();
                        while (pos)
                        {
                            PXEffect *pListEffect = (PXEffect *) m_pEffectsList->GetAt(pos);
                            if (pListEffect)
                            {
                                if (pListEffect->GetStart() > pEffect->GetStart())
                                {
                                    // Addref effect before insertion
                                    pEffect->AddRef();
                                    // Insert into the list
                                    m_pEffectsList->InsertBefore(pos, (void *) pEffect);
                                    break;
                                }
                                m_pEffectsList->GetNext(pos);
                            }
                            else
                            {
                                retVal = HXR_FAIL;
                                break;
                            }
                        }

                        // If we added ourselves in the above loop, then pos will be non-NULL.
                        // If we DIDN'T add ourselves in the above loop, then pos will be NULL.
                        if (!pos)
                        {
                            // We didn't find an effect with a start time greater, so
                            // this effect must be the new largest start time, so
                            // we add this effect to the tail of the list. 
                            //
                            pEffect->AddRef();
                            m_pEffectsList->AddTail((void *) pEffect);
                        }
                    }
                    else
                    {
                        // Addref before adding to the list
                        pEffect->AddRef();
                        // Add it to the list
                        m_pEffectsList->AddTail((void *) pEffect);
                    }
                }
                else
                {
                    retVal = HXR_FAIL;
                }
            }
            else
            {
                // Addref the effect object before adding to the list
                pEffect->AddRef();
                // Add it to the list
                m_pEffectsList->AddTail((void *) pEffect);
            }
        }
    }
    else
    {
        retVal = HXR_INVALID_PARAMETER;
    }

    return retVal;
}

UINT32 PXRealPixFile::GetNumEffects() const
{
    UINT32 ulRet = 0;

    if (m_pEffectsList)
    {
        ulRet = m_pEffectsList->GetCount();
    }

    return ulRet;
}

HX_RESULT PXRealPixFile::GetEffectHeadIterator(REF(void*) rpIterator)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pEffectsList)
    {
        LISTPOSITION pos = m_pEffectsList->GetHeadPosition();
        if (pos)
        {
            rpIterator = pos;
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetEffectTailIterator(REF(void*) rpIterator)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pEffectsList)
    {
        LISTPOSITION pos = m_pEffectsList->GetTailPosition();
        if (pos)
        {
            rpIterator = pos;
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetAtNextEffect(REF(void*) rpIterator, REF(PXEffect*) rpEffect)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pEffectsList)
    {
        if (rpIterator)
        {
            rpEffect = (PXEffect*) m_pEffectsList->GetAtNext(rpIterator);
            if (rpEffect)
            {
                rpEffect->AddRef();
            }
            else
            {
                retVal = HXR_FAIL;
            }
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_NOT_INITIALIZED;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetNextEffect(REF(void*) rpIterator, REF(PXEffect*) rpEffect)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pEffectsList)
    {
        if (rpIterator)
        {
            rpEffect = (PXEffect*) m_pEffectsList->GetNext(rpIterator);
            if (rpEffect)
            {
                rpEffect->AddRef();
            }
            else
            {
                retVal = HXR_FAIL;
            }
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_NOT_INITIALIZED;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetPrevEffect(REF(void*) rpIterator, REF(PXEffect*) rpEffect)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pEffectsList)
    {
        if (rpIterator)
        {
            rpEffect = (PXEffect*) m_pEffectsList->GetPrev(rpIterator);
            if (rpEffect)
            {
                rpEffect->AddRef();
            }
            else
            {
                retVal = HXR_FAIL;
            }
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_NOT_INITIALIZED;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetCurrentEffect(void* pIterator, REF(PXEffect*) rpEffect)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pEffectsList)
    {
        if (pIterator)
        {
            rpEffect = (PXEffect*) m_pEffectsList->GetAt(pIterator);
            if (rpEffect)
            {
                rpEffect->AddRef();
            }
            else
            {
                retVal = HXR_FAIL;
            }
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_NOT_INITIALIZED;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::GetImageInfo(UINT32 ulHandle, REF(PXImageInfo*) rpInfo)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pImageMap)
    {
        void *pVoid   = NULL;
        BOOL bPresent = m_pImageMap->Lookup((LONG32) ulHandle, pVoid);
        if (bPresent && pVoid)
        {
            rpInfo = (PXImageInfo*) pVoid;
        }
        else
        {
            retVal = HXR_FAIL;
        }
    }
    else
    {
        retVal = HXR_NOT_INITIALIZED;
    }

    return retVal;
}

HX_RESULT PXRealPixFile::PostParseInit()
{
    HX_RESULT retVal = HXR_OK;

    // First we need to set the first and last use effect flags.
    // To set the first use flags, we initialize a map and
    // run forward through the list of effects. Whenever we see
    // an image used, we check to see if there's an entry in the
    // map already. If not, then it's the first time this image
    // has been used, so we mark the first use flag as true
    // and then put an entry in the map. For last use flags,
    // we do the same thing, but run through the effects in reverse.
    //
    // First, create the map
    CHXMapLongToObj* pMap = new CHXMapLongToObj();
    if (pMap)
    {
        // Now do a forward run through the effects
        PXEffect*    pEffect = NULL;
        LISTPOSITION pos     = m_pEffectsList->GetHeadPosition();
        while (pos)
        {
            pEffect = (PXEffect*) m_pEffectsList->GetNext(pos);
            if (pEffect)
            {
                if (pEffect->HasTarget())
                {
                    // Is there already an entry in the map for this image?
                    void* pVoid    = NULL;
                    BOOL  bPresent = pMap->Lookup((LONG32) pEffect->GetTarget(), pVoid);
                    if (bPresent)
                    {
                        // This image has already been used, so clear the FirstUse flag
                        pEffect->SetFirstUse(FALSE);
                    }
                    else
                    {
                        // This image has NOT been used so far, so set the FirstUse flag
                        pEffect->SetFirstUse(TRUE);
                        // And also put an entry in the map for this image. It
                        // doesn't matter what the entry is, just that there IS
                        // an entry in the map
                        pMap->SetAt((LONG32) pEffect->GetTarget(), (void*) 1);
                    }
                }
            }
        }
        // Now clear the map to get ready for a backwards run
        pMap->RemoveAll();
        // Now run through the effects in reverse, doing the same thing
        pos = m_pEffectsList->GetTailPosition();
        while (pos)
        {
            PXEffect* pEffect = (PXEffect*) m_pEffectsList->GetPrev(pos);
            if (pEffect)
            {
                if (pEffect->HasTarget())
                {
                    // Is there already an entry in the map for this image?
                    void* pVoid    = NULL;
                    BOOL  bPresent = pMap->Lookup((LONG32) pEffect->GetTarget(), pVoid);
                    if (bPresent)
                    {
                        // This image has already been used, so clear the LastUse flag
                        pEffect->SetLastUse(FALSE);
                    }
                    else
                    {
                        // This image has NOT been used so far, so set the LastUse flag
                        pEffect->SetLastUse(TRUE);
                        // And also put an entry in the map for this image. It
                        // doesn't matter what the entry is, just that there IS
                        // an entry in the map
                        pMap->SetAt((LONG32) pEffect->GetTarget(), (void*) 1);
                    }
                }
            }
        }
    }
    else
    {
        retVal = HXR_OUTOFMEMORY;
    }
    HX_DELETE(pMap);

    // Now we need to check if we need to compute the duration. If the duration is
    // still zero at this point, then it was never set in parsing. Therefore,
    // we need to calculate it from the effects
    if (!m_ulDuration)
    {
        // Run through the effects list, find the max end time.
        UINT32       ulMaxTime = 0;
        LISTPOSITION pos       = m_pEffectsList->GetHeadPosition();
        while (pos)
        {
            PXEffect* pEffect = (PXEffect*) m_pEffectsList->GetNext(pos);
            if (pEffect)
            {
                if (pEffect->GetEnd() > ulMaxTime)
                {
                    ulMaxTime = pEffect->GetEnd();
                }
            }
        }
        // We don't let the duration be exactly zero.
        if (!ulMaxTime)
        {
            ulMaxTime = kMinDuration;
        }
        // Now set the duration of the presentation to this duration
        m_ulDuration = ulMaxTime;
    }

    return retVal;
}
