/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the IIWU program. 
 * IIWU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */

/* iiwu_dsound.c
 *
 * author Peter Hanappe 
 */

#define INITGUID

#include "iiwusynth_priv.h"
#include "iiwu_auport.h"
#include <windows.h>
#include <mmsystem.h> 
#include <dsound.h>

static HWND iiwu_wnd = NULL;
static LPDIRECTSOUND iiwu_direct_sound = NULL;
static LPDIRECTSOUNDBUFFER iiwu_primary_buffer = NULL;

iiwu_audio_driver_t* new_iiwu_dsound_audio_driver(iiwu_auport_t* port);
int delete_iiwu_dsound_audio_driver(iiwu_audio_driver_t* data);
DWORD WINAPI iiwu_dsound_audio_run(LPVOID lpParameter);

int iiwu_open_direct_sound(char *device);
int iiwu_close_direct_sound();
int iiwu_win32_create_window();
int iiwu_win32_destroy_window();
long FAR PASCAL iiwu_win32_wndproc(HWND hWnd, unsigned message, WPARAM wParam, LPARAM lParam);
char* iiwu_win32_error(HRESULT hr);


#define IIWU_HINSTANCE  ((HINSTANCE)iiwu_get_hinstance())

typedef struct {
  LPDIRECTSOUNDBUFFER dsBuffer;
  LPDIRECTSOUNDNOTIFY notify;
  WAVEFORMATEX* format;
  DSBUFFERDESC desc;
  HANDLE thread;
  DWORD threadID;
  DSBPOSITIONNOTIFY* position;
  HANDLE* event;
  int event_count;
  iiwu_auport_t* auport;
  int cont;
  int buffers;
  int buffer_size;
  int buffer_byte_size;
  int cur_buffer;
} iiwu_dsound_audio_driver_t;


/*
 * new_iiwu_dsound_audio_driver
 */
iiwu_audio_driver_t* new_iiwu_dsound_audio_driver(iiwu_auport_t* port)
{
  int err, i;
  HRESULT hr;
  iiwu_dsound_audio_driver_t* dev = NULL;
  iiwu_pcm_data_t* dev_format;

  /* open DirectSound */
  if (iiwu_direct_sound == NULL) {
    err = iiwu_open_direct_sound(iiwu_auport_get_device_name(port));
    if (err != 0) {
      IIWU_LOG(PANIC, "Can't open direct sound (error %i)", err);
      return NULL;
    }
  }

  /* create and clear the driver data */
  dev = IIWU_NEW(iiwu_dsound_audio_driver_t);
  if (dev == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;    
  }
  IIWU_MEMSET(dev, 0, sizeof(iiwu_dsound_audio_driver_t));

  dev->auport = port;
  dev->cont = 1;

  dev_format = iiwu_auport_get_dev_format(port);
  dev->buffer_size = iiwu_auport_get_buffer_size(port);
  dev->buffer_byte_size = dev->buffer_size * iiwu_pcm_data_framesize(dev_format);
  dev->buffers = iiwu_auport_get_queue_size(port) / dev->buffer_size;
  dev->event_count = dev->buffers;
  dev->cur_buffer = 0;

  /* create and initialize the buffer format */
  dev->format = (WAVEFORMATEX*) IIWU_MALLOC(sizeof(WAVEFORMATEX));
  if (dev->format == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    goto error_recovery;
  }
  ZeroMemory(dev->format, sizeof(WAVEFORMATEX));
  dev->format->wFormatTag = WAVE_FORMAT_PCM;
  dev->format->nChannels = (unsigned short) iiwu_pcm_data_get_channels(dev_format);
  dev->format->nSamplesPerSec = (DWORD) iiwu_pcm_data_get_sample_rate(dev_format);
  dev->format->nBlockAlign = iiwu_pcm_data_framesize(dev_format);
  dev->format->nAvgBytesPerSec = dev->format->nSamplesPerSec * dev->format->nBlockAlign;
  dev->format->wBitsPerSample = iiwu_pcm_data_get_bps(dev_format); 
  dev->format->cbSize = 0; 

  /* initialize the buffer description */
  ZeroMemory(&dev->desc, sizeof(DSBUFFERDESC));
  dev->desc.dwSize = sizeof(DSBUFFERDESC);
  dev->desc.dwFlags = DSBCAPS_GLOBALFOCUS | DSBCAPS_GETCURRENTPOSITION2 | DSBCAPS_CTRLPOSITIONNOTIFY;
  dev->desc.lpwfxFormat = dev->format;
  dev->desc.dwBufferBytes = dev->buffer_byte_size * dev->event_count;  

  /* create the sound buffer */
  hr = IDirectSound_CreateSoundBuffer(iiwu_direct_sound, &dev->desc, &dev->dsBuffer, NULL);
  if (hr != DS_OK) {
    IIWU_LOG(ERR, "dsound: Can't create sound buffer: %s", iiwu_win32_error(hr));
    goto error_recovery;
  }

  /* allocate an array for the notification events */
  dev->event = (HANDLE*) IIWU_MALLOC(dev->event_count * sizeof(HANDLE));
  if (dev->event == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    goto error_recovery;
  }
  ZeroMemory(dev->event, dev->event_count * sizeof(HANDLE));

  /* allocate an array for the notification positions */
  dev->position = (DSBPOSITIONNOTIFY*) IIWU_MALLOC(dev->event_count * sizeof(DSBPOSITIONNOTIFY));
  if (dev->position == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    goto error_recovery;
  }

  /* create and initialize the notification events and positions */
  ZeroMemory(dev->position, dev->event_count * sizeof(DSBPOSITIONNOTIFY));
  for (i = 0; i < dev->event_count; i++) {
    dev->event[i] = CreateEvent(NULL, FALSE, FALSE, NULL);
    if (NULL == dev->event[i]) {
      IIWU_LOG(ERR, "dsound: Can't create notify events");
      goto error_recovery;
    }
    dev->position[i].dwOffset = i * dev->buffer_byte_size;
    dev->position[i].hEventNotify = dev->event[i];
  }

  /* set the notification position for the synchronisation with the audio device */
  hr = IDirectSoundBuffer_QueryInterface(dev->dsBuffer, &IID_IDirectSoundNotify, (LPVOID*) &dev->notify);
  if (hr != S_OK) {
    dev->notify = NULL;
    IIWU_LOG(ERR, "dsound: Can't get notify interface: %s", iiwu_win32_error(hr));
    goto error_recovery;
  }
  hr = IDirectSoundNotify_SetNotificationPositions(dev->notify, dev->event_count, dev->position);
  if (hr != S_OK) {
    IIWU_LOG(ERR, "dsound: Can't set notify positions: %s", iiwu_win32_error(hr));
    goto error_recovery;
  }
  
  /* start the audio thread */
  dev->thread = CreateThread(NULL, 0, &iiwu_dsound_audio_run, (LPVOID) dev, 0, &dev->threadID);
  if (dev->thread == NULL) {
    goto error_recovery;
  }

  return (iiwu_audio_driver_t*) dev;

 error_recovery:
  delete_iiwu_dsound_audio_driver((iiwu_audio_driver_t*) dev);
  return NULL;
}


int delete_iiwu_dsound_audio_driver(iiwu_audio_driver_t* d)
{
  iiwu_dsound_audio_driver_t* dev = (iiwu_dsound_audio_driver_t*) d;

  if (dev == NULL) {
    return IIWU_OK;
  }

  dev->cont = 0;
  if (dev->dsBuffer != NULL) {
    IDirectSoundBuffer_Stop(dev->dsBuffer);
  }

  /* join the audio thread */
  if (dev->thread != 0) {
    if (WaitForSingleObject(dev->thread, 2000) != WAIT_OBJECT_0) { 
      /* on error kill the thread mercilessly */
      IIWU_LOG(DBG, "Couldn't join the audio thread. killing it.");
      TerminateThread(dev->thread, 0);
    }
  }

  if (dev->event != NULL) {
    /* FIXME event[i] ??? */
    IIWU_FREE(dev->event);
  }
  if (dev->position != NULL) {
    IIWU_FREE(dev->position);
  }
  if (dev->format != NULL) {
    IIWU_FREE(dev->format);
  }
  if (dev->notify != NULL) {
    IDirectSoundNotify_Release(dev->notify);
  }
  if (dev->dsBuffer != NULL) {
    IDirectSoundBuffer_Release(dev->dsBuffer);
  }

  IIWU_FREE(dev);

  return 0;
}


DWORD WINAPI iiwu_dsound_audio_run(LPVOID lpParameter)
{
  iiwu_dsound_audio_driver_t* dev = (iiwu_dsound_audio_driver_t*) lpParameter;
  iiwu_auport_t* port = dev->auport;
  int len = dev->buffer_size;
  void *buf1, *buf2;
  DWORD bytes1, bytes2;    
  int offset;

  /* boost the priority of the audio thread */
  SetPriorityClass(GetCurrentProcess(), REALTIME_PRIORITY_CLASS);
  SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_TIME_CRITICAL);

  IDirectSoundBuffer_Play(dev->dsBuffer, 0, 0, DSBPLAY_LOOPING);

  while ((iiwu_auport_get_state(port) == IIWU_AUPORT_PLAYING) && dev->cont) {

    WaitForSingleObject(dev->event[dev->cur_buffer], INFINITE);

    if (dev->cur_buffer == 0) {
      offset = (dev->buffers - 1) * dev->buffer_byte_size;    
    } else {
      offset = (dev->cur_buffer - 1) * dev->buffer_byte_size;    
    }
    IDirectSoundBuffer_Lock(dev->dsBuffer, offset, dev->buffer_byte_size, &buf1, &bytes1, &buf2, &bytes2, 0);

    iiwu_auport_write(port, buf1, len);

    IDirectSoundBuffer_Unlock(dev->dsBuffer, buf1, bytes1, buf2, bytes2);

    dev->cur_buffer++;
    if (dev->cur_buffer == dev->buffers) {
      dev->cur_buffer = 0;
    }
  }

  ExitThread(0);
  return 0; /* never reached */
}


char* iiwu_win32_error(HRESULT hr) {
  char *s = "Don't know why";
  switch (hr) {
  case E_NOINTERFACE: s = "No such interface"; break;
  case DSERR_GENERIC: s = "Generic error"; break;
  case DSERR_ALLOCATED: s = "Required resources already allocated"; break;
  case DSERR_BADFORMAT: s = "The format is not supported"; break;
  case DSERR_INVALIDPARAM: s = "Invalid parameter"; break;
  case DSERR_NOAGGREGATION: s = "No aggregation"; break;
  case DSERR_OUTOFMEMORY: s = "Out of memory"; break;
  case DSERR_UNINITIALIZED: s = "Uninitialized"; break;
  case DSERR_UNSUPPORTED: s = "Function not supported"; break;
  }
  return s;
}

int iiwu_open_direct_sound(char *device) 
{
  HRESULT hr;
  DSBUFFERDESC desc;
  int err = 0;

  if (IIWU_HINSTANCE == NULL) {
    return -1;
  }

  if (iiwu_wnd == NULL) {
    err = iiwu_win32_create_window();
    if (err != 0) return err;
  }

  hr = DirectSoundCreate(NULL, &iiwu_direct_sound, NULL);
  if (hr != DS_OK) {
    iiwu_direct_sound = NULL;
    return -2;
  }

  hr = IDirectSound_SetCooperativeLevel(iiwu_direct_sound, iiwu_wnd, DSSCL_PRIORITY);
  if (hr != DS_OK) {
    IDirectSound_Release(iiwu_direct_sound); 
    iiwu_direct_sound = NULL;
    return -3;
  }
  ZeroMemory(&desc, sizeof(DSBUFFERDESC));

  desc.dwSize = sizeof(DSBUFFERDESC);
  desc.dwFlags = DSBCAPS_PRIMARYBUFFER;

  hr = IDirectSound_CreateSoundBuffer(iiwu_direct_sound, &desc, &iiwu_primary_buffer, NULL);
  if (hr != DS_OK) {
    IDirectSound_Release(iiwu_direct_sound); 
    iiwu_direct_sound = NULL;
    iiwu_primary_buffer = NULL;
    return -4;
  }
  return 0;  
}


int iiwu_close_direct_sound() 
{
  IDirectSoundBuffer_Release(iiwu_primary_buffer); 
  iiwu_primary_buffer = NULL;
  IDirectSound_Release(iiwu_direct_sound); 
  iiwu_direct_sound = NULL;
  return 0;
}

long FAR PASCAL iiwu_win32_wndproc(HWND hWnd, unsigned message, WPARAM wParam, LPARAM lParam)
{
  switch (message) {
  case WM_CREATE:
    break;
  case WM_DESTROY:
    break;
  default:
    return DefWindowProc(hWnd, message, wParam, lParam);
    break;
  } 
  return(0L);
}

int iiwu_win32_create_window() 
{
  WNDCLASS myClass;
  myClass.hCursor = LoadCursor( NULL, IDC_ARROW );
  myClass.hIcon = NULL; 
  myClass.lpszMenuName = (LPSTR) NULL;
  myClass.lpszClassName = (LPSTR) "IIWUSynth";
  myClass.hbrBackground = (HBRUSH)(COLOR_WINDOW);
  myClass.hInstance = IIWU_HINSTANCE;
  myClass.style = CS_GLOBALCLASS;
  myClass.lpfnWndProc = iiwu_win32_wndproc;
  myClass.cbClsExtra = 0;
  myClass.cbWndExtra = 0;
  if (!RegisterClass(&myClass)) {
    return -100;
  }
  iiwu_wnd = CreateWindow((LPSTR) "IIWUSynth", (LPSTR) "IIWUSynth", WS_OVERLAPPEDWINDOW,
			CW_USEDEFAULT, CW_USEDEFAULT, 400, 300, (HWND) NULL, (HMENU) NULL, 
			IIWU_HINSTANCE, (LPSTR) NULL);  
  if (iiwu_wnd == NULL) {
    IIWU_LOG(ERR, "Can't create window");
    return -101;
  }
  return 0;
}

int iiwu_win32_destroy_window() 
{
  return 0;
}
