/***************************************************************************
                          kbeardirsynchpart.cpp  -  description
                             -------------------
    begin                : mn okt 14 2002
    copyright            : (C) 2002 by Bjrn Sahlstrm
    email                : kbjorn@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

//////////////////////////////////////////////////////////////////////
// Qt specific include files
#include <qlayout.h>
#include <qptrlist.h>
#include <qsizepolicy.h>
#include <qfile.h>
#include <qlabel.h>
#include <qfont.h>
#include <qpoint.h>
#include <qtooltip.h>
#include <qheader.h>
#include <qregexp.h>
#include <qwhatsthis.h>
//////////////////////////////////////////////////////////////////////
// KDE specific include files
#include <kiconloader.h>
#include <kparts/genericfactory.h>
#include <kglobal.h>
#include <kdialogbase.h>
#include <kinstance.h>
#include <klocale.h>
#include <kconfig.h>
#include <kaction.h>
#include <kfiletreeview.h>
#include <kmessagebox.h>
#include <klistview.h>
#include <kapplication.h>
#include <krun.h>
//////////////////////////////////////////////////////////////////////
// system include files
#include <time.h>
//////////////////////////////////////////////////////////////////////
// Application specific include files
#include "kbeardirsynchpart.h"
#include "dirsynchtreebranch.h"
#include "dirsynchtreeview.h"
#include "dirsynchconfigdialog.h"
#include "../../base/kbearconnectionmanager.h"
#include "../../base/kbearcopyjob.h"
#include "../../base/kbeardeletejob.h"

//-----------------------------------------------
typedef KParts::GenericFactory<KBearDirSynchPart> KBearDirSynchPartFactory;
K_EXPORT_COMPONENT_FACTORY( libkbeardirsynchpart, KBearDirSynchPartFactory );
//-----------------------------------------------
KBearDirSynchPart::KBearDirSynchPart( QWidget* parentWidget, const char* widgetName,
						QObject* parent, const char* name, const QStringList& )
	:	KBearPart( parentWidget, widgetName, parent, name ),
		m_localView( 0L ), m_remoteView( 0L ),
		m_localBranch( 0L ), m_remoteBranch( 0L ),
		m_localBranchFinished( false ), m_remoteBranchFinished( false ),
		m_isWorking( false ), m_copyTransfer( 0L ),
		m_diffColor( 237, 190, 190 ), m_missingRemoteColor( 190, 237, 190 ),
		m_missingLocalColor( 190, 190, 237 ), m_diffRule( SizePermissions ),
		m_warnDelete( true ), m_timeDiff( 0 )

{
	setInstance( KBearDirSynchPartFactory::instance() );
	KGlobal::locale()->insertCatalogue("kbear");

	m_touchList.setAutoDelete( false );
	setupActions();
	reparseConfiguration();
	setupWidget();
	setActionsEnabled( false );
	setXMLFile("kbeardirsynchpartui.rc");
	connect( m_doneButton, SIGNAL( clicked() ), this, SIGNAL( closeMe() ) );
}
//-----------------------------------------------
KBearDirSynchPart::~KBearDirSynchPart(){
	if( m_isWorking )
		m_dirLister->stop();
}
//-----------------------------------------------
bool KBearDirSynchPart::openURL( const KURL& ) {
	return true;
}
//-----------------------------------------------
void KBearDirSynchPart::openConnection( const Connection& c ) {
	setActionsEnabled( false );
	m_connection = c;
	emit setStatusBarText( i18n("Reading...") );
	m_localBranchFinished = false;
	m_remoteBranchFinished = false;
	m_isWorking = false;

	QApplication::setOverrideCursor( waitCursor );
	emit started( 0L );

	reparseConfiguration();

	// remote
	openRemoteBranch();
}
//-----------------------------------------------
void KBearDirSynchPart::openLocalBranch() {
	if( m_localBranch ) {
		m_localView->removeBranch( m_localBranch );
		m_localBranch = 0L;
	}
	m_localBranchFinished = false;

	KURL localURL = m_connection.localPath();
	localURL.adjustPath( -1 );
	m_localDirLbl->setText( localURL.path() );
	m_localBranch = static_cast<DirSynchTreeBranch*>( m_localView->addBranch( localURL, localURL.path() ) );
	m_localBranch->setColors( m_diffColor, m_missingRemoteColor, m_missingLocalColor );
	connect( m_localBranch, SIGNAL( finishedLoading( KFileTreeBranch* ) ),
				this, SLOT( slotFinishedLoading( KFileTreeBranch* ) ) );
	m_localBranch->setChildRecurse( false );
	m_localBranch->setOpen();
}
//-----------------------------------------------
void KBearDirSynchPart::openRemoteBranch() {
	if( m_remoteBranch ) {
		m_remoteView->removeBranch( m_remoteBranch );
		m_remoteBranch = 0L;
	}
	m_remoteBranchFinished = false;
	KURL remoteURL = m_connection.url();
	remoteURL.adjustPath( -1 );
	m_remoteDirLbl->setText( remoteURL.path() );
	m_remoteBranch = static_cast<DirSynchTreeBranch*>( m_remoteView->addBranch( remoteURL, remoteURL.path() ) );
	m_remoteBranch->setTimeDiff( m_timeDiff );
	m_remoteBranch->setColors( m_diffColor, m_missingRemoteColor, m_missingLocalColor );
	connect( m_remoteBranch, SIGNAL( finishedLoading( KFileTreeBranch* ) ),
				this, SLOT( slotFinishedLoading( KFileTreeBranch* ) ) );
	connect( m_dirLister, SIGNAL( started() ), this, SLOT( slotStartLoading() ) );
	m_remoteBranch->setChildRecurse( false );
	m_remoteBranch->setDirLister( m_dirLister );
	m_remoteBranch->setOpen();
}
//-----------------------------------------------
KAboutData* KBearDirSynchPart::createAboutData() {
	KAboutData* about = new KAboutData( "kbeardirsynchpart", I18N_NOOP("KBearDirSynchPart"), "1.0", 0,
							KAboutData::License_GPL, I18N_NOOP("(C) 2002, The KBear team") );
	about->addAuthor("Björn Sahlström", 0, "kbjorn@users.sourceforge.net");

	return about;
}
//-----------------------------------------------
void KBearDirSynchPart::slotStartLoading() {
	m_isWorking = true;
}
//-----------------------------------------------
void KBearDirSynchPart::slotFinishedLoading( KFileTreeBranch* branch ) {
	if( branch == m_localBranch ) {
		m_localBranchFinished = true;
		disconnect( m_localBranch, SIGNAL( finishedLoading( KFileTreeBranch* ) ),
				this, SLOT( slotFinishedLoading( KFileTreeBranch* ) ) );

	}
	else if( branch == m_remoteBranch ) {
		disconnect( m_remoteBranch, SIGNAL( finishedLoading( KFileTreeBranch* ) ),
				this, SLOT( slotFinishedLoading( KFileTreeBranch* ) ) );
		m_isWorking = false;
		m_remoteBranchFinished = true;
		if( ! m_localBranchFinished )
			openLocalBranch();
	}
	if( m_localBranchFinished && m_remoteBranchFinished ) {
		// we now need to find out the difference between the two views
		checkDiff();
		setActionsEnabled( true );
		QApplication::restoreOverrideCursor();
		emit setStatusBarText( i18n("Ready.") );
 		emit completed();
	}
}
//-----------------------------------------------
DirSynchTreeViewItem* KBearDirSynchPart::findCorrespondingItemAbove( DirSynchTreeViewItem* item ) {
	DirSynchTreeViewItem* itemAbove = static_cast<DirSynchTreeViewItem*>( item->itemAbove() );
	if( ! itemAbove )
		return 0L;

	DirSynchTreeViewItem* corrItemAbove = findCorrespondingItem( itemAbove );
	if( ! corrItemAbove )
		 return findCorrespondingItemAbove( itemAbove );

	return corrItemAbove;
}
//-----------------------------------------------
DirSynchTreeViewItem* KBearDirSynchPart::findCorrespondingItemBelow( DirSynchTreeViewItem* item ) {
	DirSynchTreeViewItem* itemBelow = static_cast<DirSynchTreeViewItem*>( item->itemBelow() );
	if( ! itemBelow )
		return 0L;

	DirSynchTreeViewItem* corrItemBelow = findCorrespondingItem( itemBelow );
	if( ! corrItemBelow )
		 return findCorrespondingItemBelow( itemBelow );

	return corrItemBelow;
}
//-----------------------------------------------
DirSynchTreeViewItem* KBearDirSynchPart::findCorrespondingItem( DirSynchTreeViewItem* item ) {
	if( ! item || (m_localView->childCount() == 0) || (m_remoteView->childCount() == 0) )
		return 0L;
	DirSynchTreeBranch* branch = 0L;
	QString rootURL;
	if( item->listView() == m_localView ) {
		rootURL = m_localBranch->rootUrl().url(+1);
		branch = m_remoteBranch;
	}
	else if( item->listView() == m_remoteView ) {
		rootURL = m_remoteBranch->rootUrl().url(+1);
		branch = m_localBranch;
	}
	QString relURL = item->url().url(-1);
	relURL = relURL.remove( 0, rootURL.length() );
	KURL url( branch->rootUrl().url(+1) + relURL );
//	kdDebug()<<"KBearDirSynchPart::findCorrespondingItem rootURL="<<rootURL<<endl;
//	kdDebug()<<"KBearDirSynchPart::findCorrespondingItem relURL="<<relURL<<endl;
//	kdDebug()<<"KBearDirSynchPart::findCorrespondingItem url="<<url.prettyURL()<<endl;
	return dynamic_cast<DirSynchTreeViewItem*>( branch->findTVIByURL( url ) );
}
//-----------------------------------------------
void KBearDirSynchPart::setDiff( DirSynchTreeViewItem* currentItem, DirSynchTreeViewItem* otherItem ) {
	if( ! currentItem ) {
     	return;
	}

	if( ! otherItem ) { // the corresponding item is missing
		// so we nee to the item above and below instead
		otherItem = findCorrespondingItemAbove( currentItem );
		if( otherItem )
			otherItem->setMissingBelow( true );
		otherItem = findCorrespondingItemBelow( currentItem );
		if( otherItem )
			otherItem->setMissingAbove( true );

		currentItem->setCorrespondingMissing( true );
		return;
	}

	if( m_diffRule == Modified ) {
//		time_t currentTime = currentItem->/*fileItem()->*/time( KIO::UDS_MODIFICATION_TIME );
//		time_t otherTime = otherItem->/*fileItem()->*/time( KIO::UDS_MODIFICATION_TIME );

/*
		kdDebug()<<"KBearDirSynchPart::checkDiff() localTime created="<<localItem->timeString( KIO::UDS_CREATION_TIME )<<endl;
		kdDebug()<<"KBearDirSynchPart::checkDiff() localTime modified="<<localItem->timeString( KIO::UDS_MODIFICATION_TIME )<<endl;
		kdDebug()<<"KBearDirSynchPart::checkDiff() localTime access="<<localItem->timeString( KIO::UDS_ACCESS_TIME )<<endl;
		kdDebug()<<"KBearDirSynchPart::checkDiff() remoteTime created="<<remoteItem->timeString( KIO::UDS_CREATION_TIME )<<endl;
		kdDebug()<<"KBearDirSynchPart::checkDiff() remoteTime modified="<<remoteItem->timeString( KIO::UDS_MODIFICATION_TIME )<<endl;
		kdDebug()<<"KBearDirSynchPart::checkDiff() remoteTime access="<<remoteItem->timeString( KIO::UDS_ACCESS_TIME )<<endl;
		kdDebug()<<""<<endl;
*/
//		if( difftime( currentTime, otherTime ) != double( 0 ) ) {
		if( currentItem->timeString() !=  otherItem->timeString() ) {
			currentItem->setIsDifferent( true );
			otherItem->setIsDifferent( true );
			return;
		}
	}
	else {
		if( currentItem->fileItem()->size() != otherItem->fileItem()->size() ) {
			currentItem->setIsDifferent( true );
			otherItem->setIsDifferent( true );
			return;
		}
		if( currentItem->fileItem()->permissions() != otherItem->fileItem()->permissions() ) {
			currentItem->setIsDifferent( true );
			otherItem->setIsDifferent( true );
			return;
		}
	}
     // if we get here there are no difference
	if( currentItem ) {
		currentItem->setIsDifferent( false );
		currentItem->setCorrespondingMissing( false );
	}
	if( otherItem ) {
		otherItem->setIsDifferent( false );
		otherItem->setCorrespondingMissing( false );
	}
}
//-----------------------------------------------
void KBearDirSynchPart::clearDiff() {

	 DirSynchTreeViewItem* localItem = 0L;
	DirSynchTreeViewItem* remoteItem = 0L;
	// first we clear the local tree
	if( m_localView->firstChild() ) {
		QListViewItemIterator localit( m_localView->firstChild() );
		for( ++localit ; localit.current(); ++localit ) {
			localItem = static_cast<DirSynchTreeViewItem*>( localit.current() );
			localItem->setIsDifferent( false );
			localItem->setCorrespondingMissing( false );
			localItem->setMissingAbove( false );
			localItem->setMissingBelow( false );
		}
	}
	// now we clear the remote tree
	if( m_remoteView->firstChild() ) {
		QListViewItemIterator remoteit( m_remoteView->firstChild() );
		for( ++remoteit ; remoteit.current(); ++remoteit ) {
			remoteItem = static_cast<DirSynchTreeViewItem*>( remoteit.current() );
			remoteItem->setIsDifferent( false );
			remoteItem->setCorrespondingMissing( false );
			remoteItem->setMissingAbove( false );
			remoteItem->setMissingBelow( false );
		}
	}
}
//-----------------------------------------------
void KBearDirSynchPart::checkDiff() {
	if( ! m_localView || ! m_remoteView )
		return;

	DirSynchTreeViewItem* localItem = 0L;
	DirSynchTreeViewItem* remoteItem = 0L;

	emit setStatusBarText( i18n("Looking for differences...") );


	clearDiff();
	// first we parse the local tree
	if( m_localView->firstChild() ) {
		QListViewItemIterator localit( m_localView->firstChild() );
		for( ++localit ; localit.current(); ++localit ) {
			localItem = static_cast<DirSynchTreeViewItem*>( localit.current() );
			remoteItem = findCorrespondingItem( localItem );
			setDiff( localItem, remoteItem );
		}
	}
	// now we need to parse the remote tree
	// because there might be remote files/directories
	// that was missing in local tree
	// but we only need to test for missing
	// since time difference already been made
	if( m_remoteView->firstChild() ) {
		QListViewItemIterator remoteit( m_remoteView->firstChild() );
		for( ++remoteit ; remoteit.current(); ++remoteit ) {
			remoteItem = static_cast<DirSynchTreeViewItem*>( remoteit.current() );
			localItem = findCorrespondingItem( remoteItem );
			setDiff( remoteItem, localItem );
		}
	}
}
//-----------------------------------------------
void KBearDirSynchPart::slotSynchAll() {
}
//-----------------------------------------------
void KBearDirSynchPart::slotSynchFromLocal() {
	m_action = FROM_LOCAL;
	setActionsEnabled( false );
	QApplication::setOverrideCursor( Qt::WaitCursor );
	emit setStatusBarText( i18n("Preparing...") );

	m_copyTransfer = new Transfer;
	m_copyTransfer->setDestConnection( m_dirLister->connection() );
	m_copyTransfer->setSourceConnection( m_localBranch->url() );
	KFileTreeViewItem* root = m_localBranch->root();
	QListViewItemIterator it( root->firstChild() );
	m_touchList.clear();
	for( ; it.current(); it++ ) {
		DirSynchTreeViewItem* i = static_cast<DirSynchTreeViewItem*>( it.current() );
		if( i->isDifferent() ) {
			m_copyTransfer->sourceList().append( i->url() );
			m_touchList.append( i );
		}
	}
	m_copyTransfer->setDestURL( m_connection.url() );

	m_deleteList.clear();

	root = m_remoteBranch->root();
	QListViewItemIterator it2( root ->firstChild() );
	for( ; it2.current(); it2++ ) {
		DirSynchTreeViewItem* i = static_cast<DirSynchTreeViewItem*>( it2.current() );
		if( i->correspondingMissing() )
			m_deleteList.append( i->url() );
	}

	if( m_deleteList.isEmpty() ) {
		m_state = DELETING;
		slotSynchResult( 0L );
	}
	else {
		m_state = DELETING;
		emit setStatusBarText( i18n("Deleting files...") );
		KIO::Job* job = m_dirLister->deleteFiles( m_deleteList, false, false );
		connect( job, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotSynchResult( KIO::Job* ) ) );
	}
}
//-----------------------------------------------
void KBearDirSynchPart::slotSynchFromRemote() {
	m_action = FROM_REMOTE;
	QApplication::setOverrideCursor( Qt::WaitCursor );
	setActionsEnabled( false );
	emit setStatusBarText( i18n("Preparing...") );

	m_copyTransfer = new Transfer;
	m_copyTransfer->setDestConnection( m_localBranch->url() );
	m_copyTransfer->setSourceConnection( m_dirLister->connection() );
	KFileTreeViewItem* root = m_remoteBranch->root();
	QListViewItemIterator it( root->firstChild() );
	m_touchList.clear();
	for( ; it.current(); it++ ) {
		DirSynchTreeViewItem* i = static_cast<DirSynchTreeViewItem*>( it.current() );
		if( i->isDifferent() ) {
			m_copyTransfer->sourceList().append( i->url() );
			m_touchList.append( i );
		}
	}
	m_copyTransfer->setDestURL( m_localBranch->url() );

	m_deleteList.clear();

	root = m_localBranch->root();
	QListViewItemIterator it2( root ->firstChild() );
	for( ; it2.current(); it2++ ) {
		DirSynchTreeViewItem* i = static_cast<DirSynchTreeViewItem*>( it2.current() );
		if( i->correspondingMissing() )
			m_deleteList.append( i->url() );
	}

	if( m_deleteList.isEmpty() ) {
		m_state = DELETING;
		slotSynchResult( 0L );
	}
	else {
		m_state = DELETING;
		emit setStatusBarText( i18n("Deleting files...") );
		KIO::Job* job = KIO::del( m_deleteList, false, false );
		connect( job, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotSynchResult( KIO::Job* ) ) );
	}
}
//-----------------------------------------------
void KBearDirSynchPart::slotSynchResult( KIO::Job* job ) {
	kdDebug()<<"KBearDirSynchPart::slotSynchResult job="<<job<<endl;
	switch( m_action ) {
		case FROM_LOCAL: {
			if( ( ! job || (job && ! job->error()) ) && m_state == DELETING ) {
				kdDebug()<<"KBearDirSynchPart::slotSynchResult FROM_LOCAL DELETING job="<<job<<endl;
				m_state = COPYING;
				emit setStatusBarText( i18n("Copying files...") );
				KBearCopyJob* j = connectionManager->copy( m_copyTransfer, 0, (unsigned long)m_dirLister );
				connect( j, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotSynchResult( KIO::Job* ) ) );
				connect( j, SIGNAL( infoMessage( KIO::Job*, const QString& ) ),
							this, SLOT( slotInfoMessage( KIO::Job*, const QString& ) ) );
				j->slotStart();
				return;
			}
			else if( ! job->error() ) {
				kdDebug()<<"KBearDirSynchPart::slotSynchResult FROM_LOCAL COPYING job="<<job<<endl;
				DirSynchTreeViewItem* touchItem;
				for( touchItem = m_touchList.first(); touchItem; touchItem = m_touchList.next() ) {
					QString exec = QString::fromLatin1("touch");
					QString cmd = QString::fromLatin1("touch --time=modify %1").arg( touchItem->path() );
					KRun::runCommand( cmd, exec, QString::null );
				}
			}
			openConnection( m_connection );
			break;
		}
		case FROM_REMOTE: {
			if( ( ! job || ! job->error() ) && m_state == DELETING ) {
				kdDebug()<<"KBearDirSynchPart::slotSynchResult FROM_REMOTE DELETING job="<<job<<endl;
				m_state = COPYING;
				emit setStatusBarText( i18n("Copying files...") );
				KBearCopyJob* j = connectionManager->copy( m_copyTransfer, (unsigned long)m_dirLister, 0 );
				connect( j, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotSynchResult( KIO::Job* ) ) );
				connect( j, SIGNAL( infoMessage( KIO::Job*, const QString& ) ),
					this, SLOT( slotInfoMessage( KIO::Job*, const QString& ) ) );
				j->slotStart();
				return;
			}
			else if( ! job->error() ) {
				kdDebug()<<"KBearDirSynchPart::slotSynchResult FROM_REMOTE COPYING job="<<job<<endl;
				DirSynchTreeViewItem* item;
				DirSynchTreeViewItem* touchItem;
				for( item = m_touchList.first(); item; item = m_touchList.next() ) {
					touchItem = findCorrespondingItem( item );
					if( touchItem ) {
						QString time = item->timeString();
						time = time.replace( QRegExp( "[\\s:-]" ), "" );
						QString exec = QString::fromLatin1("touch");
						QString cmd = QString::fromLatin1("touch --time=modify -t %1 %2").arg( time ).arg( touchItem->path() );
						KRun::runCommand( cmd, exec, QString::null );
					}
				}
			}
			openConnection( m_connection );
			break;
		}
		case SELECTED_FROM_LOCAL: {
			if( ! job->error() ) {
				kdDebug()<<"KBearDirSynchPart::slotSynchResult SELECTED_FROM_LOCAL job="<<job<<endl;
				QString exec = QString::fromLatin1("touch");
				QString cmd = QString::fromLatin1("touch --time=modify %1").arg( m_selectedTouchPath );
				KRun::runCommand( cmd, exec, QString::null );
			}
			openConnection( m_connection );
			break;
		}
		case SELECTED_FROM_REMOTE: {
			if( ! job->error() ) {
				kdDebug()<<"KBearDirSynchPart::slotSynchResult SELECTED_FROM_REMOTE job="<<job<<endl;
				DirSynchTreeViewItem* item;
				for( item = m_touchList.first(); item; item = m_touchList.next() ) {
					QString time = item->timeString();
					time = time.replace( QRegExp( "[\\s:-]" ), "" );
					QString exec = QString::fromLatin1("touch");
					QString cmd = QString::fromLatin1("touch --time=modify -t %1 %2").arg( time ).arg( m_selectedTouchPath );
					KRun::runCommand( cmd, exec, QString::null );
				}
			}
			openLocalBranch();
			break;
		}
		if( m_copyTransfer )
			delete m_copyTransfer;
		emit setStatusBarText( i18n("Ready.") );
		QApplication::restoreOverrideCursor();
	}
}
//-----------------------------------------------
void KBearDirSynchPart::slotSynchSelectedFromLocal() {
	m_action = SELECTED_FROM_LOCAL;
	emit setStatusBarText( i18n("Preparing...") );
	m_deleteList.clear();
	m_copyTransfer = 0L;

	// first we need to find out exactly what we are going to do:
	// 1. if local item is selected we should just copy
	// 2. if remote item is selected and local is missing, then we should delete
	KURL sourceURL = m_localView->currentURL();
	KURL destURL = m_remoteView->currentURL();
	m_selectedTouchPath = sourceURL.path();

	if( ! sourceURL.isEmpty() ) { // case 1
		m_state = COPYING;
		emit setStatusBarText( i18n("Copying files...") );
		m_copyTransfer = new Transfer;
		m_copyTransfer->setDestConnection( m_dirLister->connection() );
		m_copyTransfer->setSourceConnection( m_localBranch->url() );
		m_copyTransfer->sourceList().append( sourceURL );
		// we now need to create the destination URL
		QString relURL = sourceURL.url();
		relURL = relURL.remove( 0, m_localBranch->rootUrl().url(+1).length() );
		if( destURL.isEmpty() )
			destURL = m_remoteBranch->rootUrl().url(+1) + relURL;
		m_copyTransfer->setDestURL( destURL );

	     setActionsEnabled( false );
		QApplication::setOverrideCursor( waitCursor );
		emit started( 0L );
		KBearCopyJob* job = connectionManager->copy( m_copyTransfer, 0, (unsigned long)m_dirLister );
		job->setOverwriteAll();
		connect( job, SIGNAL( infoMessage( KIO::Job*, const QString& ) ),
							this, SLOT( slotInfoMessage( KIO::Job*, const QString& ) ) );
		connect( job, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotSynchResult( KIO::Job* ) ) );
		job->slotStart();
	}
	else if( ! destURL.isEmpty() ) {  // case 2
		m_deleteList.append( destURL );
		if( ! m_warnDelete || quizDelete( m_deleteList ) ) {
			m_state = DELETING;
		     setActionsEnabled( false );
			QApplication::setOverrideCursor( waitCursor );
			emit started( 0L );
			emit setStatusBarText( i18n("Deleting files...") );
			KIO::Job* job = m_dirLister->deleteFiles( m_deleteList, false, false );
			connect( job, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotSynchResult( KIO::Job* ) ) );
		}
	}
	else
		return;

}
//-----------------------------------------------
void KBearDirSynchPart::slotSynchSelectedFromRemote() {
     m_action = SELECTED_FROM_REMOTE;
	setActionsEnabled( false );
	QApplication::setOverrideCursor( waitCursor );
	emit started( 0L );
	emit setStatusBarText( i18n("Preparing...") );
	m_deleteList.clear();
	m_copyTransfer = 0L;

     // first we need to find out exactly what we are going to do:
	// 1. if remote item is selected we should just copy
	// 2. if local item is selected and remote is missing, then we should delete
	KURL sourceURL = m_remoteView->currentURL();
	KURL destURL = m_localView->currentURL();
	m_touchList.clear();
	DirSynchTreeViewItem* item = static_cast<DirSynchTreeViewItem*>( m_remoteView->currentItem() );
	m_touchList.append( item );
	QString rootURL = m_remoteBranch->rootUrl().url(+1);
	QString relURL = item->url().url(-1);
	relURL = relURL.remove( 0, rootURL.length() );
	KURL url( m_localBranch->rootUrl().url(+1) + relURL );
	m_selectedTouchPath = url.path();

	setActionsEnabled( false );
	m_localBranchFinished = false;
	m_remoteBranchFinished = true;
	m_isWorking = false;
	if( ! sourceURL.isEmpty() ) { // case 1
		m_state = COPYING;
		emit setStatusBarText( i18n("Copying files...") );
		m_copyTransfer = new Transfer;
		m_copyTransfer->setDestConnection( m_localBranch->url() );
		m_copyTransfer->setSourceConnection( m_dirLister->connection() );
		m_copyTransfer->sourceList().append( sourceURL );
		// we now need to create the destination URL
		QString relURL = sourceURL.url();
		relURL = relURL.remove( 0, m_remoteBranch->rootUrl().url(+1).length() );
		if( destURL.isEmpty() )
			destURL = m_localBranch->rootUrl().url(+1) + relURL;
		m_copyTransfer->setDestURL( destURL );

		QFile::remove( destURL.path() );
		KBearCopyJob* job = connectionManager->copy( m_copyTransfer, (unsigned long)m_dirLister, 0 );
		job->setOverwriteAll();
		connect( job, SIGNAL( infoMessage( KIO::Job*, const QString& ) ),
							this, SLOT( slotInfoMessage( KIO::Job*, const QString& ) ) );
		connect( job, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotSynchResult( KIO::Job* ) ) );
		job->slotStart();
	}
	else if( ! destURL.isEmpty() ) {  // case 2
		m_deleteList.append( destURL );
		if( ! m_warnDelete || quizDelete( m_deleteList ) ) {
			m_state = DELETING;
		     setActionsEnabled( false );
			QApplication::setOverrideCursor( waitCursor );
			emit started( 0L );
			emit setStatusBarText( i18n("Deleting files...") );
			KIO::Job* job = KIO::del( m_deleteList, false, false );
			connect( job, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotSynchResult( KIO::Job* ) ) );
		}
	}
	else
		return;

}
//-----------------------------------------------
void KBearDirSynchPart::slotSelectionChanged( QListViewItem* selectedItem ) {
	const QObject* obj = QObject::sender();
	if( ! obj )
		return;

	KFileTreeView* view = 0L;
	KFileTreeView* selView = 0L;
	if( obj == m_localView ) {
		view = m_remoteView;
		selView = m_localView;
	}
	else if( obj == m_remoteView ) {
		view = m_localView;
		selView = m_remoteView;
	}
	else
		return;

	m_localView->blockSignals( true );
	m_remoteView->blockSignals( true );
	m_localView->verticalScrollBar()->blockSignals( true );
	m_remoteView->verticalScrollBar()->blockSignals( true );
	m_localView->horizontalScrollBar()->blockSignals( true );
	m_remoteView->horizontalScrollBar()->blockSignals( true );

	QListViewItemIterator it( view );
	for( ++it; it.current(); it++ ) {
		DirSynchTreeViewItem* i = static_cast<DirSynchTreeViewItem*>( it.current() );
		i->setSelectedAbove( false );
		i->setSelectedBelow( false );
		i->setSelected( false );
		i->repaint();
	}
	QListViewItemIterator it2( selView );
	for( ++it2; it2.current(); it2++ ) {
		DirSynchTreeViewItem* i = static_cast<DirSynchTreeViewItem*>( it2.current() );
		i->setSelectedAbove( false );
		i->setSelectedBelow( false );
		i->repaint();
	}

	DirSynchTreeViewItem* selectTVI = 0L;
	DirSynchTreeViewItem* selectedTVI = static_cast<DirSynchTreeViewItem*>( selectedItem );
	if( selectedTVI && ! selectedTVI->isDir() ) {
		selectTVI = findCorrespondingItem( selectedTVI );
		if( selectTVI ) {
			view->setSelected( selectTVI, true );
			view->ensureItemVisible( selectTVI );
		}
		else {
			selectTVI = findCorrespondingItemBelow( selectedTVI );
			if( selectTVI ) {
				selectTVI->setSelectedAbove( true );
				view->ensureItemVisible( selectTVI );
				selectTVI->repaint();
			}
			selectTVI = findCorrespondingItemAbove( selectedTVI );
			if( selectTVI ) {
				selectTVI->setSelectedBelow( true );
				view->ensureItemVisible( selectTVI );
				selectTVI->repaint();
			}
		}
	}
	else {
		selectedItem->setSelected( false );
		selectedTVI = 0L;
	}

	m_localView->blockSignals( false );
	m_remoteView->blockSignals( false );
	m_localView->update();
	m_remoteView->update();
/*
	if( selectTVI ) {
		int selPos = selectedItem->itemPos() - selView->itemAt( QPoint(10, 10) )->itemPos();
		int otherPos = selectTVI->itemPos() - view->itemAt( QPoint(10, 10) )->itemPos();
//		KMessageBox::information( view, QString("selPos=%1 otherPos=%2").arg(selPos).arg(otherPos) );
		int dy = otherPos - selPos;
		view->scrollBy( 0, dy );
	}
*/
	m_localView->verticalScrollBar()->blockSignals( false );
	m_remoteView->verticalScrollBar()->blockSignals( false );
	m_localView->horizontalScrollBar()->blockSignals( false );
	m_remoteView->horizontalScrollBar()->blockSignals( false );
	setSelectedActionsEnabled( selectedTVI );
}
//-----------------------------------------------
void KBearDirSynchPart::selectChildren( QListView* view, QListViewItem* parent ) {
	for( QListViewItem* i = parent->firstChild(); i; i = i->nextSibling() ) {
		view->setSelected( i, true );
		if( static_cast<KFileTreeViewItem*>( i )->isDir()  )
			selectChildren( view, i );
	}
}
//-----------------------------------------------
void KBearDirSynchPart::slotExpanded( QListViewItem* item ) {
	if( ! item )
		return;
	DirSynchTreeViewItem* TVI = findCorrespondingItem( static_cast<KFileTreeViewItem*>( item ) );
	if( TVI && ! TVI->isOpen() )
		TVI->setOpen( true );
}
//-----------------------------------------------
void KBearDirSynchPart::slotCollapsed( QListViewItem* item ) {
	if( ! item )
		return;
	DirSynchTreeViewItem* TVI = findCorrespondingItem( static_cast<KFileTreeViewItem*>( item ) );
	if( TVI && TVI->isOpen() )
		TVI->setOpen( false );
}
//-----------------------------------------------
void KBearDirSynchPart::setActionsEnabled( bool enabled) {
//	m_synchAllAction->setEnabled( enabled );
	m_synchFromLocalAction->setEnabled( enabled );
	m_synchFromRemoteAction->setEnabled( enabled );

//	m_synchAllButton->setEnabled( enabled );
	m_synchAllFromLocalButton->setEnabled( enabled );
	m_synchAllFromRemoteButton->setEnabled( enabled );

	m_localView->setEnabled( enabled );
	m_remoteView->setEnabled( enabled );

	bool sel = ( m_localView->selectedItems().count() || m_remoteView->selectedItems().count() );
	setSelectedActionsEnabled( sel );

}
//-----------------------------------------------
void KBearDirSynchPart::setSelectedActionsEnabled( bool enabled) {
	m_synchSelFromLocalAction->setEnabled( enabled );
	m_synchSelFromRemoteAction->setEnabled( enabled );

	m_synchFromLocalButton->setEnabled( enabled );
	m_synchFromRemoteButton->setEnabled( enabled );
}
//-----------------------------------------------
void KBearDirSynchPart::setupActions() {
/*
	m_synchAllAction = new KAction( i18n("Syncronize &All"), "dirsynch", CTRL+Key_F1,
					this, SLOT( slotSynchAll() ), actionCollection(), "synch_all" );
*/
	new KAction( i18n("Configure S&ynchronize..."), "run" , 0,
					this, SLOT(slotConfigureDirSynch()), actionCollection(), "options_configure_dirsynch" );
	m_synchFromLocalAction = new KAction( i18n("Synchronize all from local directory"), "2rightarrow", CTRL+Key_F2,
					this, SLOT( slotSynchFromLocal() ), actionCollection(), "synch_from_local" );
	m_synchFromRemoteAction = new KAction( i18n("Synchronize all from remote directory"), "2leftarrow", CTRL+Key_F3,
					this, SLOT( slotSynchFromRemote() ), actionCollection(), "synch_from_remote" );
	m_synchSelFromLocalAction = new KAction( i18n("Synchronize selected from local directory"), "forward", CTRL+Key_F4,
					this, SLOT( slotSynchSelectedFromLocal() ), actionCollection(), "synch_selected_from_local" );
	m_synchSelFromRemoteAction = new KAction( i18n("Synchronize selected from remote directory"), "back", CTRL+Key_F5,
					this, SLOT( slotSynchSelectedFromRemote() ), actionCollection(), "synch_selected_from_remote" );
}
//-----------------------------------------------
void KBearDirSynchPart::setupWidget() {
	QGridLayout* dirSynchWidgetLayout = new QGridLayout( widget(), 1, 1, 11, 6, "dirSynchWidgetLayout");

	QLabel* lbl = new QLabel( widget(), "LocalDirLabel" );
	lbl->setText( i18n("Local Directory:") );
//	lbl->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, lbl->sizePolicy().hasHeightForWidth() ) );

	dirSynchWidgetLayout->addWidget( lbl, 0, 0 );


	lbl = new QLabel( widget(), "RemoteDirLabel" );
	lbl->setText( i18n("Remote Directory:") );

	dirSynchWidgetLayout->addWidget( lbl, 0, 2 );

	m_localDirLbl = new QLabel( widget(), "LocalDir" );

	dirSynchWidgetLayout->addWidget( m_localDirLbl, 1, 0 );


	m_remoteDirLbl = new QLabel( widget(), "RemoteDir" );

	dirSynchWidgetLayout->addWidget( m_remoteDirLbl, 1, 2 );

/*
	m_synchAllButton = new KPushButton( widget(), "m_synchAllButton" );
	m_synchAllButton->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, m_synchAllButton->sizePolicy().hasHeightForWidth() ) );
	m_synchAllButton->setPixmap( BarIcon("dirsynch") );
	QToolTip::add( m_synchAllButton, i18n( "Push this button if you want to synchronize all files/directories." ) );
	QWhatsThis::add( m_synchAllButton, i18n( "Push this button if you want to synchronize all files/directories.\n"
														"This will use the modified timestamp to determine if it should update "
														"the remote or the local file/directory." ) );

	dirSynchWidgetLayout->addWidget( m_synchAllButton, 2, 1 );
*/
	m_synchAllFromLocalButton = new KPushButton( widget(), "m_synchAllFromLocalButton" );
	m_synchAllFromLocalButton->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, m_synchAllFromLocalButton->sizePolicy().hasHeightForWidth() ) );
	m_synchAllFromLocalButton->setPixmap( BarIcon("2rightarrow") );
	QToolTip::add( m_synchAllFromLocalButton, i18n( "Push this button if you want to update the remote directory." ) );
	QWhatsThis::add( m_synchAllFromLocalButton, i18n( "Push this button if you want to update the remote directory." ) );

	dirSynchWidgetLayout->addWidget( m_synchAllFromLocalButton, 2, 1 );

	m_synchAllFromRemoteButton = new KPushButton( widget(), "m_synchAllFromRemoteButton" );
	m_synchAllFromRemoteButton->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, m_synchAllFromRemoteButton->sizePolicy().hasHeightForWidth() ) );
	m_synchAllFromRemoteButton->setPixmap( BarIcon("2leftarrow") );
	QToolTip::add( m_synchAllFromRemoteButton, i18n( "Push this button if you want to update the local directory." ) );
	QWhatsThis::add( m_synchAllFromRemoteButton, i18n( "Push this button if you want to update the local directory." ) );

	dirSynchWidgetLayout->addWidget( m_synchAllFromRemoteButton, 3, 1 );

	m_synchFromLocalButton = new KPushButton( widget(), "m_synchFromLocalButton" );
	m_synchFromLocalButton->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, m_synchFromLocalButton->sizePolicy().hasHeightForWidth() ) );
	m_synchFromLocalButton->setPixmap( BarIcon("forward") );
	QToolTip::add( m_synchFromLocalButton, i18n( "Push this button if you want to update the selected remote file/directory." ) );
	QWhatsThis::add( m_synchFromLocalButton, i18n( "Push this button if you want to update the selected remote file/directory." ) );

	dirSynchWidgetLayout->addWidget( m_synchFromLocalButton, 4, 1 );

	m_synchFromRemoteButton = new KPushButton( widget(), "m_synchFromRemoteButton" );
	m_synchFromRemoteButton->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, m_synchFromRemoteButton->sizePolicy().hasHeightForWidth() ) );
	m_synchFromRemoteButton->setPixmap( BarIcon("back") );
	QToolTip::add( m_synchFromRemoteButton, i18n( "Push this button if you want to update the selected local file/directory." ) );
	QWhatsThis::add( m_synchFromRemoteButton, i18n( "Push this button if you want to update the selected local file/directory." ) );

	dirSynchWidgetLayout->addWidget( m_synchFromRemoteButton, 5, 1 );

	QSpacerItem* spacer = new QSpacerItem( 0, 10, QSizePolicy::Minimum, QSizePolicy::MinimumExpanding );
	dirSynchWidgetLayout->addItem( spacer, 6, 1 );

	m_doneButton = new KPushButton( widget(), "m_synchFromRemoteButton" );
	m_doneButton->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, m_doneButton->sizePolicy().hasHeightForWidth() ) );
	QFont m_doneButton_font(  m_doneButton->font() );
	m_doneButton_font.setBold( true );
	m_doneButton->setFont( m_doneButton_font );
	m_doneButton->setText( i18n( "&Close" ) );

	dirSynchWidgetLayout->addWidget( m_doneButton, 7, 1 );

	m_localView = new DirSynchTreeView( widget(), "m_localView" );

	dirSynchWidgetLayout->addMultiCellWidget( m_localView, 2, 6, 0, 0 );

	m_remoteView = new DirSynchTreeView( widget(), "m_remoteView" );

	dirSynchWidgetLayout->addMultiCellWidget( m_remoteView, 2, 6, 2, 2 );

	connect( m_localView->header(), SIGNAL( sizeChange( int, int, int ) ),
				m_remoteView, SLOT( slotHeaderSizeChange( int, int, int ) ) );
	connect( m_remoteView->header(), SIGNAL( sizeChange( int, int, int ) ),
				m_localView, SLOT( slotHeaderSizeChange( int, int, int ) ) );
	connect( m_synchAllFromLocalButton, SIGNAL( clicked() ),
				this, SLOT( slotSynchFromLocal() ) );
	connect( m_synchAllFromRemoteButton, SIGNAL( clicked() ),
				this, SLOT( slotSynchFromRemote() ) );
	connect( m_synchFromLocalButton, SIGNAL( clicked() ),
				this, SLOT( slotSynchSelectedFromLocal() ) );
	connect( m_synchFromRemoteButton, SIGNAL( clicked() ),
				this, SLOT( slotSynchSelectedFromRemote() ) );
	connect( m_localView->verticalScrollBar(), SIGNAL( valueChanged( int  ) ),
				m_remoteView->verticalScrollBar(), SLOT( setValue( int ) ) );
	connect( m_remoteView->verticalScrollBar(), SIGNAL( valueChanged( int  ) ),
				m_localView->verticalScrollBar(), SLOT( setValue( int ) ) );
	connect( m_localView->verticalScrollBar(), SIGNAL( sliderMoved( int  ) ),
				m_remoteView->verticalScrollBar(), SLOT( setValue( int ) ) );
	connect( m_remoteView->verticalScrollBar(), SIGNAL( sliderMoved( int  ) ),
				m_localView->verticalScrollBar(), SLOT( setValue( int ) ) );

	connect( m_localView->horizontalScrollBar(), SIGNAL( valueChanged( int  ) ),
				m_remoteView->horizontalScrollBar(), SLOT( setValue( int ) ) );
	connect( m_remoteView->horizontalScrollBar(), SIGNAL( valueChanged( int  ) ),
				m_localView->horizontalScrollBar(), SLOT( setValue( int ) ) );
	connect( m_localView->horizontalScrollBar(), SIGNAL( sliderMoved( int  ) ),
				m_remoteView->horizontalScrollBar(), SLOT( setValue( int ) ) );
	connect( m_remoteView->horizontalScrollBar(), SIGNAL( sliderMoved( int  ) ),
				m_localView->horizontalScrollBar(), SLOT( setValue( int ) ) );

	connect( m_remoteView, SIGNAL( expanded( QListViewItem* ) ),
				this, SLOT( slotExpanded( QListViewItem* ) ) );
	connect( m_remoteView, SIGNAL( collapsed( QListViewItem* ) ),
				this, SLOT( slotCollapsed( QListViewItem* ) ) );
	connect( m_localView, SIGNAL( expanded( QListViewItem* ) ),
				this, SLOT( slotExpanded( QListViewItem* ) ) );
	connect( m_localView, SIGNAL( collapsed( QListViewItem* ) ),
				this, SLOT( slotCollapsed( QListViewItem* ) ) );

	connect( m_localView, SIGNAL( sortingChanged( int, bool ) ),
				m_remoteView, SLOT( setSorting( int, bool ) ) );
	connect( m_remoteView, SIGNAL( sortingChanged( int, bool ) ),
				m_localView, SLOT( setSorting( int, bool ) ) );

	connect( m_localView, SIGNAL( selectionChanged( QListViewItem* ) ),
				this, SLOT( slotSelectionChanged( QListViewItem* ) ) );
	connect( m_remoteView, SIGNAL( selectionChanged( QListViewItem* ) ),
				this, SLOT( slotSelectionChanged( QListViewItem* ) ) );


	widget()->show();
}
//-----------------------------------------------
void KBearDirSynchPart::slotConfigureDirSynch() {
	DirSynchConfigDialog dlg( widget(), "DirSynchConfigDialog", normalizeLabel( m_connection.label() ) );
	if( dlg.exec() == QDialog::Accepted )
		reparseConfiguration();
}
//-----------------------------------------------
void KBearDirSynchPart::reparseConfiguration() {
	kdDebug()<<"KBearDirSynchPart::reparseConfiguration()"<<endl;
	KConfig config("kbeardirsynchpartrc", false, false);
	QValueList<int> colList;
	if( ! config.hasGroup("DirSynchDefault") ) {
		config.setGroup( "DirSynchDefault" );
		colList.append( 237 );
		colList.append( 190 );
		colList.append( 190 );
		config.writeEntry( "Diff Color", colList );

		colList.clear();
		colList.append( 190 );
		colList.append( 237 );
		colList.append( 190 );
		config.writeEntry( "Missing Remote", colList );

		colList.clear();
		colList.append( 190 );
		colList.append( 190 );
		colList.append( 237 );
		config.writeEntry( "Missing Local", colList );

		config.writeEntry( "Confirm Delete", true );
		config.writeEntry( "Diff Type", SizePermissions );
		config.writeEntry( "Time Diff", 0 );
		config.sync();
	}
	QString group = normalizeLabel( m_connection.label() );
	if( config.hasGroup( group ) ) {
		config.setGroup( group );
		QValueList<int> colList = config.readIntListEntry( "Diff Color" );
		if( colList.size() > 2 )
			m_diffColor = QColor( colList[ 0 ], colList[ 1 ], colList[ 2 ] );

		colList = config.readIntListEntry( "Missing Remote" );
		if( colList.size() > 2 )
			m_missingRemoteColor =QColor( colList[ 0 ], colList[ 1 ], colList[ 2 ] );

		colList = config.readIntListEntry( "Missing Local" );
		if( colList.size() > 2 )
			m_missingLocalColor = QColor( colList[ 0 ], colList[ 1 ], colList[ 2 ] );

		m_warnDelete = config.readBoolEntry( "Confirm Delete", true );
		m_diffRule = config.readUnsignedNumEntry( "Diff Type", 1 );
		m_timeDiff = config.readNumEntry( "Time Diff", 0 );
		kdDebug()<<"KBearDirSynchPart::reparseConfiguration() diffTime="<<m_timeDiff<<endl;
	}
	if( m_remoteBranch ) {
		m_remoteBranch->setTimeDiff( m_timeDiff );
		m_remoteBranch->setColors( m_diffColor, m_missingRemoteColor, m_missingLocalColor );
	}
	if( m_localBranch ) {
		m_localBranch->setColors( m_diffColor, m_missingRemoteColor, m_missingLocalColor );
	}
	checkDiff();	
			
}
//-----------------------------------------------
bool KBearDirSynchPart::quizDelete( const KURL::List& urls ) {
	QStringList files;
	for( unsigned int i = 0; i < urls.count(); i++ ) {
		files.append( urls[ i ].url() );
		if( urls[ i ].isLocalFile() )
			files.append( urls[ i ].path() );
		else
			files.append( urls[ i ].prettyURL() );
	}
	int ret = false;
	if( files.count() == 1 ) {
		ret = KMessageBox::warningContinueCancel( widget(),
						i18n( "<qt>Do you really want to delete\n <b>'%1'</b>?</qt>" )
						.arg( files.first() ), i18n("Delete file"), i18n("Delete") );
	}
	else {
		ret = KMessageBox::warningContinueCancelList( widget(),
						i18n("Do you really want to delete these items?"),
						 files, i18n("Delete file"), i18n("Delete") );
	}
	return ret;
}
//-----------------------------------------------
void KBearDirSynchPart::slotInfoMessage( KIO::Job*, const QString& message ) {
	if( message.left(4) == "resp"
		|| message.left(7) == "command"
		|| message.left(10) == "multi-line"
		|| message.left(8) == "internal" )
	{
		emit logMessage( message );
	}
}
//-----------------------------------------------
