/***************************************************************************
                          ktransferimpl.cpp  -  description
                             -------------------
    begin                : Tue Oct 17 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.26 $
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kdebug.h>
#include <qdir.h>
#include <klocale.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include "utils.h"
#include "krootcontainerimpl.h"
#include "ktransfer.h"
#include "ktmanagerimpl.h"
#include "ktransferimpl.h"


const char KTransferImpl::DOCID[] = "Transfer";
const char KTransferImpl::DOCTYPE[] = "<!DOCTYPE Transfer >";


KTransferImpl::KTransferImpl(int type)
   :  KObjectImpl(type), _state(TRN_START), _rsmState(RSM_UNKNOWN), _resumed(0),
      _lastBytesRead(0), _mediumBandwidth(0), _bandwidth(0), _killed(false),
      _retry(0), _checkResume(0), _restart(false)
{
   _lastSampleTime.setHMS(0, 0, 0);
   _timeStart.setHMS(0, 0, 0);
   _timeEst.setHMS(0, 0, 0);
   setMessage(QString::null);
   connect(&_wait, SIGNAL(timeout()), SLOT(slotTimeout()));
}

KTransferImpl::~KTransferImpl()
{
   //kdDebug(D_INI) << name() << ": Transfer destroy" << endl;
}

/** load transfer from dom */
void KTransferImpl::loadData()
{
   KObjectImpl::loadData();
   kdDebug(D_INI) << name() << ": transfer loadData" << endl;
   if (!dom().hasAttribute("State"))
   {
      setPriority(PRT_NORMAL); // override object priority at first creation
      setState(TRN_READY);
      setRsmState(RSM_UNKNOWN);
      setResumed(0);
      setLen(0);
      setPartial(0);
      setCheckResume(true);
      setMaxResume(10);
      setAutoResume(true);
      setWaitResume(20);
      setMaxRetry(3);
      setWaitRetry(2);
      setLogProcess(false);
   }
   // state
   _remote = dom().attribute("Remote");
   _local.setPath(dom().attribute("Local")); //FIXME !! is correct??
   if (isNormal())
   {
      _tmp.setPath(dom().attribute("Tmp"));
      if (_tmp.path().isEmpty())
	 setTmp(getTempFile(remote(), local()).path());
   }
   else
      _tmp = QString::null;
   _state = static_cast<TrnState>(
      dom().attribute("State", QString::number(TRN_READY)).toInt());
   _rsmState = static_cast<RsmState>(
      dom().attribute("ResumeState", QString::number(RSM_UNKNOWN)).toInt());
   _resumed = dom().attribute("Resumed", QString::number(0)).toInt();
   _len = dom().attribute("Len", QString::number(0)).toInt();
   _partial = dom().attribute("Partial", QString::number(0)).toInt();

   _retry = 0;

   // config
   _cfg_checkResume = dom().attribute("CheckResume",
				      QString::number(true)).toInt();
   _cfg_maxResume = dom().attribute("MaxResume", QString::number(10)).toInt();
   _cfg_autoResume = dom().attribute("AutoResume",
				     QString::number(true)).toInt();
   _cfg_waitResume = dom().attribute("WaitResume", QString::number(20)).toInt();
   _cfg_maxRetry = dom().attribute("MaxRetry", QString::number(3)).toInt();
   _cfg_waitRetry = dom().attribute("WaitRetry", QString::number(2)).toInt();
   _cfg_logProcess = dom().attribute("LogProcess",
				     QString::number(false)).toInt();

   kdDebug(D_INI) << name() << " :" << endl
		  << " : rmt = " << _remote.url() << endl
		  << " : lcl = " << _local.path() << endl
		  << " : tmp = " << _tmp.path() << endl
		  << " : state " << stateStr() << endl
		  << " : rsmState " << resumeStr() << endl;
}

void KTransferImpl::checkState()
{
   if (!isNormal())
      return;
   QFileInfo tmpfile(tmp().path());
   QFileInfo file(local().path());
   if (tmpfile.exists())
   {
      // tmp file exists
      if ((unsigned int)partial() != tmpfile.size())
      {
	 kdWarning(D_INI) << name() << ": tmp file len (" << tmpfile.size() << ") != partial (" << partial() << ")" << endl;
	 setPartial(tmpfile.size());
      }
      // check if transfer is completed
      if (isComplete())
	 // transfer is complete, move tmp to final
	 moveTmp();
   }
   else if (file.exists())
   {
      // tmp file doesn't exists, but exists final file
      if ((unsigned int)len() != file.size())
      {
	 kdWarning(D_INI) << name() << ": file len (" << file.size() << ") != len (" << len() << ")" << endl;
	 //setLen(file.size());
      }
   }
   else
   {
      // either file and tmp doesn't exists
      if (partial() != 0)
      {
	 kdWarning(D_INI) << name() << ": file len (" << file.size() << ") != len (" << len() << ")" << endl;
	 clear();
      }
   }
   // resetting state to ready
   // if transfer is complete, leave state to TRN_END_OK
   if (!isComplete())
      setState(TRN_READY);
}

const KTransferImpl * KTransferImpl::global() const
{
   return dynamic_cast<const KTransferImpl*>(KObjectImpl::global());
}

QString KTransferImpl::stateStr(TrnState state)
{
   switch (state)
   {
   case TRN_READY:
      return i18n("Ready");
   case TRN_START:
      return i18n("Start");
   case TRN_CONNECT:
      return i18n("Connect");
   case TRN_DOWNLOAD:
      return i18n("Download");
   case TRN_END_OK:
      return i18n("Finish");
   case TRN_END_ERROR:
      return i18n("Error");
   case TRN_END_FATAL:
      return i18n("Fatal");
   case TRN_END_KILL:
      return i18n("Killed");
   case TRN_END_READY:
      return i18n("Restartable");
   }
   kdFatal(D_RUN) << "stateStr: unknown state " << state << endl;
   return QString::null;
}

QString KTransferImpl::progressStr() const
{
   if (partial() == 0)
      return i18n("Empty");
   else if (partial() < len())
      return i18n("Partial");
   else if (partial() == len())
      return i18n("Complete");
   kdError() << name() << ": partial > len " << stateStr() << endl;
   return QString::null;
};

QString KTransferImpl::resumeStr(RsmState state)
{
   switch (state)
   {
   case RSM_UNKNOWN:
      return i18n("???");
   case RSM_CHECKING:
      return i18n("***");
   case RSM_CHECKFAIL:
      return i18n("---");
   case RSM_YES:
      return i18n("Yes");
   case RSM_NO:
      return i18n("No");
   }
   kdFatal(D_RUN) << "resumeStr: unknown resume state " << state << endl;
   return QString::null;
};

QString KTransferImpl::resumeString(RsmState state)
{
   switch (state)
   {
   case RSM_UNKNOWN:
      return i18n("Unknown");
   case RSM_CHECKING:
      return i18n("Checking");
   case RSM_CHECKFAIL:
      return i18n("Check Failed");
   case RSM_YES:
      return i18n("Yes");
   case RSM_NO:
      return i18n("No");
   }
   kdFatal(D_RUN) << "resumeString: unknown resume state " << state << endl;
   return QString::null;
};

int KTransferImpl::cmpPriorityShort(const KObjectImpl *obj) const
{
   const KTransferImpl *t = dynamic_cast<const KTransferImpl*>(obj);
   // if you are comparing apple with oranges
   kdFatal(t == 0, D_RUN) << name() << ": cmpPriority type error in " << obj->name() << endl;
   return (len() == t->len() ?
	   (id() > t->id() ? -1 : 1) : (len() > t->len() ? -1 : 1));
}

int KTransferImpl::cmpPriorityLong(const KObjectImpl *obj) const
{
   const KTransferImpl *t = dynamic_cast<const KTransferImpl*>(obj);
   // if you are comparing apple with oranges
   kdFatal(t == 0, D_RUN) << name() << ": cmpPriority type error in " << obj->name() << endl;
   return (len() == t->len() ?
	   (id() > t->id() ? -1 : 1) : (len() < t->len() ? -1 : 1));
}

KURL KTransferImpl::getTempFile(const KURL &, const KURL &lcl) const
{
   KURL tmp;
   KTManagerImpl *parent = dynamic_cast<KTManagerImpl*>(container());
   kdFatal(parent == 0, D_RUN) << name() << ": getTempFile on object not in item's hierarchy" << endl;
   tmp.setPath(QFileInfo(parent->getWorkingDir(), lcl.fileName()).absFilePath());
   return tmp;
}

bool KTransferImpl::isComplete() const
{
   return (state() == TRN_END_OK && partial() == len() && len() > 0);
}

void KTransferImpl::clear()
{
   QString tmpfile = tmp().path();
   QString file = local().path();
   if (QDir::current().exists(tmpfile))
      QDir::current().remove(tmpfile);
   if (QDir::current().exists(file))
      QDir::current().remove(file);
   setPartial(0);
   setState(TRN_READY);
   setRsmState(RSM_UNKNOWN);
   setMessage(QString::null);
   setMod(MDI_CLEAR);
   // refresh view
   emitMod();
}

bool KTransferImpl::isRunning() const
{
   switch (state())
   {
   case TRN_START:
   case TRN_CONNECT:
   case TRN_DOWNLOAD:
      return true;
   default:
      return false;
   }
   return false;
}

bool KTransferImpl::isFinished() const
{
   switch (state())
   {
   case TRN_END_OK:
   case TRN_END_ERROR:
   case TRN_END_FATAL:
   case TRN_END_KILL:
   case TRN_END_READY:
      return true;
   default:
      return false;
   }
   return false;
}

bool KTransferImpl::isRunnable(bool automatic) const
{
   if (isComplete() || isRunning() || priority() == 0)
      return false;
   return
      ((automatic && state() == TRN_READY) // automatic download
       || ( state() == TRN_END_READY // resuming
	    && getAutoResume()
	    && (isResumable() || rsmState() == RSM_UNKNOWN
		|| rsmState() == RSM_CHECKFAIL || partial() == 0)
	    && ((!getMaxResume() || resumed() < getMaxResume())
		|| (!getMaxResume() || resumed() == getMaxResume()
		    && retry() < getMaxRetry())))
	 );
}

bool KTransferImpl::isRunnableInFuture(bool dLock, bool automatic) const
{
   return (!isComplete() && priority() > 0
	   && ((!dLock && isRunnable(automatic)) // runnable now
	       || (getAutoResume()
		   && (isResumable() || rsmState() == RSM_UNKNOWN
		       || rsmState() == RSM_CHECKFAIL || partial() == 0)
		   && (state() == TRN_END_OK || state() == TRN_END_ERROR)
		   && ((!getMaxRetry() || retry() < getMaxRetry())
		       || (!dLock && !getMaxResume()
			   || resumed() < getMaxResume())))));
}


void KTransferImpl::raiseFatalError()
{
   if (getLogProcess()) {
      QString msg;
      QTextStream log(&msg, IO_WriteOnly | IO_Append);
      log << "\n-------------------------------------------\n";
      _logFile.open(IO_ReadOnly);
      QTextStream flog(&_logFile);
      //_logFile.at(_logFile.size() - 10000 > 0 ? _logFile.size() - 10000 : 0);
      while (!flog.atEnd()) {
	 QString line = flog.readLine();
	 log << line << endl;
      }
      _logFile.close();
      setMessage(message() + msg);
   }
   //setFatalError(false);
   setMod(MDI_FATAL);
   // refresh view
   emitMod();
}

void KTransferImpl::moveTmp()
{
   if (tmp() != local())
   {
      if (QDir::current().exists(local().path()))
	 saveFile(local().path());
      //FIXME recupero errore
      bool check = QDir::current().rename(tmp().path(), local().path());
      kdError(check == false, D_RUN) << name() << ": move tmpfile " << tmp().path() << " -> " << local().path() << endl;
   }
}

bool KTransferImpl::start()
{
   kdDebug(D_RUN) << name() << ": " << stateStr() << ": start " << endl;
   if (isRunning())
   {
      kdError(D_RUN) << name() << ": start a running process" << endl;
      return false;
   }
   if (isComplete())
      clear();
   // _checkResume can be set in startWithCheckResume()
   if (_checkResume == 0 && rsmState() == RSM_UNKNOWN && getCheckResume())
   {
      _checkResume = 2;
   }
   if (_checkResume)
   {
      kdDebug(D_RUN) << name() << ": check resume begin" << endl;
      processCheckResumeBegin();
      setRsmState(RSM_CHECKING);
   }
   setState(TRN_READY);
   return processStart();
}

bool KTransferImpl::startCheckResume()
{
   _checkResume = 1;
   return start();
}

bool KTransferImpl::kill(bool wait)
{
   kdDebug(D_RUN) << name() << ":" << stateStr() << ": kill " << endl;
   //kdError(!isRunning(), D_RUN) << name() << ": kill process not running" << endl;
   _killed = true;
   if (!isRunning() && (state() == TRN_END_OK || state() == TRN_END_ERROR))
   {
      _wait.stop();
      setState(TRN_END_KILL);
      return true;
   }
   if (processKill())
   {
      if (wait)
	 waitForKill();
      return true;
   }
   _killed = false;
   return false;
}

bool KTransferImpl::stop(bool wait)
{
   kdDebug(D_RUN) << name() << ":" << stateStr() << ": stop " << endl;
   _stopped = true;
   if (kill(wait))
      return true;
   _stopped = false;
   return false;
}

void KTransferImpl::opProcessStart()
{
   kdDebug(D_RUN) << name() << ":" << stateStr() << ": process start " << endl;
   kdFatal(!isReady(), D_RUN) << name() << ": process start with state " << stateStr() << endl;
   setFatalError(false);
   setMessage(QString::null);
   _lastBytesRead = 0;
   _lastSampleTime.setHMS(0, 0, 0);
   _mediumBandwidth = 0;
   _bandwidth = 0;
   _timeEst.setHMS(0, 0, 0);
   _killed = false;
   _stopped = false;
   _restart = false;
   if (getLogProcess())
   {
      QString logname = QFileInfo(
	 dynamic_cast<KTManagerImpl*>(container())->getWorkingDir(),
	 local().fileName() + ".log").absFilePath();
      _logFile.setName(logname);
      _logFile.open(IO_WriteOnly | IO_Append);
   }
   _timeStart.start();
   _lastSampleTime.start();
   _lastBytesRead = partial();
   _lastBytesStart = partial();
   setRetry(_retry >= getMaxRetry() ? 1 : _retry + 1);
   if (retry() == 1)
      setResumed(_resumed + 1);
   setState(TRN_START);
   container()->incrItemActive(1);
   setMod(MDI_START);
   // update view
   emitMod();
}

void KTransferImpl::opConnect()
{
   kdDebug(D_RUN) << name() << ": " << stateStr() << " : connect " << endl;
   setState(TRN_CONNECT);
   // update view
   emitMod();
}

void KTransferImpl::opDownload()
{
   kdDebug(D_RUN) << name() << ": " << stateStr() << " : download p=" << partial() << ",l=" << len() << endl;
   _timeStart.start();
   _lastSampleTime.start();
   _lastBytesRead = partial();
   _lastBytesStart = partial();
   setState(TRN_DOWNLOAD);
   // update view
   emitMod();
}

void KTransferImpl::opDataRead(int datalen)
{
   kdDebug(D_RUN) << name() << ": " << stateStr() << " : read data " << datalen << endl;
   kdError(state() != TRN_DOWNLOAD, D_RUN) << name() << ": opDataRead not int Download State" << endl;
   int oldPercent = percent();
   setPartial(partial() + datalen);
   int newPercent = percent();
   if (newPercent != oldPercent)
      emit sigPercent(newPercent);
}

void KTransferImpl::opCheckResume(RsmState newState)
{
   setRsmState(newState);
   if (_checkResume && !_killed)
   {
      kdDebug(D_RUN) << name() << ": opCheckResume : kill" << endl;
      // hack to delay a little the kill operation, needed in kio
      QTimer::singleShot(200, this, SLOT(slotKill()));
   }
}

void KTransferImpl::opError(const QString &err)
{
   // signal an error, but continue the normal operations
   kdDebug(D_RUN) << name() << ": " << stateStr() << " : error " << err << endl;
   setMessage(err);
   // update view
   emitMod();
   //FIXME problems to maintain message visible
}

void KTransferImpl::opFatalError(const QString &err)
{
   // signal a fatal error, kill the current transfer
   kdDebug(D_RUN) << name() << ": " << stateStr() << " : fatal error :" << err << endl;
   setFatalError(true);
   setMessage(err);
   if (isRunning())
      kill();
}

void KTransferImpl::opProcessEnd(int exitStatus)
{
   kdDebug(D_RUN) << name() << ": " << stateStr() << " : process end " << exitStatus << endl;

   // set state
   _timeStart.setHMS(0,0,0);
   if (getLogProcess())
      _logFile.close();
   container()->incrItemActive(-1);
   setMod(MDI_STOP);

   // analyze the returned state
   if (!isFinished())
   {
      opFatalError(QString("%1 : exit state not correct").arg(stateStr()));
   }

   // check for fatal errors
   if (fatalError())
   {
      raiseFatalError();
      setState(TRN_END_FATAL);
      _checkResume = 0;
      return;
   }

   // check if this is a resume checking run
   if (_checkResume != 0)
   {
      kdDebug(D_RUN) << name() << ": check resume end : " << stateStr() << endl;
      switch (rsmState())
      {
      case RSM_UNKNOWN:
      case RSM_CHECKING:
	 kdWarning(D_RUN) << name() << ": resume uncorrect after check :" << resumeString(rsmState()) << endl;
	 setRsmState(RSM_CHECKFAIL);
	 break;
      case RSM_CHECKFAIL:
      case RSM_YES:
      case RSM_NO:
	 kdDebug(D_RUN) << name() << ": resume after check :" << resumeString(rsmState()) << endl;
	 break;
      default:
	 kdFatal(D_RUN) << name() << ": unknown resume state :" << rsmState() << endl;
      }
      processCheckResumeEnd();
      if ((state() == TRN_END_OK || state() == TRN_END_KILL
	   || state() == TRN_END_ERROR))
      {
	 if (_checkResume == 1)
	    setState(TRN_READY);
	 else
	 {
	    _restart = true;
	    _wait.start(100, true);
	 }
      }
      kdDebug(D_RUN) << name() << ": state after check :" << stateStr() << endl;
      _checkResume = 0;
      emitMod();
      return;
   }

   // check for completeness and determine what to do
   if (isComplete())
   {
      // transfer complete, move tmp to final
      moveTmp();
   }
   // check for transfer stopped
   if (state() == TRN_END_KILL && _stopped)
   {
      setState(TRN_END_OK);
      // resume, not retry
      setRetry(getMaxRetry());
      if (priority() > 0)
	 setPriority(priority() - 1);
   }

   // wait before retrying / resuming
   if (!isComplete() && (state() == TRN_END_OK || state() == TRN_END_ERROR))
   {
      if (retry() < getMaxRetry())
	 _wait.start(getWaitRetry()*1000, true);
      else
      {
	 _wait.start(getWaitResume()*1000, true);
	 // if not stopped, decrease priority
	 if (!_stopped && priority() > 0)
	    setPriority(priority() - 1);
      }
   }
   // update views
   emitMod();
}

void KTransferImpl::opProcessOutput(const QString &data)
{
   if (getLogProcess())
      _logFile.writeBlock(data, data.length());
   emit sigProcessOutput(data);
}

void KTransferImpl::calcBandwidth()
{
   int time = _lastSampleTime.elapsed();
   // FIXME!! gestisci meglio intervallo di campionamento
   if (time > 0)
   {
      setMod(MDI_BAND);
      // calculate instantaneous bandwidth
      _bandwidth = (int)(((1000.0 * (partial() - _lastBytesRead)) / time) * 0.66 + _bandwidth * 0.33);
      _lastSampleTime.start();
      _lastBytesRead = partial();
   }
   time = _timeStart.elapsed();
   if (time > 0)
      _mediumBandwidth = (int)((1000.0 * (partial() - _lastBytesStart)) / time);
   //debug("READ %l %l", _bandwidth, _mediumBandwidth);//(const char*)_parsebuf);
}

QTime KTransferImpl::estTime()
{
   if (mediumBandwidth())
      return QTime().addSecs((len() - partial())/ mediumBandwidth());
   return QTime();
}

void KTransferImpl::runPeriodically()
{
   if (isRunning())
      calcBandwidth();
   KObjectImpl::runPeriodically();
}

void KTransferImpl::resetState()
{
   setState(TRN_READY);
   _wait.stop();
}

/////////// state members

void KTransferImpl::setLocal(const KURL &l)
{
   kdDebug(D_RUN) << name() << " set local = " << l.path() << endl;
   _local = l;
   dom().setAttribute("Local", _local.path());
   setMod(MDI_TRN_LOCAL);
   root()->setModified();
}

void KTransferImpl::setRemote(const KURL &r)
{
   kdDebug(D_RUN) << name() << " set remote = " << r.path() << endl;
   _remote = r;
   dom().setAttribute("Remote", _remote.url());
   setMod(MDI_TRN_REMOTE);
   root()->setModified();
}

void KTransferImpl::setTmp(const KURL &t)
{
   kdDebug(D_RUN) << name() << " set tmp = " << t.path() << endl;
   _tmp = t;
   dom().setAttribute("Tmp", _tmp.path());
   setMod(MDI_TRN_TMP);
   root()->setModified();
}

void KTransferImpl::setLen(int l)
{
   kdDebug(D_RUN) << name() << " set len = " << l << endl;
   _len = l;
   dom().setAttribute("Len", _len);
   setMod(MDI_TRN_LEN);
   root()->setModified();
}

void KTransferImpl::setPartial(int p)
{
   kdDebug(D_RUN) << name() << " set partial = " << p << endl;
   _partial = p;
   dom().setAttribute("Partial", _partial);
   setMod(MDI_TRN_PARTIAL);
   root()->setModified();
}


void KTransferImpl::setState(TrnState s)
{
   if (_state != s)
   {
      _state = s;
      dom().setAttribute("State", _state);
      setMod(MDI_TRN_STATE);
      root()->setModified();
   }
}

void KTransferImpl::setRetry(int r)
{
   _retry = r;
   setMod(MDI_TRN_RETRY);
   root()->setModified();
}

void KTransferImpl::setResumed(int r)
{
   _resumed = r;
   dom().setAttribute("Resumed", _resumed);
   setMod(MDI_TRN_RESUMED);
   root()->setModified();
}

void KTransferImpl::setRsmState(RsmState s)
{
   _rsmState = s;
   dom().setAttribute("ResumeState", _rsmState);
   setMod(MDI_TRN_RSMSTATE);
   root()->setModified();
}

/////////// config entries

bool KTransferImpl::getCheckResume() const
{
   return useGlobal() ? global()->getCheckResume() : _cfg_checkResume;
}

void KTransferImpl::setCheckResume(bool c)
{
   if (_cfg_checkResume != c)
   {
      _cfg_checkResume = c;
      dom().setAttribute("CheckResume", _cfg_checkResume);
      setMod(MDI_TRN_CFG_CHKRESUME);
      root()->setModified();
   }
}

int KTransferImpl::getMaxResume() const
{
   return useGlobal() ? global()->getMaxResume() : _cfg_maxResume;
}

void KTransferImpl::setMaxResume(int r)
{
   if (_cfg_maxResume != r)
   {
      _cfg_maxResume = r;
      dom().setAttribute("MaxResume", _cfg_maxResume);
      setMod(MDI_TRN_CFG_MAXRESUME);
      root()->setModified();
   }
}

bool KTransferImpl::getAutoResume() const
{
   return useGlobal() ? global()->getAutoResume() : _cfg_autoResume;
}

void KTransferImpl::setAutoResume(bool m)
{
   if (_cfg_autoResume != m)
   {
      _cfg_autoResume = m;
      dom().setAttribute("AutoResume", _cfg_autoResume);
      setMod(MDI_TRN_CFG_AUTORESUME);
      root()->setModified();
   }
}

int KTransferImpl::getWaitResume() const
{
   return useGlobal() ? global()->getWaitResume() : _cfg_waitResume;
}

void KTransferImpl::setWaitResume(int w)
{
   if (_cfg_waitResume != w)
   {
      _cfg_waitResume = w;
      dom().setAttribute("WaitResume", _cfg_waitResume);
      setMod(MDI_TRN_CFG_WAITRESUME);
      root()->setModified();
   }
}

int KTransferImpl::getWaitRetry() const
{
   return useGlobal() ? global()->getWaitRetry() : _cfg_waitRetry;
}

int KTransferImpl::getMaxRetry() const
{
   return useGlobal() ? global()->getMaxRetry() : _cfg_maxRetry;
}

void KTransferImpl::setMaxRetry(int m)
{
   if (_cfg_maxRetry != m)
   {
      _cfg_maxRetry = m;
      dom().setAttribute("MaxRetry", _cfg_maxRetry);
      setMod(MDI_TRN_CFG_MAXRETRY);
      root()->setModified();
   }
}

void KTransferImpl::setWaitRetry(int w)
{
   if (_cfg_waitRetry != w)
   {
      _cfg_waitRetry = w;
      dom().setAttribute("WaitRetry", _cfg_waitRetry);
      setMod(MDI_TRN_CFG_WAITRETRY);
      root()->setModified();
   }
}

bool KTransferImpl::getLogProcess() const
{
   return useGlobal() ? global()->getLogProcess() : _cfg_logProcess;
}

void KTransferImpl::setLogProcess(bool l)
{
   if (_cfg_logProcess != l)
   {
      _cfg_logProcess = l;
      dom().setAttribute("LogProcess", _cfg_logProcess);
      setMod(MDI_TRN_CFG_LOGPROCESS);
      root()->setModified();
   }
}

//////////////  SLOTS

void KTransferImpl::slotTimeout()
{
   // set the transfer ready for processing
   setState(TRN_END_READY);
}

void KTransferImpl::slotKill()
{
   kill(false);
}

#include "ktransferimpl.moc"
