/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Software for Humanity
 *      NJN
 *
 *	This program is freely distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 *	$Id: tsubarray.c,v 1.0 1999/07/21 19:00:19 jsquyres Exp $
 *
 *	Function:	- create local array derived datatype
 *	Accepts:	- number of array dimensions
 *			- array of sizes
 *			- array of sub sizes
 *			- array of starting coordinates
 *			- storage order
 *			- element datatype
 *			- new datatype (out)
 *	Returns:	- MPI_SUCCESS or error code
 *
 *	Notes:		- this code is derived from code in the ROMIO
 *			  distibution authored by Rajeev Thakur.
 */

#include <errno.h>
#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

static void freetype();

int
MPI_Type_create_subarray(ndims, sizes, subsizes,
			starts, order, oldtype, newtype)

int			ndims;
int			*sizes;
int			*subsizes;
int			*starts;
int			order;
MPI_Datatype		oldtype;
MPI_Datatype		*newtype;

{
    MPI_Datatype	types[1];
    MPI_Aint		disps[1];
    int			blklens[1];
    MPI_Aint		ub;
    MPI_Datatype	ntype;
    MPI_Datatype	tmptype;
    MPI_Aint		extent;
    int			size;
    int			err;
    int			i;

    lam_initerr_m();
    lam_setfunc_m(BLKMPITCREATESUBARRAY);
/*
 * Check the arguments.
 */
    if (ndims <= 0 || sizes == 0
		|| subsizes == 0 || starts == 0 || newtype == 0) {
	return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATESUBARRAY, lam_mkerr(MPI_ERR_ARG, 0)));
    }

    if (oldtype == MPI_DATATYPE_NULL) {
	return(lam_errfunc(MPI_COMM_WORLD,
	    		BLKMPITCREATESUBARRAY, lam_mkerr(MPI_ERR_TYPE, 0)));
    }

    for (i = 0; i < ndims; i++) {
	if (sizes[i] <= 0 || subsizes[i] <= 0 || starts[i] < 0) {
	    return(lam_errfunc(MPI_COMM_WORLD,
		    	BLKMPITCREATESUBARRAY, lam_mkerr(MPI_ERR_ARG, 0)));
	}
    }

    MPI_Type_extent(oldtype, &extent);

    if (order == MPI_ORDER_FORTRAN) {
/*
 * Dimension 0 changes fastest.
 */
	if (ndims == 1) {
	    err = MPI_Type_contiguous(subsizes[0], oldtype, &ntype);
	    if (err != MPI_SUCCESS) {
		return(err);
	    }
	} else {
	    err = MPI_Type_vector(subsizes[1], subsizes[0],
			    		sizes[0], oldtype, &ntype);
	    if (err != MPI_SUCCESS) {
		return(err);
	    }

	    size = sizes[0] * extent;

	    for (i = 2; i < ndims; i++) {
		size *= sizes[i-1];
		
		err = MPI_Type_create_hvector(subsizes[i], 1, size,
		    		ntype, &tmptype);
		if (err != MPI_SUCCESS) {
		    return(err);
		}

		freetype(&ntype);
		ntype = tmptype;
	    }
	}
/*
 * Add displacement and UB.
 */
	disps[0] = starts[0];
	size = 1;
	for (i = 1; i < ndims; i++) {
	    size *= sizes[i-1];
	    disps[0] += size * starts[i];
	}
    }
    else if (order == MPI_ORDER_C) {
/*
 * Dimension ndims-1 changes fastest.
 */
	if (ndims == 1) {
	    err = MPI_Type_contiguous(subsizes[0], oldtype, &ntype);
	    if (err != MPI_SUCCESS) {
		return(err);
	    }
	} else {
	    err = MPI_Type_vector(subsizes[ndims-2], subsizes[ndims-1],
			    		sizes[ndims-1], oldtype, &ntype);
	    if (err != MPI_SUCCESS) {
		return(err);
	    }

	    size = sizes[ndims-1] * extent;

	    for (i = ndims - 3; i >= 0; i--) {
		size *= sizes[i+1];

		err = MPI_Type_create_hvector(subsizes[i], 1, size,
		    		ntype, &tmptype);
		if (err != MPI_SUCCESS) {
		    return(err);
		}

		freetype(&ntype);
		ntype = tmptype;
	    }
	}
/*
 * Add displacement and UB.
 */
	disps[0] = starts[ndims-1];
	size = 1;
	for (i = ndims - 2; i >= 0; i--) {
	    size *= sizes[i+1];
	    disps[0] += size * starts[i];
	}
    }
    else {
	return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITCREATESUBARRAY, lam_mkerr(MPI_ERR_ARG, 0)));
    }

    for (ub = extent, i = 0; i < ndims; i++) {
	ub *= sizes[i];
    }

    disps[0] *= extent;
    blklens[0] = 1;
    types[0] = ntype;

    err = MPI_Type_create_struct(1, blklens, disps, types, &ntype);
    if (err != MPI_SUCCESS) {
	return(err);
    }

    freetype(&types[0]);

    ntype->dt_flags &= ~(LAM_DTHASLB | LAM_DTHASUB);
    ntype->dt_lower = 0;
    ntype->dt_upper = ub;
/*
 * Set the no extent adjustment flag if the upper and lower bounds match
 * exactly the upper and lower limits of the data.
 */
    if (ntype->dt_upper == ntype->dt_dataup
		&& ntype->dt_lower == ntype->dt_datalow) {
	ntype->dt_flags |= LAM_DTNOXADJ;
    } else {
	ntype->dt_flags &= ~LAM_DTNOXADJ;
    }

    ntype->dt_format = LAM_DTSUBARRAY;
/*
 * Record user arguments.
 */
    ntype->dt_dtype = oldtype;
    oldtype->dt_refcount++;

    ntype->dt_uargs = (int *) malloc((3*ndims + 2) * sizeof(int));
    if (ntype->dt_uargs == 0) {
	return(lam_errfunc(MPI_COMM_WORLD, BLKMPITCREATESUBARRAY,
	    			lam_mkerr(MPI_ERR_OTHER, errno)));
    }

    ntype->dt_uargs[0] = ndims;
    ntype->dt_uargs[3*ndims + 1] = order;

    for (i = 0; i < ndims; i++) {
	ntype->dt_uargs[i + 1] = sizes[i];
	ntype->dt_uargs[ndims + i + 1] = subsizes[i];
	ntype->dt_uargs[2*ndims + i + 1] = starts[i];
    }

    *newtype = ntype;

    lam_resetfunc_m(BLKMPITCREATESUBARRAY);
    return(MPI_SUCCESS);
}

static void
freetype(MPI_Datatype *type)
{
    if (!((*type)->dt_flags & LAM_PREDEF)) {
	MPI_Type_free(type);
    }
}
