#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <pthread.h>

#include <sys/stat.h>

#include <lufs/proto.h>
#include <lufs/fs.h>

#include "list.h"
#include "vtree.h"
#include "gnet.h"
#include "gnetfs.h"
#include "search.h"

#define MAX_PATH	1024

static char path_buf[MAX_PATH];

void
delete_result(struct result *res){
    struct list_head *p, *tmp;
    struct gnet_locator *loc;

    list_for_each_safe(p, tmp, &res->locators){
	loc = list_entry(p, struct gnet_locator, list);

	list_del(&loc->list);
	free(loc);
    }

    list_del(&res->list);
    free(res->name);
    free(res);
}

void
delete_search(struct search *srch){
    struct list_head *p, *tmp;

    list_for_each_safe(p, tmp, &srch->results)
	delete_result(list_entry(p, struct result, list));

    list_del(&srch->list);
    free(srch->txt);
    free(srch);
}

struct search*
find_search_by_txt(struct global_ctx *glob, char *txt){
    struct list_head *p;
    struct search *srch;

    list_for_each(p, &glob->searches){
	srch = list_entry(p, struct search, list);
	if(!strcmp(srch->txt, txt)){
	    TRACE("search found");
	    return srch;
	}
    }

    TRACE("search not found");

    return NULL;    
}

struct search*
find_search_by_id(struct global_ctx *glob, unsigned long id){
    struct list_head *p;
    struct search *srch;

    list_for_each(p, &glob->searches){
	srch = list_entry(p, struct search, list);
	if(srch->id == id){
	    TRACE("search found");
	    return srch;
	}
    }

    TRACE("search not found");

    return NULL;
}

struct result*
find_result_by_name(struct search *srch, char *name){
    struct list_head *p;
    struct result *r;

    list_for_each(p, &srch->results){
	r = list_entry(p, struct result, list);

	if(!strcmp(r->name, name)){
	    TRACE("result found");
	    return r;
	}
    }

    TRACE("result not found");

    return NULL;

}

static struct result*
find_result_by_locator(struct search *srch, struct gnet_locator *loc){
    struct list_head *p;
    struct result *r;

    list_for_each(p, &srch->results){
	r = list_entry(p, struct result, list);

	if(!(strcmp(r->name, loc->name)) && (r->size == loc->size)){
	    TRACE("result found");
	    return r;
	}
    }

    TRACE("result not found");

    return NULL;
}

static void
search_hit(void *c, struct gnet_locator *loc, unsigned long id){
    struct global_ctx *glob = c;
    struct search *srch;
    struct result *res;
    struct gnet_locator *new_loc;
    struct lufs_fattr fattr;

    TRACE("got a search hit, search id=%lx, file=%s", id, loc->name);
    
    pthread_mutex_lock(&glob->lock);

    if(!(srch = find_search_by_id(glob, id))){
	WARN("unknown search id %lx", id);
	goto out;
    }

    if(!(res = find_result_by_locator(srch, loc))){
	TRACE("new result");

	if(!(res = malloc(sizeof(struct result)))){
	    WARN("could not allocate result: %s", strerror(errno));
	    goto out;
	}
	    
	memset(res, 0, sizeof(struct result));

	if(!(res->name = malloc(strlen(loc->name) + 1))){
	    WARN("could not allocate result: %s", strerror(errno));
	    free(res);
	    goto out;	    
	}

	INIT_LIST_HEAD(&res->locators);
	res->stamp = time(NULL);
	res->size = loc->size;
	strcpy(res->name, loc->name);	
	list_add(&res->list, &srch->results);
    }

    if(!(new_loc = malloc(sizeof(struct gnet_locator)))){
	WARN("could not allocate result locator");
	goto out;
    }

    memset(new_loc, 0, sizeof(struct gnet_locator));

    new_loc->name = res->name;
    memcpy(new_loc->ip, loc->ip, 4);
    memcpy(new_loc->guid, loc->guid, 16);
    new_loc->port = loc->port;
    new_loc->index = loc->index;
    new_loc->size = loc->size;
    new_loc->bwidth = loc->bwidth;

    list_add(&new_loc->list, &res->locators);
    res->nr_locators ++;

    if(snprintf(path_buf, MAX_PATH, "%s/%s", SEARCH_DIR, srch->txt) >= MAX_PATH){
	WARN("search string too long");
	goto out;
    }

    TRACE("adding %s to %s", loc->name, path_buf);

    memset(&fattr, 0, sizeof(struct lufs_fattr));
    fattr.f_nlink = res->nr_locators;
    fattr.f_uid = fattr.f_gid = 1;
    fattr.f_mode = S_IRUSR | S_IRGRP | S_IROTH | S_IFREG;
    fattr.f_mtime = fattr.f_ctime = fattr.f_atime = res->stamp;
    fattr.f_size = res->size;
    
    lu_vtree_add(glob->vtree, path_buf, res->name, NULL, &fattr, NULL);

  out:
    pthread_mutex_unlock(&glob->lock);
}

int
start_search(struct local_ctx *ctx, char *txt){
    struct search *srch;
    struct global_ctx *glob = *ctx->global;

    TRACE("starting search for \"%s\"", txt);

    if(!(srch = malloc(sizeof(struct search))))
	goto fail;

    memset(srch, 0, sizeof(struct search));

    if(!(srch->txt = malloc(strlen(txt) + 1)))
	goto fail_srch;

    strcpy(srch->txt, txt);

    INIT_LIST_HEAD(&srch->results);
    srch->stamp = time(NULL);
    
    if(gnet_start_search(glob->gnet, txt, search_hit, glob, 0, &srch->id) < 0){
	WARN("could not start gnet search: %s", strerror(errno));
	goto fail_txt;
    }
    
    pthread_mutex_lock(&glob->lock);

    list_add_tail(&srch->list, &glob->searches);

    pthread_mutex_unlock(&glob->lock);

    TRACE("search started");

    return 0;

  fail_txt:
    free(srch->txt);
  fail_srch:
    free(srch);
  fail:
    return -1;
}
