/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <stdio.h>
/* APR */
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apr_file_info.h>
#include <apr_file_io.h>
/* local */
#include <libbttracker.h>

/* Private Functions */

static char* shmem_filename(const char* homedir) {
    static char tmpfile[BT_FILE_LEN];
    size_t len;

    if((len = strlen(homedir) + strlen(BTT_SHMEM)) > sizeof(tmpfile) - 1) {
        fprintf(
            stderr,
            "shmem_filename(): Path \"%s%s\" is too long! (%zd > %zd chars)\n",
            homedir, BTT_SHMEM, len, sizeof(tmpfile) - 1
        );
        fflush(stderr);
        return NULL;
    }

    strcpy(tmpfile, homedir);
    strcat(tmpfile, BTT_SHMEM);
    return tmpfile;
}

static apr_shm_t* join_shmem_segment (apr_pool_t* p, const char* homedir) {
    apr_shm_t *m = NULL;
    int ret;
    char* shfile = apr_pstrdup(p, shmem_filename(homedir));
 
    if(!shfile)
        return NULL;
 
    if((ret = apr_shm_attach(&m, shfile, p)) != APR_SUCCESS) {
        fprintf(
            stderr,
            "join_shmem_segment(): Failed to connect to segment \"%s\": %s\n",
            shfile, apr_strerror(ret, btt_error_msg, sizeof(btt_error_msg))
        );
        
        fflush(stderr);
        m = NULL;
    }
 
    return m;
}

/* stay compatible with apr-0 */
static int apr0_shm_remove (const char *shfile, apr_pool_t *p) {
    int ret;
    apr_shm_t* shmem = NULL;

    if((ret = apr_shm_attach(&shmem, shfile, p)) != APR_SUCCESS) {
        if(APR_STATUS_IS_EINVAL(ret) || APR_STATUS_IS_ENOENT(ret)) {
            ret = apr_file_remove(shfile, p);
            if(APR_STATUS_IS_ENOENT(ret)) {
                return APR_SUCCESS;
            } else {
                return ret;
            }
        }
        
        return ret;
    } else {
        return apr_shm_destroy(shmem);
    }
}

static apr_shm_t* new_shmem_segment(
    apr_pool_t* p, apr_file_t** shlock_p, const char* homedir, apr_size_t size
) {
    apr_shm_t* rv = NULL;
    apr_file_t* shlock = NULL;
    int ret;
    char *shfile = apr_pstrdup(p, shmem_filename(homedir));
    char *shlockfile = apr_pstrcat(p, shfile, ".lock", NULL);

    if(!shfile)
        return NULL;
 
    /* TODO: move this into small inline functions for readability */
    /* obtain an exclusive lock before attempting to create the tracker. */
    if(
        (ret = apr_file_open(
            &shlock, shlockfile, APR_WRITE | APR_CREATE, APR_OS_DEFAULT, p
        ))
        != APR_SUCCESS
    ) {
        fprintf(
            stderr,
            "apr_file_open(%s) for writing failed: %s\n", shlockfile,
            apr_strerror(ret, btt_error_msg, sizeof(btt_error_msg))
        );
        
        fflush(stderr);
        rv = NULL;
    } else {
        if(
            (ret = apr_file_lock(
                shlock, APR_FLOCK_EXCLUSIVE | APR_FLOCK_NONBLOCK
            ))
            != APR_SUCCESS
        ) {
            fprintf(
                stderr, "apr_file_lock(%s) failed: %s\n", shlockfile,
                apr_strerror(ret, btt_error_msg, sizeof(btt_error_msg))
            );
            
            fflush(stderr);
            rv = NULL;
        } else {
            /* now that we have our lock, attempt to create a segment. */
            *shlock_p = shlock;
            if((ret = apr_shm_create(&rv, size, shfile, p)) != APR_SUCCESS) {
                /* if the segment already exists, 
                 * try to get rid of it and create it again.
                 */
                
                if(APR_STATUS_IS_EEXIST(ret)) {
                    if((ret = apr0_shm_remove(shfile, p)) != APR_SUCCESS) {
                        fprintf(
                            stderr, "apr0_shm_remove(%s) failed: %s\n",
                            shfile,
                            apr_strerror(
                                ret, btt_error_msg, sizeof(btt_error_msg)
                            )
                        );
                        
                        fflush(stderr);
                        rv = NULL;
                    } else {
                        if(
                            (ret = apr_shm_create(&rv, size, shfile, p))
                            != APR_SUCCESS
                        ) {
                            fprintf(
                                stderr,
                                "apr_shm_create("
                                BT_SIZE_T_FMT
                                ", \"%s\") failed: %s\n",
                                (bt_size_t) size, shfile,
                                apr_strerror(
                                    ret, btt_error_msg, sizeof(btt_error_msg)
                                )
                            );
                            
                            fflush(stderr);
                            rv = NULL;
                        }
                    }
                } else {
                    fprintf(
                        stderr, "apr_shm_create("
                        BT_SIZE_T_FMT
                        ", \"%s\") failed: %s\n",
                        (bt_size_t) size, shfile,
                        apr_strerror(ret, btt_error_msg, sizeof(btt_error_msg))
                    );
                    
                    fflush(stderr);
                    rv = NULL;
                }
            }
        }
    }

    return rv;
}

/* Public Functions */

btt_tracker* btt_tracker_alloc(
    apr_pool_t* p, const char* homedir, int master
) {
    apr_file_t* shlock = NULL;
    btt_tracker* rv = NULL;
    int ret;
 
    if(p == NULL) {
        if(!bt_random_seed()) {
            fprintf(stderr, "bt_random_seed() failed!\n");
            fflush(stderr);
            return NULL;
        }

        if(apr_pool_initialize() != APR_SUCCESS) {
            fprintf(stderr, "apr_pool_initialize() failed!\n");
            fflush(stderr);
            return NULL;
        }
 
        if((ret = apr_pool_create(&p, NULL)) != APR_SUCCESS) {
            fprintf(
                stderr,
                "bt_tracker_alloc(): Failed to create a pool: %s\n",
                apr_strerror(ret, btt_error_msg, sizeof(btt_error_msg))
            );
            fflush(stderr);
            return NULL;
        }
        
        if((rv = apr_pcalloc(p, sizeof(btt_tracker)))) {
            *rv = new_btt_tracker;
            
            rv->p = p;
            rv->myapr = 1;
        } else {
            fprintf(stderr, "allocating tracker memory failed!\n");
            fflush(stderr);
            apr_pool_destroy(p);
            apr_pool_terminate();
            return NULL;
        }
    } else {
        if((rv = apr_pcalloc(p, sizeof(btt_tracker)))) {
            *rv = new_btt_tracker;
            
            if((ret = apr_pool_create(&(rv->p), p)) == APR_SUCCESS) {
                rv->myapr = 0;
            } else {
                fprintf(
                    stderr,
                    "bt_tracker_alloc(): Failed to create sub-pool: %s\n",
                    apr_strerror(ret, btt_error_msg, sizeof(btt_error_msg))
                );
               return NULL;
            }
        } else {
            fprintf(
                stderr,
                "bt_tracker_alloc(): Failed to allocate "
                BT_SIZE_T_FMT " bytes!\n",
                (bt_size_t) sizeof(btt_tracker)
            );
            fflush(stderr);
            return NULL;
        }
            
    }
    
    rv->homedir = apr_pstrdup(p, homedir);

    if(master) {
        if(
            (rv->m = new_shmem_segment(
                rv->p, &shlock, homedir,
                sizeof(btt_tracker_config) + sizeof(btt_tracker_stats)
            ))
            == NULL
        ) {
            /* new_shmem_segment will print an error. */
            btt_free_tracker_pool(rv);
            return NULL;
        }
        rv->shlock = shlock;
    } else {
        if((rv->m = join_shmem_segment(rv->p, homedir)) == NULL) {
            /* join_shmem_segment will print an error. */
            btt_free_tracker_pool(rv);
            return NULL;
   	}
    }
    
    if(rv->myapr) {
        if(!btt_tracker_connect(rv, master)) {
            fprintf(stderr, "bt_tracker_connect(%i) failed!\n", master);
            fflush(stderr);
            btt_tracker_free(&rv, master);
            return NULL;
        }
    }
    
    return rv;
}
