/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <time.h>
/* other libs */
#include <db.h>
/* local */
#include <libbttracker.h>

btt_peer* btt_txn_load_peer(
    btt_tracker* tracker, apr_pool_t* p, DB_TXN* txn, DBT* key,
    int cc_flags, int c_flags, btt_infohash* hash
) {
    DBT* val = apr_pcalloc(p, sizeof(DBT));
    DBC* cursor = NULL;
    int ret = 0;
 
    val->data = apr_pcalloc(p, sizeof(btt_peer));
    val->ulen = sizeof(btt_peer);
    val->flags = DB_DBT_USERMEM;
 
    /* only one possible error location, already logged by load_hashcursor */
    if((ret = btt_txn_load_peercursor(
        tracker, p, txn, key, val, &cursor, cc_flags, c_flags, hash
    )) != 0)
        return NULL;

    cursor->c_close(cursor);
    return (btt_peer*)val->data;
}

int btt_txn_save_peer(
    btt_tracker* tracker, apr_pool_t* p, DB_TXN* txn,
    btt_peer* peer, btt_infohash* hash
) {
    DBT key;
    DBT val;
    DBC *cur = NULL;
    int ret = 0;
 
    bzero(&key, sizeof(key));
    bzero(&val, sizeof(val));
 
    key.data = apr_pcalloc(p, BT_PEERID_LEN + BT_INFOHASH_LEN);
    key.size = BT_PEERID_LEN + BT_INFOHASH_LEN;
    key.ulen = BT_PEERID_LEN + BT_INFOHASH_LEN;
    key.flags = DB_DBT_USERMEM;
 
    memcpy(key.data, hash->infohash, BT_INFOHASH_LEN);
    memcpy(key.data + BT_INFOHASH_LEN, peer->peerid, BT_PEERID_LEN);
 
    val.data = apr_palloc(p, sizeof(btt_peer));
    val.size = 0;
    val.ulen = sizeof(btt_peer);
    val.flags = DB_DBT_USERMEM;
 
    if((ret = btt_txn_load_peercursor(
        tracker, p, txn, &key, &val, &cur, BTT_WRITE_CURSOR(tracker),
        DB_RMW, hash
    )) != 0)    {
        tracker->db.peers->err(
            tracker->db.peers, ret,
            "bt_txn_save_peer(): bt_txn_load_peercursor()"
        );
        return ret;
    }
 
    val.data = peer;
 
    if((ret = cur->c_put(cur, &key, &val, DB_CURRENT)) != 0)
        tracker->db.peers->err(
            tracker->db.peers, ret, "bt_txn_save_peer(): c_put()"
        );

    cur->c_close(cur);
    return ret;
}

/* does not check if the hash exists! */
int btt_txn_load_peercursor(
    btt_tracker* tracker, apr_pool_t* p, DB_TXN* txn, DBT* key, DBT* val,
    DBC** cursor, int cc_flags, int c_flags, btt_infohash* hash
) {
    int ret = 0;
    DBC *rv = *cursor = NULL;

    DBT re_key;
    re_key = *key;
    re_key.data = apr_palloc(p, re_key.ulen);
    memcpy(re_key.data, key->data, key->size);

    if(key->ulen < BT_PEERID_LEN + BT_INFOHASH_LEN) {
        tracker->db.peers->errx(
            tracker->db.peers, "bt_txn_load_peercursor(): key->ulen < %d",
            BT_PEERID_LEN + BT_INFOHASH_LEN
        );
        return EINVAL;
    }
  
    if(key->size > BT_PEERID_LEN + BT_INFOHASH_LEN) { 
        tracker->db.peers->errx(
            tracker->db.peers, "bt_txn_load_peercursor(): key->size > %d",
            BT_PEERID_LEN + BT_INFOHASH_LEN
        );
        return EINVAL;
    }

    if(val->ulen < sizeof(btt_peer)) {
        tracker->db.peers->errx(
            tracker->db.peers, "bt_txn_load_peercursor(): val->ulen < %d",
            sizeof(btt_peer)
        );
        return EINVAL;
    }
  
    if(val->size) {
        tracker->db.peers->errx(
            tracker->db.peers, "bt_txn_load_peercursor(): val->size > 0"
        );
        return EINVAL;
    }

    if((ret = tracker->db.peers->cursor(
        tracker->db.peers, txn, &rv, cc_flags
    )) != 0) {
        tracker->db.peers->err(
            tracker->db.peers, ret, "bt_txn_load_peercursor(): cursor()"
        );
        return ret;
    }
 
    ret = rv->c_get(rv, key, val, c_flags | DB_SET);
 
    if(ret != 0 && ret != DB_NOTFOUND) {
        tracker->db.peers->err(
            tracker->db.peers, ret, "bt_txn_load_peercursor(): c_get()"
        );
        rv->c_close(rv);
        rv = NULL;
    } else if(ret == DB_NOTFOUND) {
        *((btt_peer*)val->data) = new_btt_peer;
        val->size = sizeof(btt_peer);
        memcpy(
            ((btt_peer*)val->data)->peerid,
            re_key.data + BT_INFOHASH_LEN,
            BT_PEERID_LEN
        );
        memcpy(
            ((btt_peer*)val->data)->infohash, hash->infohash, BT_INFOHASH_LEN
        );
        ((btt_peer*)val->data)->first_t = ((btt_peer*)val->data)->last_t =
            time(NULL);
        ((btt_peer*)val->data)->return_interval = tracker->c->return_interval;
        key->size = re_key.size;
        memcpy(key->data, re_key.data, re_key.size);

        if((ret = rv->c_put(rv, key, val, DB_KEYFIRST)) != 0) {
            tracker->db.peers->err(
                tracker->db.peers, ret, "bt_txn_load_peercursor(): c_put"
            );
            rv->c_close(rv);
            rv = NULL;
        } else if((ret = rv->c_get(rv, key, val, c_flags | DB_SET)) != 0) {
            tracker->db.peers->err(
                tracker->db.peers, ret,
                "bt_txn_load_peercursor(): c_get after c_put"
            );
            rv->c_close(rv);
            rv = NULL;
        } else {
            tracker->s->num_peers++;
            btt_iter_peer_stats(p, txn, rv, key, val, (void*)hash);
        }
    }
 
    *cursor = rv;
    return ret;
}
