/*
**  IMAPStore.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#import <Pantomime/IMAPStore.h>

#import <Pantomime/Constants.h>
#import <Pantomime/IMAPCacheManager.h>
#import <Pantomime/IMAPFolder.h>
#import <Pantomime/NSStringExtensions.h>
#import <Pantomime/TCPConnection.h>
#import <Pantomime/URLName.h>

@implementation IMAPStore

//
// This method implements a part of the Service Protocol.
//
- (id) initWithName: (NSString *) theName
               port: (int) thePort
{
  NSString *aString;
  
  self = [super init];

  [self setName: theName];
  [self setPort: thePort];

  messagesHaveBeenPrefetched = NO;
  tag = 1;
  
  tcpConnection = [[TCPConnection alloc] initWithName: theName
					 port: thePort];

  if ( !tcpConnection )
    {
      AUTORELEASE(self);
      return nil;
    }
  
  aString = [[self tcpConnection] readLineBySkippingCR: YES];
  
  if ( [aString hasCaseInsensitivePrefix: @"* OK"] )
    {
      NSLog(@"IMAPStore: Connected!");
    }
  else
    {
      AUTORELEASE(self);
      NSLog(@"IMAPStore: Not connected!");
      return nil;
    }

  return self;
}

//
// This method provide a default init method using
// the default IMAP port - 143.
// 
- (id) initWithName: (NSString *) theName
{
  return [self initWithName: theName
	       port: 143];
}


//
//
//
- (id) initWithURL: (NSString *) theURL
{
  URLName *urlName;

  urlName = [[URLName alloc] initWithString: theURL];

  self = [self initWithName: [urlName host]
	       port: 143];

  RELEASE(urlName);
  
  return self;
}

//
//
//
- (void) dealloc
{
  RELEASE(name);
  
  TEST_RELEASE(tcpConnection);

  [super dealloc];
}

//
// This method authenticates the Store to the IMAP server.
// In case of an error, it returns NO.
//
- (BOOL) authenticateWithUsername: (NSString*) username
			 password: (NSString*) password
{
  NSString *aString, *aPassword;
  NSRange aRange;

  // We must verify if we must quote the password
  aRange = [password rangeOfCharacterFromSet: [NSCharacterSet punctuationCharacterSet]];

  if ( aRange.length )
    {
      aPassword = [NSString stringWithFormat: @"\"%@\"", password];
    }
  else
    {
      aPassword = password;
    }
  

  [[self tcpConnection] writeLine: [NSString stringWithFormat:@"%@ LOGIN %@ %@",
					     [self nextTag], username, aPassword] ];
  
  aString = [[self tcpConnection] readLineBySkippingCR: YES];

  //
  // The answer can be:  0001 OK User logged in
  //                     0001 OK LOGIN completed
  //
  if ( [aString hasCaseInsensitivePrefix: [NSString stringWithFormat:@"%@ NO", [self lastTag]] ] ||
       [aString hasCaseInsensitivePrefix: [NSString stringWithFormat:@"%@ BAD", [self lastTag]] ] )
    {
      NSLog([NSString stringWithFormat: @"IMAPStore: IMAP username (or password) is invalid on %@.", [self name]]);
      return NO;
    }
  else
    {
      while ( ! [aString hasCaseInsensitivePrefix: [NSString stringWithFormat:@"%@ OK",
							     [self lastTag]] ] )
	{
	  aString = [[self tcpConnection] readLineBySkippingCR: YES];
	}
    }
  
  return YES;
}


- (NSString *) name
{
  return name;
}

- (void) setName: (NSString *) theName
{
  RETAIN(theName);
  RELEASE(name);
  name = theName;
}

- (int) port
{
  return port;
}

- (void) setPort: (int) thePort
{
  port = thePort;
}

//
//
//
- (TCPConnection *) tcpConnection
{
  return tcpConnection;
}

//
// The default folder in IMAP is always Inbox. This method will prefetch
// the messages of a IMAP folder if they haven't been prefetched before.
//
- (id) defaultFolder
{
  return [self folderForName: @"Inbox"];
}


//
//
//
- (id) folderForName: (NSString *) theName
{
  return [self folderForName: theName
	       prefetch: YES];
}

//
//
//
- (IMAPFolder *) folderForName: (NSString *) theName
		      prefetch: (BOOL) aBOOL
{  
  NSString *aString;
  NSMutableArray *aMutableArray;
 
  [[self tcpConnection] writeLine: [NSString stringWithFormat:@"%@ SELECT %@",
					     [self nextTag], theName]];
  
  aMutableArray = [[NSMutableArray alloc] init];
  
  aString = [[self tcpConnection] readLineBySkippingCR: YES];
  
  // We first verify if we got an error
  if ([aString hasCaseInsensitivePrefix: [NSString stringWithFormat:@"%@ NO",
						   [self lastTag]]] )
    {
      NSLog(@"IMAPStore: Error while obtaining the folder for name: %@", theName);
      return nil;
    }

  // If everything is alright, we read all the lines from the server
  while ( ![aString hasCaseInsensitivePrefix: [NSString stringWithFormat:@"%@ OK",
							[self lastTag]]] )
    {
      [aMutableArray addObject: aString];
      aString = [[self tcpConnection] readLineBySkippingCR: YES];
    }

  if ([aMutableArray count] > 0)
    {
      IMAPFolder *aFolder;
      int i;

      aFolder = [[IMAPFolder alloc] initWithName: theName];

      for (i = 0; i < [aMutableArray count]; i++)
	{
	  NSString *str;

	  str = [aMutableArray objectAtIndex: i];
	  //NSLog(@"|%@|", str);

	  if ([str hasCaseInsensitiveSuffix: @"EXISTS"])
	    {
	      // FIXME
	      //[aFolder setCount: [self parseExists: str]];
	    }
	  else if ([str hasCaseInsensitivePrefix: @"* OK [UIDVALIDITY"])
	    {
	      [aFolder setUIDValidity: [self parseUIDValidity: str]];
	    }
	}
      
      [aFolder setStore: (Store *)self];

      // If we have to prefetch, let's do it!
      if ( aBOOL )
	{
	  [aFolder prefetch];
	}
      
      return AUTORELEASE(aFolder);
    }

  return nil;
}


//
//
//
- (IMAPFolder *) folderForName: (NSString *) theName
	  withIMAPCacheManager: (IMAPCacheManager *) theIMAPCacheManager
{
  IMAPFolder *aFolder;

  aFolder = (IMAPFolder *)[self folderForName: theName
				prefetch: NO];
  
  if ( aFolder )
    {
      // We verify if we mush flush our cache
      if ([theIMAPCacheManager uidValidity] == 0)
	{
	  NSLog(@"IMAPStore: We set the UIDVALIDITY to the cache.");
	  [theIMAPCacheManager setUIDValidity: [aFolder uidValidity] ];
	}
      else if ([theIMAPCacheManager uidValidity] != [aFolder uidValidity])
	{
	  NSLog(@"IMAPStore: Flushing the cache...");
	  [theIMAPCacheManager flush];
	  [theIMAPCacheManager setUIDValidity: [aFolder uidValidity] ];
	}
      else
	{
	  NSLog(@"IMAPStore: The UIDVALIDITY of the cache is ok.");
	}

      [aFolder setIMAPCacheManager: theIMAPCacheManager];
      [aFolder prefetch];
    }

  return aFolder;
}


//
//
//
- (id) folderForURL: (NSString *) theURL
{
  URLName *urlName;
  id aFolder;

  urlName = [[URLName alloc] initWithString: theURL];

  aFolder = [self folderForName: [urlName foldername]];

  RELEASE(urlName);
  
  return aFolder;
}

- (NSString *) nextTag
{
  tag = tag + 1;  
  return [self lastTag];
}

- (NSString *) lastTag
{
  char str[5];
  
  sprintf(str, "%04x", tag);

  return [NSString stringWithCString: str];
}

- (void) close
{
  NSString *aString;

  // We ask our IMAP server to logout
  [[self tcpConnection] writeLine: [NSString stringWithFormat: @"%@ LOGOUT", [self nextTag]]];

  aString = [[self tcpConnection] readLineBySkippingCR: YES];

  while (! [aString hasCaseInsensitivePrefix: [NSString stringWithFormat: @"%@ OK", [self lastTag]] ] )
    {
      aString = [[self tcpConnection] readLineBySkippingCR: YES];
    }

  [[self tcpConnection] close];
}


//
//
//
- (int) parseExists: (NSString *) theLine
{
  NSRange aRange;

  theLine = [theLine substringFromIndex: 2];
  
  aRange = [theLine rangeOfString: @"EXISTS"];

  if ( aRange.length > 0 )
    {
      NSString *aString;
      
      aString = [theLine substringWithRange: NSMakeRange(0, aRange.location)];
      
      if ( [aString length] > 0 )
	{
	  return [aString intValue];
	}
    }

  return 0;
}


//
// Example: * OK [UIDVALIDITY 948394385] UID validity status
//
- (int) parseUIDValidity: (NSString *) theLine
{
  NSRange aRange;

  // We trim the  * OK [UIDVALIDITY part
  theLine = [theLine substringFromIndex: 17];

  // We find the ]
  aRange = [theLine rangeOfString: @"]"];

  if ( aRange.length )
    {
      NSString *aString;

      aString = [theLine substringWithRange: NSMakeRange(0, aRange.location)];
      
      if ( [aString length] > 0 )
	{
	  return [aString intValue];
	}
    }

  return 0;
}


//
//
//
- (BOOL) createFolderWithName: (NSString *) theName
{
  NSLog(@"IMAPStore: -createFolderWithName NOT IMPLEMENTED!");
  
  return NO;
}


//
//
//
- (BOOL) deleteFolderWithName: (NSString *) theName
{
  NSLog(@"IMAPStore: -deleteFolderWithName NOT IMPLEMENTED!");
  
  return NO;
}
@end
