#include <Foundation/NSObject.h>
#include <Foundation/NSString.h>
#include <Foundation/NSData.h>

#include "TestCase.h"

#import <Pantomime/MimeUtility.h>
#import <Pantomime/Constants.h>


@interface test_MimeUtility : TestCase
@end

@implementation test_MimeUtility

/*
This also tests base-64 decoding, quoted printable decoding, and character
set handling.
*/
- (void) test_decodeHeader
{
  struct { const char *in; NSString *out; } tests[] = {
    /* normal string */
    {"foo",@"foo"},
    /* 'almost' valid */
    {"foo=?bar",@"foo=?bar"},
    /* quoted printable */
    {"=?iso-8859-1?q?A=42C?=",@"ABC"},
    /* quoted printable + others */
    {"foo =?iso-8859-1?Q?Hall=E5_d=E4r?= zot",@"foo Hall dr zot"},
    /* utf-8 + base 64 + ignore whitespace between quoted words */
    {"foo =?utf-8?B?SGFs?= \n\t =?utf-8?b?bMOl?= bar",@"foo Hall bar"},
    /* pass invalid stuff through */
    {"foo =?invalid?q?=41=42=43?= zot",@"foo =?invalid?q?=41=42=43?= zot"},
    /* TODO: test invalid quoted or base-64 sequences? */
  };
  
  int i;
  NSData *d;
  NSString *s;
  
  for ( i = 0; i < sizeof(tests)/sizeof(tests[0]); i++)
    {
      d = [[NSData alloc] initWithBytes: tests[i].in 
			  length: strlen(tests[i].in)];

      s = [MimeUtility decodeHeader: d];
      [self equalString: tests[i].out : s];
      [d release];
    }
}

- (void) test_decode_quotedprintable
{
  static const char *in="Hall=E5 d=E4r! foo_bar_zot\n";
  static const char *expected="Hall dr! foo_bar_zot\n";
  NSData *got;
  
  got = [MimeUtility decodeQuotedPrintable: [NSData dataWithBytes: in
						    length: strlen(in)]
		     inHeader: NO];

  [self equalData: [NSData dataWithBytes: expected
			   length: strlen(expected)] : got];
}

-(void) test_unfoldLinesFromData
{
  static const char *in=
    "Header: foo bar zot\n"
    "\t quux\n"
    "Header-2: abcde\n"
    "Header-3: fgh\n"
    " ijk\n";
  
  static const char *out=
    "Header: foo bar zot\t quux\n"
    "Header-2: abcde\n"
    "Header-3: fgh ijk\n";
  
  NSData *din,*dout,*dgot;
  
  din = [NSData dataWithBytes: in
		length: strlen(in)];

  dout = [NSData dataWithBytes: out
		 length: strlen(out)];
  
  dgot = [MimeUtility unfoldLinesFromData: din];
  
  [self equalData: dout : dgot];
}


- (void) test_basic_encoding
{
  static const unsigned char in[]=
    "foo \x10\x1f\xff -==-\n";
  
  static const char *out_qp=
    "foo =10=1F=\n"
    "=FF -=3D=E5=\n"
    "=E4=F6=3D-=\n"
    "\n";
  
  static const char *out_b64=
    "Zm9vIBAf\n/yAtPeXk\n9j0tCg==\n";
  
  NSData *din,*dexp_qp,*dexp_b64;
  NSData *got_qp,*got_b64;
  
  din = [NSData dataWithBytes: in
		length: sizeof(in)-1];

  dexp_qp = [NSData dataWithBytes: out_qp
		    length: strlen(out_qp)];
  
  dexp_b64 = [NSData dataWithBytes: out_b64
		     length: strlen(out_b64)];
 
  got_qp = [MimeUtility encodeQuotedPrintable: din
		       lineLength: 10
		       inHeader: NO];
  
  got_b64 = [MimeUtility encodeBase64: din
			 lineLength: 10];
  
  [self equalData: dexp_qp : got_qp];
  [self equalData: dexp_b64 : got_b64];
}


-(void) test_encode_header
{
  static NSString *in1=@"Test foo_bar hall!";
  static unichar in2[]={
    0x0048,0x0065,0x0062,0x0072,0x0065,0x0077,0x0020,0x05e9,0x05dc,0x05d5,0x05dd};
  static unichar in3[]={'M','a','n','y',' ','e','n','c',' ',0x05e9,0x05dc,0x05d5,0x05dd,' ',0xe5,0xe4,0xf6,'!'};
  
  static const char *out1="Test_foo=5Fbar_hall=E5=21";
  static const char *out2="Hebrew_=F9=EC=E5=ED";
  /* AM-TODO: this is correct, I guess, but not really meaningful since it
     doesn't say which encoding it used. */
   
  static const char *out1b="VGVzdCBmb29fYmFyIGhhbGzDpSE=";
  static const char *out2b="Hebrew_=D7=A9=D7=9C=D7=95=D7=9D";
  
  /* AM-TODO: these are highly sub-optimal, but they work */
  static const char *out1c="=?iso-8859-1?q?Test_foo=5Fbar_hall=E5=21?=";
  static const char *out2c="=?iso-8859-8?q?Hebrew_=F9=EC=E5=ED?=";
  static const char *out3c="=?utf-8?q?Many_enc_=D7=A9=D7=9C=D7=95=D7=9D_=C3=A5=C3=A4=C3=B6=21?=";
  /*
    better if it were
    "Test foo_bar =?iso-8859-1?q?hall=E5=21?="
    and
    "Hebrew =?iso-8859-8?q?=F9=EC=E5=ED?="
  */

  
  NSData *got1,*got2,*got3;
  
  got1=[MimeUtility encodeHeader: in1];
  got2=[MimeUtility encodeHeader: [NSString stringWithCharacters: in2  length: sizeof(in2)/sizeof(in2[0])]];
  
  [self equalData: [NSData dataWithBytes: out1 length: strlen(out1)] : got1];
  [self equalData: [NSData dataWithBytes: out2 length: strlen(out2)] : got2];
  
  
  got1=[MimeUtility encodeHeader: in1
		    usingCharset: @"utf-8"
		    encoding: BASE64];

  got2=[MimeUtility encodeHeader: [NSString stringWithCharacters: in2  length: sizeof(in2)/sizeof(in2[0])]
		    usingCharset: @"utf-8"
		    encoding: QUOTEDPRINTABLE];
  
  [self equalData: [NSData dataWithBytes: out1b length: strlen(out1b)] : got1];
  [self equalData: [NSData dataWithBytes: out2b length: strlen(out2b)] : got2];
  
  
  got1=[MimeUtility encodeWordUsingQuotedPrintable: in1
		    prefixLength: 0];
  got2=[MimeUtility encodeWordUsingQuotedPrintable: [NSString stringWithCharacters: in2  length: sizeof(in2)/sizeof(in2[0])]
		    prefixLength: 0];
  
  got3=[MimeUtility encodeWordUsingQuotedPrintable: [NSString stringWithCharacters: in3  length: sizeof(in3)/sizeof(in3[0])]
		    prefixLength: 0];
  
  [self equalData: [NSData dataWithBytes: out1c length: strlen(out1c)] : got1];
  [self equalData: [NSData dataWithBytes: out2c length: strlen(out2c)] : got2];
  [self equalData: [NSData dataWithBytes: out3c length: strlen(out3c)] : got3];
}

@end

