#!/usr/bin/env python
# arch-tag: 2b579fa5-a9f4-4024-9473-d1d7ef4e4b76
# Copyright (C) 2005 Canonical Limited
# Author: David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for ancestry traversal.
"""

import shutil

import pybaz as arch

import framework
import fixtures


class AncestryFixture(fixtures.WorkingTreeFixture):

    memento_attrs = fixtures.WorkingTreeFixture.memento_attrs + (
        'version0', 'version1')

    def setUp(self):
        super(AncestryFixture, self).setUp()
        self.tree.import_() # version0['base-0']
        self.commit() # version0['patch-1']
        self.commit() # version0['patch-2']
        self.version0 = self.version
        self.version1 = self.version.branch['1']
        tree_path = str(self.tree)
        shutil.rmtree(tree_path)
        self.version0['patch-1'].make_continuation(self.version1)
        self.tree = self.version1.get(tree_path)
        self.commit() # version1['patch-1']
        self.commit() # version1['patch-2']
        self.commit(seal=True) # version1['version-0']
        self.commit(fix=True) # version1['versionfix-1']
        self.commit(fix=True) # version1['versionfix-2']

    def commit(self, **kwargs):
        m = self.tree.log_message()
        m['summary'] = 'dummy'
        self.tree.commit(m, **kwargs)


class Ancestry(framework.NewTestCase):
    fixture = AncestryFixture()

    def test_iter_ancestors(self):
        """Revision.iter_ancestors works."""
        version0, version1 = self.fixture.version0, self.fixture.version1
        last = version1['versionfix-2']
        ancestors = list(last.iter_ancestors())
        expected = [version1[level] for level in
                    ['versionfix-1', 'version-0',
                     'patch-2', 'patch-1', 'base-0']]
        expected += [version0[level] for level in ['patch-1', 'base-0']]
        self.assertEqual(ancestors, expected)
        ancestors = list(last.iter_ancestors(metoo=True))
        expected = [last] + expected
        self.assertEqual(ancestors, expected)

    def test_ancestor_import(self):
        """Revision.ancestor is None for an import revision."""
        version0 = self.fixture.version0
        self.assertEqual(version0['base-0'].ancestor, None)

    def test_ancestor_tag(self):
        """Revision.ancestor is correct for a continuation revision."""
        version0, version1 = self.fixture.version0, self.fixture.version1
        self.assertEqual(version1['base-0'].ancestor, version0['patch-1'])

    def test_ancestor_plain(self):
        """Revision.ancestor works for a plain revision."""
        version0 = self.fixture.version0
        self.assertEqual(version0['patch-2'].ancestor, version0['patch-1'])

    def test_previous_base0(self):
        """Revision.previous is None for a base-0, even if continuation."""
        version1 = self.fixture.version1
        self.assertEqual(version1['base-0'].previous, None)

    def test_previous_patch1(self):
        """Revision.previous is correct for a patch-1."""
        version1 = self.fixture.version1
        self.assertEqual(version1['patch-1'].previous, version1['base-0'])

    def test_previous_patch2(self):
        """Revision.previous is correct for a patch-2."""
        version1 = self.fixture.version1
        self.assertEqual(version1['patch-2'].previous, version1['patch-1'])

    def test_previous_version0(self):
        """Revision.previous works with a version-0."""
        version1 = self.fixture.version1
        self.assertEqual(version1['version-0'].previous, version1['patch-2'])

    def test_previous_versionfix1(self):
        """Revision.previous is correct with a versionfix-1."""
        version1 = self.fixture.version1
        self.assertEqual(version1['versionfix-1'].previous,
                         version1['version-0'])

    def test_previous_versionfix2(self):
        """Revision.previous is correct with a versionfix-2."""
        version1 = self.fixture.version1
        self.assertEqual(version1['versionfix-2'].previous,
                         version1['versionfix-1'])


framework.register(__name__)
