#include "plot.h"
#include <qwt_scale_engine.h>
#include <qwt_plot_curve.h>
#include <qwt_plot_grid.h>
#include <qwt_symbol.h>

Plot::Plot(QWidget *parent) :
    QwtPlot(parent),
    xmin(10.0),
    xmax(1000.0)
{
    setAutoReplot(false);
    initializeScales();
    initializeCanvas();
    initializeCurve();
    initializeMarkers();
    setAutoReplot(true);
}

Plot::Plot(QwtText title, QWidget *parent) :
    QwtPlot(title, parent),
    xmin(10.0),
    xmax(1000.0)
{
    setAutoReplot(false);
    initializeScales();
    initializeCanvas();
    initializeCurve();
    initializeMarkers();
    setAutoReplot(true);
}

Plot::~Plot()
{
    curve->setSamples(QVector<QPointF>());
    replot();
}

void Plot::plot(const QList<double> x, QList<double> y)
{
    QVector<QPointF> samples;
    double xpeak = xmax, ypeak = -120;
    double x3 = xmax, y3 = -120;

    /* roll in inverse order to find lower fpeak/f3 */
    for (int i = x.size() - 1; i > -1; i--) {
        /* search for fpeak */
        if (y[i] >= ypeak) {
            ypeak = y[i];
            xpeak = x[i];
        }

        /* search for f3 */
        if (round(y[i]) == round(ypeak - 3.0)) {
            x3 = x[i];
            y3 = y[i];
        }


        QPointF p;
        p.setX(x[i]);
        if (i < y.size())
            p.setY(y[i]);
        samples.append(p);
    }

    curve->setSamples(samples);
    //ypeak = round(ypeak);
    //y3 = round(y3);
    showPeak(xpeak, ypeak);
    show3dB(x3, y3);
    replot();
}

void Plot::initializeCurve() {
    curve = new QwtPlotCurve("Amplitude");
    curve->setRenderHint(QwtPlotItem::RenderAntialiased);
    curve->setPen(QPen(Qt::darkBlue, 1, Qt::SolidLine));
    curve->setLegendAttribute(QwtPlotCurve::LegendShowLine);
    curve->setYAxis(QwtPlot::yLeft);
    curve->attach(this);
}

void Plot::initializeMarkers()
{
    marker3 = new QwtPlotMarker();
    marker3->setLineStyle(QwtPlotMarker::VLine);
    marker3->setLabelAlignment(Qt::AlignRight | Qt::AlignBottom);
    marker3->setLinePen(QPen(Qt::blue, 1, Qt::DashDotLine));
    marker3->attach(this);

    markerP = new QwtPlotMarker();
    markerP->setLineStyle(QwtPlotMarker::HLine);
    markerP->setLabelAlignment(Qt::AlignRight | Qt::AlignBottom);
    markerP->setLinePen(QPen(Qt::red, 1, Qt::DashDotLine));
    markerP->attach(this);

}

double Plot::getXmax() const
{
    return xmax;
}

double Plot::getXmin() const
{
    return xmin;
}

void Plot::initializeCanvas() {

    setCanvasBackground(QColor(Qt::white));

    QwtPlotGrid *grid = new QwtPlotGrid;
    grid->enableXMin(true);
#if (QWT_VERSION >= QT_VERSION_CHECK(6,1,0))
    grid->setMajorPen(QPen(Qt::black, 1, Qt::DotLine));
    grid->setMinorPen(QPen(Qt::gray, 1, Qt::DotLine));
#else
    grid->setMajPen(QPen(Qt::black, 1, Qt::DotLine));
    grid->setMinPen(QPen(Qt::gray, 1, Qt::DotLine));
#endif
    grid->attach(this);
}

void Plot::initializeScales() {
    QwtText XAxisLabel;
    XAxisLabel.setText("Frequency [Hz]");
    setAxisTitle(QwtPlot::xBottom, XAxisLabel);
    setAxisMaxMajor(QwtPlot::xBottom, 3);
    setAxisMaxMinor(QwtPlot::xBottom, 10);
#if (QWT_VERSION >= QT_VERSION_CHECK(6,1,0))
    setAxisScaleEngine(QwtPlot::xBottom, new QwtLogScaleEngine());
#else
    setAxisScaleEngine(QwtPlot::xBottom, new QwtLog10ScaleEngine());
#endif
    setAxisScale(QwtPlot::xBottom, xmin, xmax);

    QwtText YAxisLabel;
    YAxisLabel.setText("Sound pressure [dB]");
    setAxisTitle(QwtPlot::yLeft, YAxisLabel);
    setAxisMaxMajor(QwtPlot::yLeft, 5);
    setAxisMaxMinor(QwtPlot::yLeft, 10);
    setAxisScale(QwtPlot::yLeft, -40.0, 10.0);
}

void Plot::showPeak(double x, double y)
{
    QString label;
    label.sprintf("Peak: %d dB", (int)round(y));

    QwtText text(label);
    text.setColor(Qt::red);

    markerP->setValue(x, y);
    markerP->setLabel(text);
}

void Plot::show3dB(double x, double y)
{
    QString label;
    label.sprintf("%d dB at %d Hz", (int)round(y), (int)x);

    QwtText text(label);
    text.setColor(Qt::blue);

    marker3->setValue(x, -3.0);
    marker3->setLabel(text);
}
