#include "sjog.h"
#include "sjog_config.h"
#include "sjog_list.h"

#define LINE_MAX 256

/* -------------------------------------------------------------------------- */

/*
 * Read the sjog config file and put the parsed entries in a list.
 */
gint
sjog_init_list_entries()
{

   FILE *config_stream;
   gchar line[LINE_MAX];
   gboolean err = FALSE;
   gchar rcname[LINE_MAX];

   /* 1st try to get the user config file */
   snprintf(rcname, LINE_MAX, "%s/.sjogrc", getenv("HOME"));

   config_stream = fopen(rcname, "r");

   /* if the user has no config file we try to get the system one */
   if (config_stream == NULL)
   {
      config_stream = fopen(strdup(SYSCONFDIR "/sjogrc"), "r");
      if (config_stream == NULL)
      {
         fprintf(stderr, "Can't open config file !\n");
         exit(1);
      }
   }

   /* foreach line, parse it and fill a list of sjog_cmd_t */
   while (!err && fgets(line, sizeof(line), config_stream) != NULL)
   {
      sjog_cmd_t *cmd;

      cmd = sjog_parse(line, &err);
      if (err == TRUE)
      {
         fclose(config_stream);
         fprintf(stderr, "Error in config file !\n");
         exit(1);
      }
      else if (cmd == NULL || cmd->path == NULL)
      {
         /* white or comment line, skip it */
         continue;
      }

#ifdef DEBUG
      fprintf(stdout, "sjog_init_config: name=%s path=%s\n", cmd->name,
              cmd->path);
#endif
      list = g_list_append(list, cmd);

#ifdef DEBUG
      fprintf(stdout, "list contains %d items\n", g_list_length(list));
#endif

   }

}

/* -------------------------------------------------------------------------- */

/*
 * takes a line and parse a name/path pair in it.
 */
sjog_cmd_t *
sjog_parse(gchar * line, gboolean * err)
{
   gchar *tmp, *name;
   sjog_cmd_t *sjog_cmd;

   *err = FALSE;

   sjog_cmd = (sjog_cmd_t *) malloc(sizeof(sjog_cmd_t));
   if (sjog_cmd == NULL)
   {
      *err = TRUE;
      return NULL;
   }

   /* strip comment or final \n */

   for (tmp = line; *tmp != '\0'; tmp++)
   {
      if (*tmp == '#' || *tmp == '\n')
      {
         *tmp = '\0';
         break;
      }
   }

   /* strip trailing whitespaces */

   while (tmp != line)
   {
      tmp--;
      if (*tmp == ' ' || *tmp == '\t')
      {
         *tmp = '\0';
      }
      else
      {
         break;
      }
   }

   /* find the name */

   tmp = line;
   while (*tmp == ' ' || *tmp == '\t')
   {
      tmp++;
   }
   if (*tmp == '\0')                        /* premature end of line */
   {
      return NULL;
   }

   /* spaces are allowed if the name is put between doublequotes */
   if (*tmp == '"')
   {
      /* skip first " */
      tmp++;

      /* name will start from here */
      name = tmp;

      /* allow everything except doublequotes and EOL */
      while (*tmp != '\0' && *tmp != '"')
      {
         tmp++;
      }
   }
   /* common case, no spaces or tabs */
   else
   {
      /* name starts from first char */
      name = tmp;
      while (*tmp != ' ' && *tmp != '\t' && *tmp != '\0')
      {
         tmp++;
      }
   }

   if (*tmp == '\0')                        /* no path */
   {
      sjog_cmd->name = strdup(name);
      if (sjog_cmd->name == NULL)
      {
         *err = TRUE;
         return NULL;
      }
      sjog_cmd->path = NULL;
      return sjog_cmd;
   }

   *tmp = '\0';

   /* copy the name */

   sjog_cmd->name = strdup(name);
   if (sjog_cmd->name == NULL)
   {
      *err = TRUE;
      return NULL;
   }

   /* strip the blanks */

   tmp++;
   while (*tmp == ' ' || *tmp == '\t' || *tmp == '"')
   {
      tmp++;
   }
   if (*tmp == '\0')                        /* premature end of line */
   {
      sjog_cmd->path = NULL;
      return sjog_cmd;
   }

   /* copy the path */

   sjog_cmd->path = strdup(tmp);
   if (sjog_cmd->path == NULL)
   {
      *err = TRUE;
      return NULL;
   }

   return sjog_cmd;
}
