#!/usr/bin/python

# $Id: VASSR.py 49 2010-07-07 04:59:24Z yamamoto2 $

# Copyright (c) 2010 VA Linux Systems Japan K.K. All rights reserved.
#
# LICENSE NOTICE
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. Neither the name of the Company nor the names of its contributors
#    may be used to endorse or promote products derived from this software
#    without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COMPANY AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE COMPANY OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.

# Copyright (C) 2006-2007 XenSource Ltd.
# Copyright (C) 2008-2009 Citrix Ltd.
#
# This program is free software; you can redistribute it and/or modify 
# it under the terms of the GNU Lesser General Public License as published 
# by the Free Software Foundation; version 2.1 only.
#
# This program is distributed in the hope that it will be useful, 
# but WITHOUT ANY WARRANTY; without even the implied warranty of 
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
# GNU Lesser General Public License for more details.

#
# VASSR: vastsky SR driver
#
# this code is derived from ISCSISR
#
# mapping of terms:
#	VAS		XCP
#	-----------------------------
#	SM		SR
#	LV		VDI
#	HS		host
#	lvolid		vdi location
#
# instead of using a vdi uuid generated by SRCommand.py, we generates uuid
# from vas lvolid so that xcp can use vas logical volumes created by others.
# eg. ones created with vas lvol_create command.
#

# XXX TODO
# - revisit XXXs
# - should check max lvol size etc, which is likely different between
#   vas and xcp
# - should report sr status (available space etc)

# XXX
import sys
sys.path = ['/opt/xensource/sm'] + sys.path

import SR
import VDI
import SRCommand
import util

import statvfs
import time
import os
import socket
import sys
import re
import glob
import xml.dom.minidom
import shutil
import xmlrpclib
import xs_errors
import errno
import uuid
import iscsilib

sys.path = sys.path + ['/opt/vas/lib']
import vas_subr

CAPABILITIES = \
    ["SR_SCAN", "VDI_ATTACH", "VDI_DETACH", "VDI_CREATE", "VDI_DELETE", \
     "VDI_ACTIVATE", "VDI_DEACTIVATE"]

CONFIGURATION = \
    [ ]

DRIVER_INFO = {
    'name': 'vastsky',
    'description': 'vastsky SR driver',
    'vendor': 'VA Linux Systems Japan, K.K.',
    'copyright': '(C) 2008 Citrix Systems Inc (C) 2010 VA Linux Systems Japan, K.K',
    'driver_version': '1.0',
    'required_api_version': '1.0',
    'capabilities': CAPABILITIES,
    'configuration': CONFIGURATION
    }

def log(msg):
    util.SMlog(msg)

def howmany(a, b):
    return (a + b - 1) / b

def roundup(a, b):
    return howmany(a, b) * b

VAS_ALLOCATION_UNIT = 1024 * 1024 * 1024

def vas_lvol_size_roundup(size):
    return roundup(size, VAS_ALLOCATION_UNIT)

def vas_lvol_string(lvolid):
    return "lvol-%08x" % lvolid

def vas_send_request(method, args):
    try:
	args['ver'] = vas_subr.XMLRPC_VERSION
        res = vas_subr.send_request(vas_subr.host_storage_manager_list, \
	    vas_subr.port_storage_manager, method, args)
    except xmlrpclib.Fault, inst:
	log("VASSR %s fault %d" % (method, inst.faultCode))
	# XXX XXX
	raise
    except:
	log("VASSR %s exception" % method)
	raise
    return res

def vas_create_lvol(name, size):
    log("VASSR vas_create_lvol %s" % size)
    assert roundup(size, VAS_ALLOCATION_UNIT) == size
    res = vas_send_request("createLogicalVolume", \
	{'lvolname' : name, 'capacity' : size / VAS_ALLOCATION_UNIT})
    log("VASSR vas_create_lvol res=%s" % res)
    return res # lvolid

def vas_delete_lvol(lvolid):
    log("VASSR vas_delete_lvol %s" % lvolid)
    res = vas_send_request("deleteLogicalVolume", {'lvolid' : lvolid})
    log("VASSR vas_delete_lvol res=%s" % res)
    assert res == 0

def vas_attach_lvol(lvolid):
    log("VASSR vas_attach_lvol %s" % lvolid)
    try:
	hsvrid = vas_get_hsvrid()
    except:
	log("VASSR vas_attach_lvol failed to get hsvrid")
	# XXX XXX
	raise
    log("VASSR vas_attach_lvol hsvrid=%d" % hsvrid)
    res = vas_send_request("attachLogicalVolume", \
	{'lvolid' : lvolid, 'hsvrid' : hsvrid})
    log("VASSR vas_attach_lvol res=%s" % res)
    return res

def vas_detach_lvol(lvolid):
    log("VASSR vas_detach_lvol %s" % lvolid)
    # XXX should check if the volume is actually attached to this head server?
    res = vas_send_request("detachLogicalVolume", {'lvolid' : lvolid})
    log("VASSR vas_detach_lvol res=%s" % res)

def vas_get_lvols():
    log("VASSR vas_get_lvols")
    res = vas_send_request("listLogicalVolumes", {})
    log("VASSR vas_get_lvols res=%s" % res)
    return res

def vas_lvolid_to_uuid(lvolid):
    # XXX
    vas_lvol_ns = uuid.UUID('{f4588257-6245-11df-8038-00188b7a51cc}')
    return str(uuid.uuid5(vas_lvol_ns, "%08x" % lvolid))

def vas_get_hsvrid():
    f = open("/var/lib/vas/hsvrid", "r")
    hsvrid_str = f.read()
    f.close()
    # hsvrid_str should be something like "hsvr-00000000" here
    t = hsvrid_str.split("-")
    if len(t) != 2 or t[0] != "hsvr":
	raise # XXX
    return int(t[1], 16)

class VASSR(SR.SR):
    """vastsky SR"""

    # get lvols from sm and construct self.vdis
    def _get_lvols(self):
	self.vdis = {}
	lvols = vas_get_lvols()
	if lvols:
	    for lvol in lvols:
		size = lvol['capacity'] * VAS_ALLOCATION_UNIT
		lvolid = lvol['lvolid']
		name = vas_lvol_string(lvolid)
		uuid = vas_lvolid_to_uuid(lvolid)
		vdi = VASVDI(self, uuid)
		vdi.utilisation = size
		vdi.size = size
		vdi.vdi_type = 'disk'
		vdi.location = str(lvolid)
		self.vdis[uuid] = vdi
	log("VASSR._get_lvols result %s" % self.vdis)

    # called by SR.__init__
    def load(self, uuid):
	log("VASSR.load %s" % uuid)
	self.sr_vditype = 'phy'

    # called by SRCommand
    def create(self, uuid, size):
	log("VASSR.create %s" % uuid)
	# nothing to do.
	# vas sm parameters are per-host (in /etc), not per-sr.
	pass

    # called by SRCommand
    def attach(self, uuid):
	log("VASSR.attach %s" % uuid)
	# nothing to do here as we attach/detach our VDI directly.
	# XXX is it worth to ping sm here?
	pass

    # called by SRCommand
    def detach(self, uuid):
	log("VASSR.detach %s" % uuid)
	# nothing to do here as we attach/detach our VDI directly.
	pass

    # called by SRCommand
    def delete(self, uuid):
	log("VASSR.delete %s" % uuid)
	self.detach(uuid)
    
    # called by SRCommand
    def scan(self, uuid):
	log("VASSR.scan %s" % uuid)
	# XXX TODO
        #	self.virtual_allocation
        #	self.physical_size
        #	self.physical_utilisation
	self._get_lvols()
        return super(VASSR, self).scan(uuid)

    # called by SRCommand
    def vdi(self, uuid):
	log("VASSR.vdi %s" % uuid)
	self._get_lvols()
        if not self.vdis.has_key(uuid):
	    # XXX SRCommand feeds us a random uuid created by uuid.gen_uuid
	    # at least for vdi_create.  what to do?
	    log("VASSR.vdi unknown uuid %s" % uuid)
	    assert(self.cmd == "vdi_create")
	    return VASVDI(self, uuid)
        return self.vdis[uuid]

    def _find_vdi_by_lvolid(self, lvolid):
	log("VASSR._find_vdi_by_lvolid %s" % lvolid)
	for uuid in self.vdis:
	    vdi = self.vdis[uuid]
	    if vdi.location == str(lvolid):
		return vdi
	log("VASSR._find_vdi_by_lvolid unknown lvolid %s" % lvolid)
	raise Exception, "lvolid not found"

class VASVDI(VDI.VDI):
    """vastsky VDI"""

    def __repr__(self):
	return "VASVDI(location %s uuid %s)" % (self.location, self.uuid)

    # called by SRCommand
    def attach(self, sr_uuid, vdi_uuid):
	log("VASVDI.attach %s" % vdi_uuid)
	assert vdi_uuid == self.uuid
	# actually we don't attach lvol here.
	# as vas doesn't support read-only attach at all,
	# we defer the attach operation to vdi.activate.
	# http://lists.xensource.com/archives/html/xen-devel/2010-06/msg00445.html
	local_path = "/dev/mapper/lvol-%08x" % int(self.location)
	log("VASVDI.attach local_path=%s" % local_path)
        return xmlrpclib.dumps((local_path,), "", True)

    # called by SRCommand
    def detach(self, sr_uuid, vdi_uuid):
	log("VASVDI.detach %s" % vdi_uuid)
	assert vdi_uuid == self.uuid
	# nothing to do here.  see deactivate

    # called by SRCommand
    def activate(self, sr_uuid, vdi_uuid):
	log("VASVDI.activate %s" % vdi_uuid)
	assert vdi_uuid == self.uuid
	sr = self.sr
	host_conf = sr.session.xenapi.host.get_other_config(sr.host_ref)
	local_iqn = host_conf['iscsi_iqn']
	local_path = "/dev/mapper/lvol-%08x" % int(self.location)
	log("VASVDI.activate local_iqn=%s local_path=%s" % \
	    (local_iqn, local_path))
	iscsilib.ensure_daemon_running_ok(local_iqn)
	# XXX hack; vas relies on the automatic creation of multipath
	# devices, which xcp doesn't seem to provide.
	# this hack relies on a possibly wrong assumption; no simultaneous
	# calls of SR drivers can happen.
	util.pread2(['ln', '-sf', '/etc/multipath-vas.conf', \
	    '/etc/multipath.conf'])
	util.pread2(['ln', '-sf', '/etc/multipath-vas.rules', \
	    '/etc/udev/rules.d/41-multipath-vas.rules'])
	try:
	    res = vas_attach_lvol(self.location)
	    if res != 0:
		# XXX
		# just ignore as we don't know the reason of failure.
		log("VASVDI.activate attach failed with res=%d.  ignored" % res)
	except:
	    log("VASVDI.activate attach failed with an exception.")
	    raise # XXX
	# clean up symlinks we created above.
	# XXX probably we should restore the multipath.conf link.
	util.pread2(['rm', '-f', \
	    '/etc/multipath.conf', \
	    '/etc/udev/rules.d/41-multipath-vas.rules'])
	if not os.path.exists(local_path):
	    # XXX VDIActivate is not in XE_SR_ERRORCODES.xml
	    raise xs_errors.XenError("VDIActivate")

    # called by SRCommand
    def deactivate(self, sr_uuid, vdi_uuid):
	log("VASVDI.deactivate %s" % vdi_uuid)
	assert vdi_uuid == self.uuid
	lvolid = int(self.location)
	vas_detach_lvol(lvolid)

    # called by SRCommand
    def create(self, sr_uuid, vdi_uuid, size):
	log("VASVDI.create %s %s" % (vdi_uuid, size))
	# XXX we have no way to know name-label of this vdi at this point.
	# as vas refuses lvols with duplicated names, we need to generate
	# a likely-unique name.  note that vdi_uuid will not be the uuid of
	# the vdi.  it's ok as what we want here is merely a likely-unique
	# string.
	name = "xcp-%s" % vdi_uuid
	rsize = vas_lvol_size_roundup(size)
	lvolid = vas_create_lvol(name, rsize)
	if lvolid == 0:
	    raise xs_errors.XenError("VDICreate")
	# XXX we ignore given vdi_uuid.  is this ok?
	self.sr._get_lvols()
	try:
	    vdi = self.sr._find_vdi_by_lvolid(lvolid)
	except:
	    # XXX a bug or race with other clients
	    raise
	vdi._db_introduce()
	return vdi.get_params()

    # called by SRCommand
    def delete(self, sr_uuid, vdi_uuid):
	log("VASVDI.delete %s" % vdi_uuid)
	assert vdi_uuid == self.uuid
	lvolid = int(self.location)
	try:
	    vas_delete_lvol(lvolid)
        except xmlrpclib.Fault, inst:
	    # ignore ENOENT as vdi.delete is an idempotent operation.
	    if inst.faultCode != ENOENT:
	        raise
	# update xapi's db
	# XXX i guess there's a smarter way...
	self.sr.scan(sr_uuid)

if __name__ == '__main__':
    log("VASSR run %s" % sys.argv)
    SRCommand.run(VASSR, DRIVER_INFO)
