/*
 *  XMMS Crossfade Plugin
 *  Copyright (C) 2000-2001  Peter Eisenlohr <p.eisenlohr@gmx.net>
 *
 *  based on the original OSS Output Plugin
 *  Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */

/*
 *  Rate conversion for 16bit stereo samples.
 * 
 *  The algorithm (Least Common Multiple Linear Interpolation) was
 *  adapted from the rate conversion code used in
 *
 *    sox-12.16, Copyright 1998  Fabrice Bellard, originally
 *               Copyright 1991  Lance Norskog And Sundry Contributors.
 *  
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "crossfade.h"
#include "rate.h"


static glong
gcd(glong m, glong n)
{
  glong r;
  while(1) {
    r = m % n;
    if(r == 0) return n;
    m = n;
    n = r;
  }
}

static glong
lcm(gint i, gint j)
{
  return ((glong)i * j) / gcd(i, j);
}


void
rate_init(rate_context_t *rc)
{
  memset(rc, 0, sizeof(*rc));
}

void
rate_config(rate_context_t *rc, gint in_rate, gint out_rate)
{
  rate_free(rc);

  if((in_rate  <= 0) || (in_rate  > 65535) || 
     (out_rate <= 0) || (out_rate > 65535)) {
    DEBUG(("[crossfade] rate_config: "
	   "illegal rates (in=%d, out=%d)!\n", in_rate, out_rate));
    return;
  }

  rc->in_rate  = in_rate;
  rc->out_rate = out_rate;
  rc->lcm_rate = lcm(in_rate, out_rate);
  rc->in_skip  = rc->lcm_rate / in_rate;
  rc->out_skip = rc->lcm_rate / out_rate;
  rc->in_ofs   = 0;
  rc->out_ofs  = 0;
  rc->started  = FALSE;
  rc->valid    = TRUE;
}

gint
rate_flow(rate_context_t *rc, gpointer *buffer, gint length)
{
  gpointer data;
  gint isamp, size, emitted = 0;
  gint16 *out, *in = *buffer;

  /* some sanity checks */
  if(length & 3) {
    DEBUG(("[crossfade] rate_flow: truncating %d bytes!", length & 3));
    length &= -4;
  }
  isamp = length / 4;
  if(isamp <= 0) return 0;
  if(!rc || !rc->valid) return length;
  if(rc->in_skip == rc->out_skip) return length;

  /* (re)allocate buffer */
  size = ((isamp * rc->in_skip) / rc->out_skip + 1) * 4;
  if(!rc->data || (size > rc->size)) {
    if(!(data = g_realloc(rc->data, size))) {
      DEBUG(("[crossfade] rate_flow: g_realloc(%d) failed!\n", size));
      return 0;
    }
    rc->data = data;
    rc->size = size;
  }
  *buffer = out = rc->data;

  /* first sample? */
  if(!rc->started) {
    rc->last_l  = in[0];
    rc->last_r  = in[1];
    rc->started = TRUE;
  }

  /* advance input range to span next output */
  while(((rc->in_ofs + rc->in_skip) <= rc->out_ofs) && (isamp-- > 0)) {
    rc->last_l  = *in++;
    rc->last_r  = *in++;
    rc->in_ofs += rc->in_skip;
  }
  if(isamp == 0) return emitted * 4;

  /* interpolate */
  for(;;) {
    *out++ = rc->last_l + (((gfloat)in[0] - rc->last_l)
			   * (rc->out_ofs - rc->in_ofs)
			   / rc->in_skip);
    
    *out++ = rc->last_r + (((gfloat)in[1] - rc->last_r)
			   * (rc->out_ofs - rc->in_ofs)
			   / rc->in_skip);
    
    /* count emitted samples*/
    emitted++;
    
    /* advance to next output */
    rc->out_ofs += rc->out_skip;
    
    /* advance input range to span next output */
    while((rc->in_ofs + rc->in_skip) <= rc->out_ofs) {
      rc->last_l  = *in++;
      rc->last_r  = *in++;
      rc->in_ofs += rc->in_skip;
      if(--isamp == 0) return emitted * 4;
    }
    
    /* long samples with high LCM's overrun counters! */
    if(rc->out_ofs == rc->in_ofs)
      rc->out_ofs = rc->in_ofs = 0;
  }
  
  return 0;  /* program flow never reaches this point */
}

void
rate_free(rate_context_t *rc)
{
  if(rc->data) {
    g_free(rc->data);
    rc->data = NULL;
  }
  rc->valid = FALSE;
}
