/*
 * (C) Finite State Machine Labs Inc. 1999-2000 <business@fsmlabs.com>
 *
 * Released under the terms of GPL 2.
 * Open RTLinux makes use of a patented process described in
 * US Patent 5,995,745. Use of this process is governed
 * by the Open RTLinux Patent License which can be obtained from
 * www.fsmlabs.com/PATENT or by sending email to
 * licensequestions@fsmlabs.com
 */

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/malloc.h>
#include <linux/timex.h>
#include <linux/spinlock.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/console.h>
#include <linux/irq.h>
#include <linux/config.h>

#include <asm/uaccess.h>
#include <asm/system.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/ptrace.h>

#include <arch/constants.h>
#include <rtl_conf.h>
#include <rtl_printf.h>
#include <rtl_core.h>
#include <rtl_sync.h>
#include <rtl.h>
#include <reserve_cpu.h>
#include <rtl_debug.h>
#include <linux/irq.h>

/*
 * Thanks to the wonders of EXTRAVERSION in the kernel, poor naming, 
 * bad Linux planning and plagues of frogs we have to have this and can't
 * trigger off KERNEL_VERSION() or the like.
 *    -- Cort
 */
#ifndef softirq_active
#define softirq_active(x) (softirq_state[x].active)
#endif
#ifndef softirq_mask
#define softirq_mask(x) (softirq_state[x].mask)
#endif

/* do checks for and print errors when we stop handling interrupts */
#undef DEBUG_PENDING 1

#ifdef DEBUG_PENDING
static unsigned long global_pending = 0;
#ifdef __LOCAL_IRQS__
static unsigned long local_pending = 0;
#endif /* __LOCAL_IRQS__ */
#endif /* DEBUG_PENDING */

void rtl_free_soft_irq(unsigned int irq)
{
	free_irq(irq, 0);
}

void rtl_hard_disable_irq(unsigned int ix);
void rtl_debug(void);
static inline void debug_test_enabled(char *s);
unsigned long last_cli[NR_CPUS];

#if __GNUC__ < 2 || (__GNUC__ == 2 && __GNUC_MINOR__ < 96)
/*
 * Workaround a bug that causes bad accesses when doing
 * __builtin_return_address() from leaf functions so we make
 * those functions non-leaf.
 *   -- Cort
 */ 
void non_leaf(void) {}
#define rtl_builtin_return_address(X) ({ 		\
	void *i = __builtin_return_address(X);	\
	non_leaf();					\
	i;						\
})
#define __builtin_return_address(x) rtl_builtin_return_address(x)
#endif /* gcc bug */

#ifdef CONFIG_RTL_TRACER
#ifndef rtl_trace
void rtl_trace_default(int event_id, long event_data, void * eip) { }
void (*rtl_trace)(int event_id, long event_data, void * eip) = rtl_trace_default;
void rtl_trace2(int event_id, long event_data) { rtl_trace(event_id, event_data, __builtin_return_address(0)); }
#endif
#endif

/* unsigned rtl_reserved_cpumask = 0; */

void conpr(const char *s)
{
	long flags;
	static spinlock_t rtl_conpr_lock = SPIN_LOCK_UNLOCKED;
	struct console *c;
	int len = strlen(s);

	rtl_hard_savef_and_cli(flags);
	rtl_spin_lock(&rtl_conpr_lock);

	c = console_drivers;
	while(c) {
		if ((c->flags & CON_ENABLED) && c->write)
			c->write(c, s, len);
		c = c->next;
	}
	rtl_spin_unlock(&rtl_conpr_lock);
	rtl_hard_restore_flags(flags);
}

void conprn(const unsigned int hexnum)
{
	int i;
	unsigned int d;
	unsigned int n = hexnum;
    	char s[10];
	s[9] = 0;
	s[8] = ' ';
	for (i=7; i>=0; i--) {
		d = n % 16;
		if (d < 10) {
			d += '0';
		} else {
			d += 'a' - 10;
		}
		s[i] = d; 
		n = n / 16;
	}
    	conpr(s);
}

/* assuming 255 global irqs and 31 max local vectors 
   On the x86 we only have local irqs when we are smp.
   But in PowerPC and other we may have local irqs from
   on chip timers and other advanced technologies
   */

/* bit positions for flags, constants, and macros for global structure */
#define IRQ_NOT_VALID -1
#define IRQ_ARRAY_SIZE ((256/8)/sizeof(ulong)) /*256 global irqs */
#define IRQ_ZINIT {0}
#if BITS_PER_LONG == 32
#define IRQ_NZINIT {~0x0,~0x0,~0x0,~0x0,~0x0,~0x0,~0x0,~0x0}
#else
#define IRQ_NZINIT {~0x0,~0x0,~0x0,~0x0}
#endif

#if BITS_PER_LONG == 32
#define irq_toi(x)  ((x>>5)&7)
#define irq_top(x)  ((x)& 0x1fUL)
#else
#define irq_toi(x)  ((x>>6)&15)
#define irq_top(x)  ((x)& 0x3fUL)
#endif
#define pi_toirq(p,i) ( (p) + ((i)*BITS_PER_LONG))

#define G_PEND(f) set_bit(irq_top(f),&rtl_global.pending[irq_toi(f)])
#define G_UNPEND(f) clear_bit(irq_top(f),&rtl_global.pending[irq_toi(f)])
#define G_ISPEND(f) test_bit(irq_top(f),&rtl_global.pending[irq_toi(f)])
/* clear and set global enabled irq bits */
#define G_ENABLED(f) set_bit(irq_top(f),&rtl_global.soft_enabled[irq_toi(f)])
#define G_DISABLE(f) clear_bit(irq_top(f),&rtl_global.soft_enabled[irq_toi(f)])
#define G_ISENABLED(f) test_bit(irq_top(f),&rtl_global.soft_enabled[irq_toi(f)])
/* clear and set real time handlers (RealTimeHandlers) */
#define G_SET_RTH(f) set_bit(irq_top(f),&rtl_global.rtirq[irq_toi(f)])
#define G_CLEAR_RTH(f) clear_bit(irq_top(f),&rtl_global.rtirq[irq_toi(f)])
#define G_TEST_RTH(f) test_bit(irq_top(f),&rtl_global.rtirq[irq_toi(f)])
#define G_TEST_AND_SET_RTH(f) test_and_set_bit(irq_top(f),&rtl_global.rtirq[irq_toi(f)])
#define G_TEST_AND_CLEAR_RTH(f) test_and_clear_bit(irq_top(f),&rtl_global.rtirq[irq_toi(f)])
/* global flags */
#define g_rtl_started 0
#define g_pend_since_sti 1
#define g_initializing 2
#define g_initialized 3
#define G_SET(f) set_bit(f,&rtl_global.flags)
#define G_CLEAR(f) clear_bit(f,&rtl_global.flags)
#define G_TEST(f) test_bit(f,&rtl_global.flags)
#define G_TEST_AND_SET(f) test_and_set_bit(f,&rtl_global.flags)
#define G_TEST_AND_CLEAR(f) test_and_clear_bit(f,&rtl_global.flags)

/* Bit positions of flags for local structure and macros
   for operating on them
 */
#define l_idle 0
#define l_ienable 1
#define l_pend_since_sti 2
#define l_busy 3
#define l_psc_active 4
#define L_SET(f) set_bit(f,&rtl_local[cpu_id].flags)
#define L_CLEAR(f) clear_bit(f,&rtl_local[cpu_id].flags)
#define L_TEST(f) test_bit(f,&rtl_local[cpu_id].flags)
#define L_TEST_AND_SET(f) test_and_set_bit(f,&rtl_local[cpu_id].flags)
#define L_PEND(f) set_bit(f,&rtl_local[cpu_id].pending)
#define L_UNPEND(f) clear_bit(f,&rtl_local[cpu_id].pending)
#define L_ISPEND(f) test_bit(f,&rtl_local[cpu_id].pending)

#define L_SET_RTH(f) set_bit(f,&rtl_local[cpu_id].rtirq)
#define L_CLEAR_RTH(f) clear_bit(f,&rtl_local[cpu_id].rtirq)
#define L_TEST_RTH(f) test_bit(f,&rtl_local[cpu_id].rtirq)
#define L_TEST_AND_SET_RTH(f) test_and_set_bit(f,&rtl_local[cpu_id].rtirq)
#define L_TEST_AND_CLEAR_RTH(f) test_and_clear_bit(f,&rtl_local[cpu_id].rtirq)

#define dispatch_rtl_handler(irq,r) rtl_global_handlers[irq].handler(irq,r)

/* TODO soft smp_processor_id doesn't work here???? -- Michael */
#define DeclareAndInit(cpu_id)  unsigned int cpu_id = rtl_getcpuid()
#define HardDeclareAndInit(cpu_id)  unsigned int cpu_id = rtl_getcpuid()

/* The basic control data structures local and global*/
struct rtl_local rtl_local[NR_CPUS];

struct rtl_global{
	spinlock_t hard_irq_controller_lock;
	unsigned long flags;
	unsigned long pending[IRQ_ARRAY_SIZE];
	unsigned long soft_enabled[IRQ_ARRAY_SIZE];
	unsigned long rtirq[IRQ_ARRAY_SIZE];
};
struct rtl_global rtl_global ={ SPIN_LOCK_UNLOCKED,0,IRQ_ZINIT,IRQ_NZINIT,IRQ_ZINIT} ;

/* RTLinux interrupts */
struct rtl_global_handlers{
        unsigned int (*handler)(unsigned int irq, struct pt_regs *r);
}rtl_global_handlers[IRQ_MAX_COUNT];

#ifdef __LOCAL_IRQS__
void rtl_local_pend_vec(int vector,int cpu_id)
{
	int i = VECTOR_TO_LOCAL_PND(vector);
	L_PEND(i);
	L_SET(l_pend_since_sti);
}

int rtl_local_ispending_irq(int ix)
{
 	HardDeclareAndInit(cpu_id);
	return L_ISPEND(ix);
}
#endif

extern void * rtl_code[];
#include "arch/arch.h"

static inline void debug_test_enabled(char *s)
{
	unsigned long flags;

	rtl_hard_save_flags(flags);
	if ( (ARCH_DEFINED_ENABLE && !(flags & ARCH_DEFINED_ENABLE))
	     || (!ARCH_DEFINED_ENABLE && (flags & ARCH_DEFINED_DISABLE)) )
	{
		do_first(10) {
			rtl_printf("%s: intrs hard disabled! called from %p\n",\
			   s, __builtin_return_address(0));
		}
	}
}

/* rtl_intercept intercepts global interrupts */
#define RUN_LINUX_HANDLER(irq) (G_ISPEND(irq) && !L_TEST(l_busy)\
       	&& L_TEST(l_ienable) && G_ISENABLED(irq))
intercept_t rtl_intercept(MACHDEPREGS regs)
{
	int irq;
	HardDeclareAndInit(cpu_id);
	
	rtl_spin_lock(&rtl_global.hard_irq_controller_lock);
	if ((irq = rtl_irq_controller_get_irq(regs)) != -1)
	{
		rtl_trace (RTL_TRACE_INTERCEPT, irq, (void *) instruction_pointer(MACHDEPREGS_PTR(regs)));
		rtl_irq_controller_ack(irq); /* may also mask, if needed */
		
		if(G_TEST_RTH(irq)){ /* this is a RT irq */
			/* if RT wants to share it pends */
			rtl_spin_unlock(&rtl_global.hard_irq_controller_lock);
			dispatch_rtl_handler(irq,MACHDEPREGS_PTR(regs));
			rtl_spin_lock(&rtl_global.hard_irq_controller_lock);
		} else {
			G_PEND(irq);
			G_SET(g_pend_since_sti);
		}

		if(RUN_LINUX_HANDLER(irq))
		{
			/* unpend so dispatch doesn't dispatch 2 times*/
			G_UNPEND(irq);
			rtl_soft_cli(); /* disable local soft interrupts */
			G_DISABLE(irq); /* disable this irq */
			rtl_spin_unlock(&rtl_global.hard_irq_controller_lock);
			rtl_hard_sti();
#ifdef DEBUG_PENDING
			global_pending = 0;
#endif /* DEBUG_PENDING */
			dispatch_linux_irq(MACHDEPREGS_PTR(regs),irq);
			rtl_trace2 (RTL_TRACE_INTERCEPT_EXIT, irq);
			RETURN_FROM_INTERRUPT_LINUX; /* goes via ret_from_intr */
		}

#ifdef DEBUG_PENDING
		/*
		 * If a Linux interrupt has been pended, and we haven't
		 * handled it, increment the global pending count.
		 *  -- Cort Dougan <cort@fsmlabs.com
		 */
		if ( G_ISPEND(irq) && (global_pending++ > 20) )
		{
			rtl_hard_cli();
			printk("Too many global intrs pended irq %d from %08lx\n",
			       irq, instruction_pointer(regs));
			printk("didn't run handler because:\n"
			       "ispend %d !l_busy %d ienable %d genable %d\n",
			       G_ISPEND(irq), !L_TEST(l_busy),
			       L_TEST(l_ienable), G_ISENABLED(irq) );
			rtl_debug();
			while(1);
		}
#endif /* DEBUG_PENDING */
		
		/* get here if irq==-1 or if otherwise can't run linux handler */
	}

	rtl_spin_unlock(&rtl_global.hard_irq_controller_lock);
	rtl_trace2 (RTL_TRACE_INTERCEPT_EXIT, irq);
	RETURN_FROM_INTERRUPT;
}

#ifdef __LOCAL_IRQS__
static inline unsigned int get_lpended_irq(void)
{
	int i;
	DeclareAndInit(cpu_id); /* only called in Linux context */
	if(rtl_local[cpu_id].pending){
		i = ffz(~rtl_local[cpu_id].pending);
		clear_bit(i,&rtl_local[cpu_id].pending);
		i = LOCAL_PND_TO_VECTOR(i);
	}
	else i = IRQ_NOT_VALID;
	return i ;
}

intercept_t rtl_local_intercept(MACHDEPREGS regs)
{
	int pnd;
	HardDeclareAndInit(cpu_id);

	/* no lock needed because we are already hard cli and only
	   use local per-cpu structures. The rtl_irq_controller
	   operations MUST vector to local only hardware or must
	   use spinlocks */
	pnd = MACHDEPREGS_TO_PND(regs);
	rtl_trace (RTL_TRACE_LOCAL_INTERCEPT, LOCAL_PND_TO_VECTOR(pnd), (void *) instruction_pointer(MACHDEPREGS_PTR(regs)));
	rtl_local_irq_controller_ack();
	if(L_TEST_RTH(pnd)){ /* this is a RT irq */
		dispatch_rtl_local_handler(pnd,MACHDEPREGS_PTR(regs));/* if RT wants to share it pends */
	}
	else{
		L_PEND(pnd);
		L_SET(l_pend_since_sti);
	}
	
	if(!L_ISPEND(pnd) || L_TEST(l_busy) || !L_TEST(l_ienable) )
	{
		rtl_trace2 (RTL_TRACE_LOCAL_INTERCEPT_EXIT, LOCAL_PND_TO_VECTOR(pnd));
#ifdef DEBUG_PENDING
		if ( L_ISPEND(pnd) && (local_pending++ > 20) )
		{
			printk("Too many local intrs pended pnd %d\n", pnd);
			rtl_debug();
			rtl_soft_sti();
			while(1);
		}
#endif /* DEBUG_PENDING */
		
		RETURN_FROM_LOCAL;
	}
	else
	{
		L_UNPEND(pnd); /* yes it is stupid, see above */
		rtl_soft_cli(); /* disable local soft interrupts */
		rtl_hard_sti();
#ifdef DEBUG_PENDING
		local_pending = 0;
#endif /* DEBUG_PENDING */
		dispatch_local_linux_irq(MACHDEPREGS_PTR(regs),pnd);
	}
	rtl_trace2 (RTL_TRACE_LOCAL_INTERCEPT_EXIT, LOCAL_PND_TO_VECTOR(pnd));
	RETURN_FROM_LOCAL_LINUX;
}
#endif

/* tools for soft_sti */
static inline unsigned int get_gpended_irq(void)
{
	unsigned int i, j;
	rtl_irqstate_t flags;
	unsigned long irqs;

	rtl_spin_lock_irqsave(&rtl_global.hard_irq_controller_lock, flags);
	for (i=0; i < IRQ_ARRAY_SIZE; i++) {
		irqs = rtl_global.pending[i] & rtl_global.soft_enabled[i];
		if (!irqs)
			continue;
		j = ffz(~irqs);
		clear_bit(j, &rtl_global.pending[i]);
		rtl_spin_unlock_irqrestore(&rtl_global.hard_irq_controller_lock, flags);
		return pi_toirq (j, i);
	}
	rtl_spin_unlock_irqrestore(&rtl_global.hard_irq_controller_lock, flags);
	return IRQ_NOT_VALID;
}

void rtl_soft_cli(void)
{
	DeclareAndInit(cpu_id);
	if ( L_TEST(l_ienable) )
		last_cli[cpu_id] = (unsigned long)__builtin_return_address(0);
	L_CLEAR(l_ienable);
}

void rtl_soft_sti_no_emulation(void)
{
	DeclareAndInit(cpu_id);
	if ( !L_TEST(l_ienable) )
		last_cli[cpu_id] = 0;
	L_SET(l_ienable);
}

void rtl_process_pending(void)
{
	int irq = 0;
	int last_irq = 0;
	DeclareAndInit(cpu_id);

	rtl_soft_cli(); /*disable soft interrupts !*/
       	do{
		irq = IRQ_NOT_VALID;
	       	G_CLEAR(g_pend_since_sti);
	       	L_CLEAR(l_pend_since_sti);
#ifdef __LOCAL_IRQS__
	       	while ( (irq = get_lpended_irq()) != IRQ_NOT_VALID ) {
#ifdef DEBUG_PENDING
			local_pending = 0;
#endif /* DEBUG_PENDING */
			soft_dispatch_local(irq);
		}
#endif
#ifdef __RTL_LOCALIRQS__
		if (!test_bit(cpu_id, &rtl_reserved_cpumask))
#endif
		{
			while ( (irq = get_gpended_irq()) != IRQ_NOT_VALID )
			{
				last_irq = irq;
#ifdef DEBUG_PENDING
				global_pending = 0;
#endif /* DEBUG_PENDING */
				soft_dispatch_global(irq);
			}
		}
#ifdef __RTL_LOCALIRQS__	
	}while(irq != IRQ_NOT_VALID || (!test_bit(cpu_id, &rtl_reserved_cpumask) && G_TEST(g_pend_since_sti)) || L_TEST(l_pend_since_sti));
#else
	}while(irq != IRQ_NOT_VALID || G_TEST(g_pend_since_sti) || L_TEST(l_pend_since_sti));
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
	/* process any bottom halves */
	if (  softirq_active(cpu_id) & softirq_mask(cpu_id) )
		do_softirq();
#endif		
}

void rtl_soft_sti(void)
{
	DeclareAndInit(cpu_id);
	/*debug_test_enabled("rtl_soft_sti");*/
	if ( L_TEST(l_pend_since_sti) || G_TEST(g_pend_since_sti)
#if LINUX_VERSION_CODE >= 0x020300
	   || (softirq_active(cpu_id) & softirq_mask(cpu_id) )
#endif
	   )
		rtl_process_pending();
	rtl_soft_sti_no_emulation();
}

void rtl_soft_save_flags(unsigned long *x)
{
	DeclareAndInit(cpu_id);
	*x = (L_TEST(l_ienable)? ARCH_DEFINED_ENABLE: ARCH_DEFINED_DISABLE);
}

void rtl_soft_restore_flags(unsigned long x)
{
	if(x == ARCH_DEFINED_ENABLE)rtl_soft_sti();
	else rtl_soft_cli();
}

void rtl_soft_save_and_cli(unsigned long *x)
{
	extern unsigned long last_cli[NR_CPUS];
	DeclareAndInit(cpu_id);
	
	rtl_soft_save_flags(x);
	rtl_soft_cli();
	/*
	 * If we make a sti->cli transition record the
	 * caller so we have some useful data instead of the
	 * address of this function.
	 *   -- Cort
	 */
	if ( (ARCH_DEFINED_ENABLE && (*x & ARCH_DEFINED_ENABLE))
	     || (!ARCH_DEFINED_ENABLE && !(*x & ARCH_DEFINED_DISABLE)) )
		last_cli[cpu_id] = (ulong)__builtin_return_address(0);
}

void rtl_soft_local_irq_save(unsigned long *x)
{
	rtl_soft_save_flags(x);
	rtl_soft_cli();
}

void rtl_soft_local_irq_restore(unsigned long x)
{
	rtl_soft_restore_flags(x);
}

void rtl_virt_disable(unsigned int irq)
{
	G_DISABLE(irq);
}

void rtl_virt_enable(unsigned int irq)
{
	rtl_irqstate_t flags;
	rtl_no_interrupts(flags);
	
	G_ENABLED(irq);
	if(!G_ISPEND(irq))
	{
		rtl_hard_enable_irq(irq);
		rtl_restore_interrupts(flags);
	}
	else
	{
		HardDeclareAndInit(cpu_id);
		rtl_restore_interrupts(flags);
		if( L_TEST(l_ienable))
			__sti(); /* emulate the bastard */
	}
}

/* these are exported so that they can be called by rt drivers */
void rtl_global_pend_irq(int ix) { G_PEND(ix); G_SET(g_pend_since_sti); }

int rtl_global_ispending_irq(int ix) { return G_ISPEND(ix); }

void rtl_hard_enable_irq(unsigned int ix)
{
	rtl_irqstate_t flags;
	rtl_no_interrupts (flags);
	rtl_spin_lock(&rtl_global.hard_irq_controller_lock);
	rtl_irq_controller_enable(ix);
	rtl_spin_unlock(&rtl_global.hard_irq_controller_lock);
	rtl_restore_interrupts (flags);
}

void rtl_hard_disable_irq(unsigned int ix)
{
	rtl_irqstate_t flags;
	rtl_no_interrupts (flags);
	rtl_spin_lock(&rtl_global.hard_irq_controller_lock);
	rtl_irq_controller_disable(ix);
	rtl_spin_unlock(&rtl_global.hard_irq_controller_lock);
	rtl_restore_interrupts (flags);
}

/* these are used by schedulers to make sure that Linux interrupts
   do not advance and delay RT tasks 
 Both  need to be called with irqs disabled */
void rtl_make_rt_system_active(void)
{
	HardDeclareAndInit(cpu_id);
	L_SET(l_busy);
}

void rtl_make_rt_system_idle(void)
{
	HardDeclareAndInit(cpu_id);
        L_CLEAR(l_busy);

}

unsigned int rtl_rt_system_is_idle(void)
{
	HardDeclareAndInit(cpu_id);
        return !L_TEST(l_busy);

}

void rtl_make_psc_active(void)
{
	HardDeclareAndInit(cpu_id);
	L_SET(l_psc_active);
}

void rtl_make_psc_inactive(void)
{
	HardDeclareAndInit(cpu_id);
        L_CLEAR(l_psc_active);

}

int rtl_is_psc_active(void)
{
	HardDeclareAndInit(cpu_id);
        return L_TEST(l_psc_active);

}

/* requesting and freeing rt interrupts */
/* TODO resolve the smp synchronization problem here */
int rtl_request_global_irq(unsigned int irq, 
			   unsigned int (*handler)(unsigned int, struct pt_regs *))
{

	if (!G_TEST_RTH(irq))
	{
		rtl_global_handlers[irq].handler =handler;
		G_SET_RTH(irq);
		mb();
		if(rtl_global_handlers[irq].handler == handler)
		{
			rtl_hard_enable_irq (irq);
			return 0;
		}
	}
	return -EBUSY;
}

int rtl_free_global_irq(unsigned int irq )
{
	if (!G_TEST_AND_CLEAR_RTH(irq))
		return -EINVAL;
	return 0;
	/* don't need to clear the handler, because it will never
	   be invoked -- see rtl_intercept. If we wanted to clear the handler
	   we would have a problem with synchronization in the smp case */
}

MODULE_AUTHOR("FSMLabs <support@fsmlabs.com>");
MODULE_DESCRIPTION("RTLinux Main Module");
int quiet;
MODULE_PARM(quiet, "i");

int init_module(void)
{
	int ret;

	if ( arch_takeover() )
	{
		printk("arch_takeover failed\n");
		return -1;
	}
	if ( !quiet )
		printk("RTLinux Extensions Loaded (http://www.fsmlabs.com/)\n");

	ret = rtl_printf_init();
	if (ret < 0)
		return ret;

	rtl_soft_sti(); 
	rtlinux_suspend_linux_init();
	return 0;
}

void cleanup_module(void)
{
	HardDeclareAndInit(cpu_id);
	rtl_printf_cleanup();

	/*
	 * Process any pending interrupts, _hard_ disable
	 * then go on.  This way, we don't get any interrupts
	 * while we're vulnerable and giving up the architecture.
	 *   -- Cort
	 *
	 *   This works for the current processor only -- Michael
	 */
	rtl_hard_cli();
	rtl_soft_sti_no_emulation();
	do {
		rtl_hard_sti();
		rtl_process_pending();
		rtl_hard_cli();
	} while ( G_TEST(g_pend_since_sti) || L_TEST(l_pend_since_sti));
	arch_giveup();
	rtlinux_suspend_linux_cleanup();
	rtl_hard_sti();
}

spinlock_t debug_lock = SPIN_LOCK_UNLOCKED;

void rtl_debug(void)
{
	int i;
	unsigned long flags, xxx_last_cli[NR_CPUS];

	memcpy( (void *)xxx_last_cli, (void *)last_cli, sizeof(last_cli) );
	
#define WIDTH(x) ((int)(sizeof(x)*2))
	
	rtl_spin_lock(&debug_lock);
	rtl_hard_save_flags(flags);
	printk( "RTL: cpu %d\n", rtl_getcpuid() );
	if ( ARCH_DEFINED_ENABLE == 0 )
		printk( "RTL: hard flags %0*lx %s\n", WIDTH(flags), flags,
			(flags&ARCH_DEFINED_DISABLE) ? "disabled" : "enabled" );
	else
		printk( "RTL: hard flags %0*lx %s\n", WIDTH(flags), flags,
			(flags&ARCH_DEFINED_ENABLE) ? "enabled" : "disabled" );
		
	printk( "RTL: global flags %0*lx %s%s\n",
		WIDTH(rtl_global.flags), rtl_global.flags,
		((rtl_global.flags>>g_pend_since_sti)&1) ? "pend_since_sti " : "",
		((rtl_global.flags>>g_initializing)&1) ? "initializing " : "");
	printk( "RTL: global pending " );
	for ( i = 0; i < IRQ_ARRAY_SIZE; i++ )
		printk( "%0*lx ", WIDTH(rtl_global.pending[i]),
					rtl_global.pending[i] );
	printk("\n");
	printk( "RTL: global enabled " );
	for ( i = 0; i < IRQ_ARRAY_SIZE; i++ )
		printk( "%0*lx ", WIDTH(rtl_global.soft_enabled[i]),
					rtl_global.soft_enabled[i] );
	printk( "\n" );
	for ( i = 0 ; i < rtl_num_cpus(); i++ )
	{
		int cpu = cpu_logical_map (i);
		printk( "RTL: cpu%d "
#ifdef __LOCAL_IRQS__
			"local pending %08x "
#endif			
			"flags: %08x %s%s%s last_cli: %0*lx\n", cpu,
			rtl_local[cpu].flags,
#ifdef __LOCAL_IRQS__
			rtl_local[cpu].pending,
#endif			
			((rtl_local[cpu].flags>>l_ienable)&1)?"ienabled ":"disabled ",
			((rtl_local[cpu].flags>>l_pend_since_sti)&1)?"pend_since_sti ":"",
			((rtl_local[cpu].flags>>l_busy)&1)?"busy":"",
			WIDTH(xxx_last_cli[i]), xxx_last_cli[i] );
	}
	rtl_spin_unlock(&debug_lock);
#undef WIDTH	
}

void rtl_soft_irq_type(int unused, void *junk, struct pt_regs *garbage)
{
	printk("rtl_soft_irq_type(): shouldn't have been called!\n");
}

/* TODO VY: needs some synchronization here. Doesn't request_irq also
   have a problem? */
int rtl_get_soft_irq (void (*handler) (int, void *, struct pt_regs *),
                     const char *devname)
{
	int i;
	int debug = 0;

	for (i = RTL_NR_IRQS - 1; i > 15; i--)
	{
		if ( !(debug = request_irq (i, handler, 0, devname, 0)) )
		{
			rtl_virt_enable(i);
			/* This needs to be done this way since having the 'continue'
			 * below causes egcs-2.90.29 with target MIPS (host PPC)
			 * to emit code that will execute both the true _and_ the
			 * false case of the above if.
			 *  -- Cort
			 */
			goto bad_mips_gcc;
		}
	}
	printk("RTL_GET_SOFT_IRQ %d: request=%d\n", i, debug);
	return -1;

bad_mips_gcc:
	return i;
}

/* compatibility irq handler table */
#include <asm/rt_irq.h>
typedef void (*RTL_V1_HANDLER)(void);
RTL_V1_HANDLER rtl_v1_irq[NR_IRQS];

extern unsigned int rtl_compat_irq_handler(unsigned int irq, struct pt_regs *regs)
{
	rtl_v1_irq[irq]();
	rtl_hard_enable_irq(irq);
	return 0;
}

int request_RTirq(unsigned   int   irq,   void (*handler)(void))
{
	int ret;
	rtl_v1_irq[irq] = handler;
	ret = rtl_request_global_irq(irq, rtl_compat_irq_handler);
	if (ret)
		return ret;
	rtl_hard_enable_irq(irq);
	return 0;
}
